#!/usr/bin/env python
"""The function of this module is many-fold:
 - first, it starts a bittorrent tracker
 - second, it starts the seed node
 - third, it watches for progress from the nodes downloading and shows a progress report

 finally, when the time is ripe, it kills everybody!
"""

# Author: Ashwin Bharambe        Begin: 01/10/2005
#  $Id: btsource.py 2468 2005-11-11 20:16:10Z jeffpang $
#

import sys, os, signal
import socket
import getopt

def get_host_string(host, port):
   return "%s:%d" % (host, port)

def start_tracker(args):
   """start the tracker in a separate child process"""   
   pid = os.fork()
   if pid > 0:
      return pid

   from BitTorrent.track import track

   print >>sys.stderr, "starting tracker..."
   try:
      track(args)   # control should never go beyond this... this starts an infinite loop!
   except Exception, e:
      print >>sys.stderr, "tracker exiting due to exception", e
      sys.exit(1)
   
def start_seed(args):
   """start a seed node in a separate child process"""
   pid = os.fork()
   if pid > 0:
      return pid          # the parent returns and waits for us

   from BitTorrent.download import download
   from threading import Event

   done = False
   def finished():
      print >>sys.stderr, "seed node ready to serve the file"
      done = True

   def failed():
      sys.exit(2)

   def error(msg):
      print >>sys.stderr, "downloading error (%s)" % msg
      sys.exit(1)

   def display(dict):
      pass

   def choosefile(default, size, saveas, dir):
      """this function is needed by bittorrent for some stupid reason"""
      if saveas != '':
           default = saveas        
      return default

   def newpath(path):
      pass

   print >>sys.stderr, "starting seed node..."
   try:
      download(args, choosefile, display, finished, error, Event(), 80, newpath)
   except Exception, e:
      print >>sys.stderr, "seed node exiting due to exception", e
      sys.exit(1)

   # control should never reach here in the working case...
   # the child should get killed as soon as all the nodes have received their file(s)
   if not done:
      failed()

def kill_children(pids):
   for pid in pids:
      os.kill(pid, signal.SIGKILL)

   for pid in pids:
      os.waitpid(pid, 0)
      
def start_watchdog(port, ndownloaders, tracker_pid, seednode_pid):
   """monitor progress of all the downloaders"""
   from BitTorrent.bencode import bdecode
   import gtk
   import gobject

   print >>sys.stderr, "Starting download monitor..."
   sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
   sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
   sock.bind(("", port))

   dnldinfo = {}

   def on_receive(source, condition):
      data, addr = source.recvfrom(1500)
      
      adstr = get_host_string(addr[0], addr[1])
      if not dnldinfo.has_key(adstr):
         dnldinfo[adstr] = {}
         
      if data:
         kv = bdecode(data)
         dnldinfo[adstr] = kv

      return True

   def get_ndone():
      ndone = 0
      for addr, kv in dnldinfo.iteritems():
         if kv.has_key('percentDone'):
	    try:
		pd = float(kv['percentDone'])
	    except:
		print "* exception occurred while getting status; continuing..."
            pd = round(pd - 0.5)
            if pd == 100:
               ndone += 1

      return ndone

   def print_update():
      #print "--------------------------------------------------"
      for addr, kv in dnldinfo.iteritems():         
         print >>sys.stderr, "%22s" % addr,
         for k, v in kv.iteritems():
            if k == 'percentDone':
               print >>sys.stderr, " %3.0f%%" % float(v),
            if k == 'downRate':
               print >>sys.stderr, " %12s" % v,
            if k == 'timeEst':
               print >>sys.stderr, " %22s" % v,
               #print k, "=", v,
         print >>sys.stderr, ""


   def check_status():
      ndone = get_ndone()
      if ndone >= ndownloaders:
         print >>sys.stderr, "everybody done!"
         kill_children((tracker_pid, seednode_pid))
         sys.exit(0)

      print_update()
      print >>sys.stderr, "------------------------ #finished = %2d/%2d ------------------------" % (ndone, ndownloaders)
      
      gobject.timeout_add(5000, check_status)      
      
   gobject.timeout_add(1000, check_status)
   gobject.io_add_watch(sock, gobject.IO_IN, on_receive)
   gtk.main()      

def usage():
   print >>sys.stderr, "btsource.py [options]"
   print >>sys.stderr, "      -h      help"
   print >>sys.stderr, "      -p int  port for the tracker"
   print >>sys.stderr, "      -T str  directory which is to be pushed"
   print >>sys.stderr, "      -n int  number of downloaders"

# xxx need to make this configurable so we don't get collisions!
WATCHDOG_PORT = 60001

def main(argv=None):
   if argv == None:
      argv = sys.argv[1:]

   try:
      opts, args = getopt.getopt(argv, "p:T:hn:")
   except getopt.GetoptError:
      # print help information and exit:
      usage()
      sys.exit(2)
        
   tracker_port = "8080"   # keep these strings, since we need to pass them as if they are elements of argv[]
   torrent_prefix = None
   ndownloaders = 0
   for o, a in opts:
      if o == "-h":
         usage()
         sys.exit(0)
      elif o == "-p":
         tracker_port = a
      elif o == "-T":
         torrent_prefix = a
      elif o == "-n":
         ndownloaders = int(a)

   if not torrent_prefix:
      print >>sys.stderr, "no torrent file given!"; sys.exit(2)
   if ndownloaders <= 0:
      print >>sys.stderr, "ndownloaders <= 0?!"; sys.exit(2)

   # this signal handler will be used by the child processes
   signal.signal(signal.SIGINT, lambda n, f: None)
   
   tracker_pid = start_tracker(("--port", tracker_port, "--dfile", "/tmp/dstate", "--logfile", "%s.tracklog" % torrent_prefix))
   seednode_pid = start_seed(("--minport", "20000", "--maxport", "20009", "--responsefile", "%s.torrent" % torrent_prefix,  "--saveas", torrent_prefix))

   # we will use this signal handler!
   def catch_sigint(signum, stackframe):
      print >>sys.stderr, "caught SIGINT!"
      kill_children((tracker_pid, seednode_pid))
      sys.exit(1)
      
   signal.signal(signal.SIGINT, catch_sigint)   

   # this waits forever
   start_watchdog(WATCHDOG_PORT, ndownloaders, tracker_pid, seednode_pid)
   
   return 0

if __name__ == "__main__":
   sys.exit(main())
