#
# Mercury and Colyseus Software Distribution 
# 
# Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
#               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
#                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USA
#
###############################################################################
#
# File: Travertine.pm 
# Version: $Id: Travertine.pm 3699 2007-06-07 00:34:13Z jeffpang $
# 
# "The material of which the Coliseum was built is exactly fitted to
# the purposes of a great ruin. It is travertine, of a rich, dark,
# warm color, deepened and mellowed by time. There is nothing glaring,
# harsh, or abrupt in the harmony of tints."
#
# Infrastructure and deployment module. Here is a breif overview of the
# contents of this module; see the comments in the code below for more info.
#
# NOTE: You MUST install 'screen' in /usr/bin or /usr/local/bin on all
# remote machines for the daemon mode to work. bash is stupid and 
# doesn't allocate an interactive shell (hence refuses to load .bash_profile 
# and .bashrc) when called by ssh, so there is no way to change the PATH
# automatically before executing the screen function.
#
# 
# Objects
# =======
#
# childFunc  - a hash which contains info about a parallel local function
#   ->new(\&func, [@args], %opts)   => $childFunc
#   ->wait()                        => $retval
#   ->stop()                        => $success
#   ->isAlive()                     => $yes
#   ->serialize()                   => $string
#   ::deserialize($string)          => $childFunc
#
# sshTunnel  - a hash which contains info about an open ssh tunnel
#   ->new($user, $host, %opts)      => $sshTunnel
#   ->exec($cmd, $stdin)            => ($stdout, $stderr, $exitval)
#   ->close()                       => $success
#
# rbRef      - a hash which contains info about a remote background function
#   ->new($cfs, $sshTunnel, $log)   => $rbRef
#   ->wait()                        => $retval
#   ->stop()                        => $success
#   ->isAlive()                     => $yes
#   ->headLog($lines)               => $logString
#   ->tailLog($lines)               => $logString
#   ->removeLog()                   => $success
#   ->serialize()                   => $string 
#   ::deserialize($string)          => $rbRef
#
# rdRef      - a hash reference which contains info about a remote daemon
#   ->new($ssh, $cmd, %opts)        => $rdRef
#   ->stop()                        => $success
#   ->isAlive()                     => $yes
#   ->headLog($lines)               => $logString
#   ->tailLog($lines)               => $logString
#   ->removeLog()                   => $success
#   ->attachToTerm()                => void
#   ->attachToXterm($title)         => void
#   ::AttachToGterm(\@rdRef, %opts) => void
#   ->serialize()                   => $string
#   ::deserialize($string)          => $rdRef
#
#
# Functions
# =========
#
# ParallelExec([\&func1, @args1], ...) => @retvals
# ParallelExec0($maxparallel, [\&func1, @args1], ...) => @retvals
# ParallelExec2(\&func, [@args1], ...) => @retvals
# ParallelExec3($maxparallel, \&func, [@args1], ...) => @retvals
# ParallelExec4($batchsize, \&func, [@args1], ...) => @retvals
# ExecRemoteFunc($sshTunnel, \&func, [@args], %opts) => $ret (see below)
#
#
# Error Handling
# ==============
#
# Undef usually means effor. Warning and fatal messages are printed out as
# follows (tinfo, twarn, tdie, respectively):
#
# [II] informational message
#
# [EE] warning message
#           stacktrace...
#
# [FF] fatal message
#           stacktrace...
#
# Messages in blue background is the output on STDERR on the remote host.
#
# Example: ChildFunc
# ==================
#
# Suppose we have the following declarations:
#
#   use Travertine;
#
#   sub a { print "a @_\n"; sleep 10; return 1; }     # regular func
#   sub b { print "b @_\n"; sleep 10; return [3,4]; } # regular func
#   sub d { while (1) { print "d @_\n"; sleep 1; } }  # daemon func
#
# To run &a(1,2) in a new forked process, and then wait for its return value:
#
#   my $cf = Travertine::ChildFunc->new(\&a, [1,2]);
#   ...
#   my $retval = $cf->wait(); # $ret == 1
# 
# To run &a(1,2) and &b(1,2) and wait for both to complete:
#
#   my @retvals = ParallelExec([\&a,1,2], [\&b,1,2]);
#
# If you pass the -detach flag when creating a new Travertine::ChildFunc, the
# child process will detach from the parent and you can wait for it in another
# perl script even if the parent has exited:
#
#   my $cf = Travertine::ChildFunc->new(\&a, [1,2], -detach => 1);
#   print SAVE $cf->serialize();
#   exit 0;
#
#   ...
#
#   my $cf = Travertine::ChildFunc::deserialize(<SAVE>);
#   my $retval = $cf->wait();
#
#
# Example: SSHTunnel
# ==================
#
# XXX Note: The current SSHTunnel is pretty stupid. new() and close() don't
# do anything; each time you execute a command, a new tunnel is opened and
# then closed. This is because I can't figure out a way to pass the open ssh
# sockets around different threads/processes without having perl segfault. :P
#
# To create a new ssh tunnel:
#
#   my $ssh = Travertine::SSHTunnel->new($user, $host);
#
# Then to execute a remote command:
#
#   my ($stdout, $stderr, $exitval) = $ssh->exec("ls -l");
#
# To close the connection:
#
#   $ssh->close();
#
# 
# Exmaple: ExecRemoteFunc
# =======================
#
# To execute a perl function ("RPC") on a remote machine, you just need an
# open ssh tunnel.
#
#   my ($stdout, $stderr, $retval) = ExecRemoteFunc($ssh, \&a, [1,2]);
#
# This will wait until the remote function finishes excuting and returns.
# The $retval returned is the return value of \&a interpreted as a value
# in [0,255].
#
#
# Example: RemoteBgFunc
# =====================
# 
# If you don't want to wait for the remote function to complete before 
# returning, then you can use the -background flag. This will return
# a rbRef background process object.
#
#   my $rbRef = ExecRemoteFunc($ssh, \&a, [1,2], -background => 1);
#   ...
#   my $retval = $rbRef->wait();
#
# Like childFuncs, you can also wait or stop a rbRef even after the perl
# script exits. Note that deserializing a rbRef will cause a new sshTunnel
# to the host to be opened.
#
#
# Example: RemoteDaemon
# =====================
#
# If you have a "daemon" function to execute that will presumably execute
# "forever," then you can execute it with -daemon, which will execute the
# perl function within an instance of 'screen' on the remote host. This
# allows you to reattach to it later.
#
#    my $rdRef = ExecRemoteFunc($ssh, \&d, [3,4], -daemon => 1);
#    ...
#    $rdRef->attachToTerm(); # attach the screen session to the terminal
#    ...
#    my $lines = $rdRef->tailLog(10); # look at last 10 lines of the output log
#    ...
#    $rdRef->stop();  # kill -15 the daemon
#    $rdRef->stop(9); # kill -9 the daemon
#
# Like childFuncs and rbRefs, you can serialize rdRefs to disk and then
# recover them later for use in other perl scripts.
#
#
# Environment Configuration Variables
# ===================================
#
# TRAVERTINE_LOGDIR          = default log dir on remote hosts
# TRAVERTINE_STUBDIR         = dir to copy perl stubs to on remote hosts
# TRAVERTINE_LOCALSTUBDIR    = local dir to save perl stubs if enabled
# TRAVERTINE_STATDIR         = dir to save remote process status
#
# TRAVERTINE_SHOWSTACKTRACE  = show stacktrace with warnings and errors if true
# TRAVERTINE_COLORSTACKTRACE = colorize stacktraces if true
# TRAVERTINE_PRINTCMDS       = print all shell or ssh commands executed
# TRAVERTINE_SAVESTUBS       = save stubs in LOCALSTUBDIR executed remotely
#
# TRAVERTINE_SSHTIMEOUT      = timeout on ssh connect (currently unused)
# TRAVERTINE_SSHTUNNELHOST   = host to tunnel all ssh commands through
# TRAVERTINE_SSHVERSION      = default ssh protocol version (e.g., '2,1')
#
###############################################################################

package Travertine;
require 5.6.1;
require Exporter;

# standard libraries/pragmas
use strict;
#use Thread;
use Socket;
use Data::Dumper;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;
use Fcntl qw(F_GETFL F_SETFL);
use IO::Select;
use Carp;
use Carp qw(cluck);

# non-standard libraries
use Compress::Zlib;
use IPC::Open3;
#use Net::SSH::Perl;

our @ISA    = ("Exporter");
our @EXPORT = qw(psystem ppsystem pbacktick rerror famsymsg rsystem rsystem2 tinfo twarn tdie thighlight
		 ParallelExec ExecRemoteFunc
		 ParallelExec0 ParallelExec2 ParallelExec3 ParallelExec4,
		 rexec ssh cfork tr_export_func tr_export_scalar);
our @CARP_NOT = qw(tinfo twarn tdie);

###############################################################################
# Environment Config:

# default directory where output logs are created (on local as well as remote)
# this directory must *already* exist!
our $LOG_DEFAULT_DIR = defined $ENV{TRAVERTINE_LOGDIR} ? 
    $ENV{TRAVERTINE_LOGDIR} : "/tmp";

# where remote perl script stubs are copied to before executed
# this directory must *already* exist!
our $REMOTE_STUB_DIR = defined $ENV{TRAVERTINE_STUBDIR} ? 
    $ENV{TRAVERTINE_STUBDIR} : "/tmp";

# where remote perl script stubs are saved locally (if debug enabled)
# make sure this is *different* from $REMOTE_STUB_DIR, otherwise behavior
# when executing stuff on localhost will be screwed up
# this directory will be created if it doesn't exist
our $LOCAL_STUB_DIR = defined $ENV{TRAVERTINE_LOCALSTUBDIR} ? 
    $ENV{TRAVERTINE_LOCALSTUBDIR} : "/tmp/stubs";

# where the return/exit status of exiting remote processes is written to
# this directory must *already* exist!
our $REMOTE_STAT_DIR = defined $ENV{TRAVERTINE_STATDIR} ? 
    $ENV{TRAVERTINE_STATDIR} : "/tmp";

# if true, show a stack trace with warnings and fatal errors
our $DEBUG_SHOW_STACKTRACE = defined $ENV{TRAVERTINE_SHOWSTACKTRACE} ?
    $ENV{TRAVERTINE_SHOWSTACKTRACE} : 0;

# if true, colorize the strack traces to make them a bit easier to read
our $DEBUG_COLOR_STACKTRACE = defined $ENV{TRAVERTINE_COLORSTACKTRACE} ?
    $ENV{TRAVERTINE_COLORSTACKTRACE} : 1;

# if true, then print all shell commands executed locally or via ssh-tunnel
our $DEBUG_PRINT_CMDS = defined $ENV{TRAVERTINE_PRINTCMDS} ? 
    $ENV{TRAVERTINE_PRINTCMDS} : 0;

# if true, then save a copy of all perl script stubs in LOCAL_STUB_DIR
# (never saved in remote REMOTE_STUB_DIR)
our $DEBUG_SAVE_STUBS = defined $ENV{TRAVERTINE_SAVESTUBS} ? 
    $ENV{TRAVERTINE_SAVESTUBS} : 0;

# connection timeout for initiated ssh connections (in sec)
our $SSH_CONNECT_TIMEOUT = defined $ENV{TRAVERTINE_SSHTIMEOUT} ?
    $ENV{TRAVERTINE_SSHTIMEOUT} : 10;

# tunnel all ssh connections through this machine (i.e., instead of
# 'ssh host cmd' do 'ssh tunnel ssh host cmd'. None if undef.
our $SSH_TUNNEL_HOST = defined $ENV{TRAVERTINE_SSHTUNNELHOST} ?
    $ENV{TRAVERTINE_SSHTUNNELHOST} : undef;

# default ssh protocol version
our $SSH_PROTO_VERSION = defined $ENV{TRAVERTINE_SSHVERSION} ?
    $ENV{TRAVERTINE_SSHVERSION} : '2,1'; # planetlab requires 2,1

###############################################################################
# Options:

our $VERSION = '$Id: Travertine.pm 3699 2007-06-07 00:34:13Z jeffpang $';

# Ssh options passed to Net::SSH::Perl
#our %SSH_DEFAULT_OPTS = ( 'protocol' => '1,2',
#			  'options'  => [ "StrictHostKeyChecking no",
#					  "ForwardX11 no",
#					  "ForwardAgent yes" ] );
our @SSH_DEFAULT_OPTS = ( "-oProtocol=$SSH_PROTO_VERSION", 
			  '-oStrictHostKeyChecking=no',
			  '-oForwardX11=no',
			  '-oForwardAgent=yes' );

# Modules available for remote functions
our @REMOTEFUNC_MODULE_EXPORT = ( #"Thread",
				  "Socket",
				  "IO::File",
				  "Data::Dumper",
				  "Errno qw(EAGAIN ESRCH EPERM)",
				  "POSIX qw(:sys_wait_h)",
				  "POSIX qw(strftime)",
				  "File::Temp qw/tempfile tempdir/",
				  "Carp",
				  "Carp qw(cluck)" );
				  
# Functions exported and available in ExecRemoteFunc. NOTE: Ensure that any
# function in this list only uses modules that are declared above!
# Do not put functions in here if they depend on non-standard modules.
# Each key should be the name of the desired variable in the remote script
# Each val should be a reference to the desired value the variable is
# initialized to (e.g., $foo = '10' should be ( 'foo' => \$foo ) );
our %REMOTEFUNC_GLOBAL_EXPORT = ( 'LOG_DEFAULT_DIR'             => 
				  \$LOG_DEFAULT_DIR,
				  'REMOTE_STAT_DIR'             => 
				  \$REMOTE_STAT_DIR,
				  'REMOTE_STUB_DIR'             => 
				  \$REMOTE_STUB_DIR,
				  'LOCAL_STUB_DIR'              =>
				  \$LOCAL_STUB_DIR,
				  'DEBUG_COLOR_STACKTRACE'     => 
				  \$DEBUG_COLOR_STACKTRACE,
				  'DEBUG_PRINT_CMDS'            => 
				  \$DEBUG_PRINT_CMDS,
				  'Travertine::fancymsg'        => 
				  \&Travertine::fancymsg,
				  'Travertine::rerror'          => 
				  \&Travertine::rerror,
				  'Travertine::psystem'         => 
				  \&Travertine::psystem,
				  'Travertine::ppsystem'         => 
				  \&Travertine::ppsystem,
				  'Travertine::pbacktick'         => 
				  \&Travertine::pbacktick,
				  'Travertine::tcolorize'       => 
				  \&Travertine::tcolorize,
				  'Travertine::thighlight'       => 
				  \&Travertine::thighlight,
				  'Travertine::tinfo'           => 
				  \&Travertine::tinfo,
				  'Travertine::twarn'           => 
				  \&Travertine::twarn,
				  'Travertine::tdie'            => 
				  \&Travertine::tdie,
				  'Travertine::encode_str'      => 
				  \&Travertine::encode_str,
				  'Travertine::decode_str'      => 
				  \&Travertine::decode_str,
				  'Travertine::ssh'             => 
				  \&Travertine::ssh,
				  'Travertine::cfork'           => 
				  \&Travertine::cfork,
				  'Travertine::ChildFunc::new'  => 
				  \&Travertine::ChildFunc::new,
				  'Travertine::ChildFunc::isAlive' => 
				  \&Travertine::ChildFunc::isAlive,
				  'Travertine::ChildFunc::stop' => 
				  \&Travertine::ChildFunc::stop,
				  'Travertine::ChildFunc::wait' => 
				  \&Travertine::ChildFunc::wait,
				  'Travertine::ChildFunc::serialize'   => 
				  \&Travertine::ChildFunc::serialize,
				  'Travertine::ChildFunc::deserialize' => 
				  \&Travertine::ChildFunc::deserialize,
				  'Travertine::ParallelExec'    => 
				  \&Travertine::ParallelExec,
				  'Travertine::ParallelExec0'   => 
				  \&Travertine::ParallelExec0,
				  'Travertine::ParallelExec2'   => 
				  \&Travertine::ParallelExec2,
				  'Travertine::ParallelExec3'   => 
				  \&Travertine::ParallelExec3,
				  'Travertine::ParallelExec4'   => 
				  \&Travertine::ParallelExec4);

###############################################################################
# Data structures:

# A function that is executed as a local child process. We can later wait on
# this process and obtain the function's return value. You may specify that
# the function be "detached" in which case you can even wait on the function
# later after the parent process has exited.
#
# childFunc = { 
#     pid  => pid of forked child process
#     fd   => readable/writable end of pipe (only if non-detached)
#     stat => status file where result is written (only if detached)
#     time => time the proc was started (unix time)
# }

# An ssh tunnel opened to a remote host.
#
# sshTunnel = {
#     ssh  => Net::SSH::Perl instance
#     host => target host
#     user => logined as user
# }

# A "remote daemon" intended to execute long-lived processes. A daemon is
# executed within an instance of screen on a remote host, so that it can
# be re-attached to later. The output of the daemon is also directed to
# a log.
#
# rdRef = {
#	uid    => unique-id (identifies screen session name)
#	host   => host executed on
#	user   => user loged in as
#	pid    => pid of screen process
#	cmd    => command executed in screen
#	ssh    => open sshTunnel to the remote host
#	log    => path to output log on remote host
#       time   => time the proc was started (unix time)
# }

# A "remote background function" is a perl function that is executed in the
# background on a remote host. This is really just a remote instance of
# a detached childFunc.
#
# rbRef = {
#       cfs  => serialized form of the remote childFunc 
#       ssh  => open ssh connection to the remote host
#       log  => path to output log on the remote host (if any)
#       time => time the proc was started (unix time)
# }

###############################################################################
# General functions:

# Usage:
#    tr_export_func($function_name, $function_ref)
#
# Example:
#    sub my_func {}
#    tr_export_func('my_func', \&my_func);
#
# Notes:
#    Causes the function $function_name to be exported in remote execution
sub tr_export_func($$) {
    my $fname = shift;
    my $func = shift;
    $REMOTEFUNC_GLOBAL_EXPORT{$fname} = $func;
}

# Usage:
#    tr_export_scalar($scalar_name, \$scalar_ref)
#
# Example:
#    $foo = 2;
#    tr_export_scalar('foo', \$foo);
#
# Notes:
#    Causes the scalar variable $scalar_name to be exported in remote execution
sub tr_export_scalar($$) {
    my $vname = shift;
    my $var = shift;
    $REMOTEFUNC_GLOBAL_EXPORT{$vname} = $var;
}

# Usage:
#    psystem($cmd)
#
# Notes:
#    Run system() on a command and possibly echo it to the screen.
sub psystem {
    tinfo("$_[0]") if $DEBUG_PRINT_CMDS;
    return system($_[0]);
};

# Usage:
#    ppsystem($cmd)
#
# Notes:
#    Run system() on a command and _always_ echo it to the screen.
sub ppsystem {
    tinfo ("$_[0]");
    return system ($_[0]);
}

# Usage:
#    ptick($cmd)
#
# Notes:
#    Run `command` (possibly echoing on screen) and return the
#    return-value.
sub pbacktick {
    tinfo("$_[0]") if $DEBUG_PRINT_CMDS;
    return `$_[0]`;
}

# deprecated!
sub rerror {
    my $msg = shift;
    our $node;

    print STDERR "\n[31mERROR @ [35m$node[m: $msg\n";
    sleep(500);
    exit 4;
}

# deprecated!
sub fancymsg {
    my $msg = shift;
    our $node;

    print STDERR "\n[34mMSG @ [35m$node[m: $msg\n";
    exit 4;
}

# colorize stack trace
sub tcolorize($)
{
    my @lines = split(/(\n)/, shift());

    my $skip = 0;

    foreach my $line (@lines) {
	# don't over color parts that are already colored
	$skip = 1 if ($line =~ /\[\d*(;\d+)?m/);
	$skip = 0 if ($line =~ /\[m/);
	next if $skip;

	# XXX: This might gobble up some lines we don't want it to...
	if ($line =~ /^\t([\w_:]+)\((.*)\) called at (.*) line (\d+)$/) {
	    my ($func, $args, $file, $lineno) = ($1, $2, $3, $4);
	    $line = "\t[36m$func[m($args) called at $file line $lineno";
	}
    }

    return join('', @lines);
}

# colorize string with background
sub thighlight($@)
{
    my ($str, $fgcolor, $bgcolor) = @_;
    $bgcolor = 44 if !defined $bgcolor; # blue
    $fgcolor = 36 if !defined $fgcolor; # white

    my @lines = split(/(\n)/, $str);
    map { $_ =~ s/(\[\d+(;\d+)?m)/\[m$1/g; 
	  $_ =~ s/(\[m)/$1\[${fgcolor};${bgcolor}m/g; 
	  $_} @lines;
    unshift @lines, "[${fgcolor};${bgcolor}m";
    if ($lines[$#lines] =~ /(.*)\n$/) {
	$lines[$#lines] = "$1[m\n";
    } else {
	push @lines, "[m";
    }
    return join('', @lines);
}

# Usage:
#    tinfo(@msgs)
#
# Notes:
#    Display an informational message.
sub tinfo(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "\@[35m$node[m ";
    }
    
    print STDERR ("[30;42m[II][m $host", @_, "\n");
}

# Usage:
#    twarn(@msgs)
#
# Notes:
#    Display an warning message and stacktrace.
sub twarn(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "[35m$node[m ";
    }
    if ($DEBUG_SHOW_STACKTRACE) {
	my $msg = Carp::longmess("[30;41m[EE][m $host", @_);
	$msg = tcolorize $msg if $DEBUG_COLOR_STACKTRACE;
	print STDERR $msg;
    } else {
	print STDERR ("[30;41m[EE][m $host", @_, "\n");
    }
}

# Usage:
#    tdie(@msgs)
#
# Notes:
#    Display an fatal message and stacktrace.
sub tdie(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "[35m$node[m ";
    }

    if ($DEBUG_SHOW_STACKTRACE) {
	my $msg = Carp::longmess("[30;41m[FF][m $host", @_);
	$msg = tcolorize $msg if $DEBUG_COLOR_STACKTRACE;
	print STDERR $msg;
    } else {
	print STDERR ("[30;41m[FF][m $host", @_, "\n");
    }
    exit 255;
}

sub encode_str {
    my $str = shift;

    $str =~ s/(.)/ sprintf "%02X", unpack("C", $1) /eg;
    return $str;
}

sub decode_str {
    my $str = shift;

    $str =~ s/(..)/ pack("C", hex $1) /eg;
    return $str;
}

###############################################################################
# class ChildFunc

package Travertine::ChildFunc;
#use Thread;
use Socket;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;

use overload q("") => sub { 
    my $self = shift;
    #my $thread = ($self->{thread} ? 1 : 0);
    my $detached = ($self->{stat} ? 1 : 0);
    #if ($thread) {
    #	return "(ChildFunc thread=$self->{thread})";
    #}
    return "(ChildFunc pid=$self->{pid} detached=$detached" .
	($detached?" stat=$self->{stat}":"") . " time=\"" . 
	scalar(localtime($self->{time})) . "\")";
};

# Usage:
#    Travertine::ChildFunc->new(\&func, \@args [, %opts])
#
# Args:
#    \&func - function to execute
#    \@args - arguments to execute with
#    %opts  - options:
#       -thread = 1. run in a thread instead of a process. this allows you to
#                 return non-trivial datastructures like filehandles and 
#                 sockets, but you can't detach from this. DISABLED!
#       -detach = 1. don't open a pipe, but write the result to REMOTE_STAT_DIR
#                 and don't die on SIGHUP. A detached childFunc will keep
#                 running after it has parent has exited and you can still
#                 use WaitChildFunc on it later.
#       -log    = string. If detached, then write stdout/stderr to this file.
#                 (otherwise they will be closed for detached childFuncs)
#       -rotate = int. rotate the output log every x lines.
#
# Returns:
#    childFunc on success, undef on fork() failure;
#
# Notes:
#    Run a function in a child process. Returns immediately; use
#    WaitChildFunc() to "join" the child process and retrieve the return
#    value. Assumes the function is called in scalar context.
sub new($$$%) {
    my $cls  = shift;
    my $func = shift;
    Travertine::tdie "not a \&func" if ref($func) ne 'CODE';

    my @args = @{shift()};
    my %opts = @_;

#    if ($opts{-thread}) {
#	my $thread = Thread->new($func, @args);
#
#	my $ret = {
#	    'pid'    => undef,
#	    'fd'     => undef,
#	    'stat'   => undef,
#	    'thread' => $thread,
#	    'time'   => time(),
#	};
#	return bless $ret, "Travertine::ChildFunc";
#    }
    
    my $retries = 5;

    my ($childfh, $parentfh);
    if (!$opts{-detach}) {
	socketpair($childfh, $parentfh, AF_UNIX, SOCK_STREAM, PF_UNSPEC);
	Travertine::tdie "can't open pipes" if !$childfh || !$parentfh;
    }

    my $out;
    if ($opts{-detach} && defined $opts{-log}) {
	# make sure we can create the output log
	my $log = $opts{-log};
	if ($opts{-rotate}) {
	    Travertine::twarn "bad -rotate option $opts{-rotate}" if 
		!($opts{-rotate} > 0);
	      my $dumper = Data::Dumper->new([\&Travertine::ChildFunc::logrotate], ['logrotate']);
	      $dumper->Indent(0);
	      my $prog = $dumper->Dump();
	      open(L, ">$REMOTE_STUB_DIR/lr.$$.pl") or
		  Travertine::twarn "couldn't create $REMOTE_STUB_DIR/lr.$$.pl script: $!" && return undef;
	      print L "use IO::File;\n";
	      print L "unlink '$REMOTE_STUB_DIR/lr.$$.pl';\n";
	      print L <<'EOT';
sub logrotate {
    my ($LOGNAME, $MAXLINES) = @_;
    my $lines = 0;
    my $start_time = time();
    my $end_time;
    my $log = new IO::File(">$LOGNAME");
    autoflush $log 1;
    die "can't create $LOGNAME" if !$log;
    close STDOUT;
    close STDERR;
    my $pid = fork();
    if ($pid > 0) {
	exit 0;
    } elsif (!defined $pid) {
	die "fork failed!";
    }
    # child
    $SIG{HUP} = 'IGNORE';
    while(<STDIN>) {
	++$lines;
	print $log $_;
	
	if ($lines >= $MAXLINES) {
	    close($log);
	    $end_time = time();
	    
	    system("mv -f $LOGNAME $LOGNAME.$start_time-$end_time");
	    
	    $start_time = time();
	    $log = new IO::File(">$LOGNAME");
	    autoflush $log 1;
	    die "can't create $LOGNAME" if !$log;
	    $lines = 0;
	}
    }
}
EOT
	      print L "logrotate('$log', $opts{-rotate});\n";
	      close(L);
	      $out = IO::File->new("| perl $REMOTE_STUB_DIR/lr.$$.pl");
	  } else {
	      $out = IO::File->new(">$log");
	  }
	if (!$out) {
	    Travertine::twarn "couldn't create childFunc output log $log: $!";
	    return undef;
	}
    }

    while ($retries-- > 0) {
	my $pid = fork();
	
	if ($pid > 0) {
	    # parent
	    close($parentfh) if !$opts{-detach};
	    my $ret = {
		'pid'  => $pid,
		'fd'   => !$opts{-detach} ? $childfh : undef,
		'stat' => ($opts{-detach} ? 
			   "$REMOTE_STAT_DIR/ChildFunc.$pid.stat" : undef),
		'time' => time(),
		'log'  => $opts{-log},
	    };
	    return bless $ret, "Travertine::ChildFunc";
	} elsif (defined $pid) {
	    # child
	    if (!$opts{-detach}) {
		close($childfh);
	    } else {
		$SIG{HUP} = 'IGNORE';
		close STDIN;
		if ($out) {
		    STDOUT->fdopen($out, 'w');
		    STDERR->fdopen($out, 'w');
		} else {
		    # if we don't redirect, then close them
		    close(STDOUT);
		    close(STDERR);
		}

		$parentfh = 
		    IO::File->new(">$REMOTE_STAT_DIR/ChildFunc.$$.stat");
		Travertine::tdie "can't create stat file for $$: $!" 
		    if !$parentfh;
	    }
	    my $ret = (&{$func}(@args));
	
	    my $dumper = Data::Dumper->new([$ret], ['datum']);
	    #$dumper->Terse(1);
	    $dumper->Purity(1);
	    $dumper->Indent(0);
	    my $text = $dumper->Dump();
	    
	    print $parentfh ( $text );
	    close($parentfh);
	    exit 0;
	} elsif ($! == EAGAIN) {
	    sleep 5;
	    next;
	} else {
	    # bad fork error
	    Travertine::twarn "fork failed: $!";
	    return undef;
	}
    }
}

# Usage:
#     childFunc->serialize()
#
# Returns:
#     String representation of a detached childFunc.
sub serialize($)
{
    my $cf = shift;
    Travertine::tdie "not a childFunc" if ref($cf) ne 'Travertine::ChildFunc';
    #Travertine::tdie "can't serialize a thread" if $cf->{thread};

    if ($cf->{fd} || !$cf->{stat}) {
	Travertine::tdie "can't seriliaze non-detached childFunc!";
    }

    return "$cf->{pid}==$cf->{time}==$cf->{log}";
}

# Usage:
#    Travertine::ChildFunc::deserialize($string)
#
# Returns:
#    The detached childFunc the string represents.
sub deserialize($)
{
    my $str = shift;
    if ($str !~ /^\d+==\d+==/) {
	Travertine::twarn "invalid string";
	  return undef;
      }
    my ($pid, $time, $log) = split(/==/, $str);

    my $cf = {
	'pid'  => $pid,
	'fd'   => undef,
	'stat' => "$REMOTE_STAT_DIR/ChildFunc.$str.stat",
	'time' => $time,
	'log'  => $log,
    };
    return bless $cf, "Travertine::ChildFunc";
}

# Usage:
#     childFunc->isAlive()
#
# Returns:
#     1 if yes or unknown, 0 if no
sub isAlive($)
{
    my $ref = shift;
    Travertine::tdie "not a childFUnc" if ref($ref) ne 'Travertine::ChildFunc';
    if ($ref->{pid} < 0) {
	return 0;
    }
    
    my $ret;
    if (kill 0, $ref->{pid}) {
	$ret = 1;
    } elsif ($! == EPERM) {
	$ret = 2;
    } elsif ($! == ESRCH) {
	$ret = 0;
    } else {
	$ret = 3;
    }

    if ($ret == 2) {
	Travertine::twarn "don't have permission to signal $ref->{pid}";
    } elsif ($ret == 3) {
	Travertine::twarn "error trying to signal $ref->{pid}";
    }
    
    return $ret;
}

# Usage:
#     childFunc->stop([signal])
#
# Returns:
#     1 on successful signal, undef otherwise
sub stop($@)
{
    my $cf = shift;
    Travertine::tdie "not a childFunc" if ref($cf) ne 'Travertine::ChildFunc';
    #Travertine::tdie "can't stop a thread" if $cf->{thread};

    my $sig = shift;
    $sig = 15 if !defined $sig;

    my $ret;

    if ( (kill $sig, $cf->{pid}) > 0 ) {
	$ret = 1;
    } elsif ($! == EPERM) {
	Travertine::twarn "don't have permission to signal $cf->{pid}";
	$ret = undef;
    } elsif ($! == ESRCH) {
	# ok -- not found, means it is dead
	$ret = 1;
    } else {
	Travertine::twarn "unknown error signalling $cf->{pid}";
	$ret = undef;
    }

    if (defined $cf->{fd}) {
	close $cf->{fd};
    } elsif (defined $cf->{stat}) {
	unlink $cf->{stat};
    }

    return $ret;
}

# Usage:
#   childFunc->wait()
#
# Args:
#    childFunc - reference to the childFunc created by CreateChildFunc
#
# Returns:
#    return value of the function executed in childFunc. OPEN FILEHANDLES
#    AND SOCKETS WILL NOT BE RETURNED CORRECTLY!!!
#
# Notes:
sub wait($) {
    my $cf = shift;
    Travertine::tdie "not a childFunc" if ref($cf) ne 'Travertine::ChildFunc';

    my $retval;

#    if (defined $cf->{thread}) {
#	return $cf->{thread}->join();
#    } elsif 
    if (defined $cf->{fd}) {
	# non-detached process...
	my $pid = waitpid $cf->{pid}, 0;
	if ($pid != $cf->{pid} && $pid != -1) {
	    # possibly already waited for it, so let -1 be ok...
	    Travertine::tdie 
		"waitpid failed! returned $pid, but expected pid $cf->{pid}";
	}
    
	my $fh = $cf->{fd};
	while(<$fh>) {
	    $retval .= $_;
	}

	close($fh);
    } elsif (defined $cf->{stat}) {
	# detached process...
	# the target might be a child, in which case, try to reap it
	waitpid $cf->{pid}, 0;
	while (kill 0, $cf->{pid}) {
	    # child is still alive
	    sleep 1;
	}
	if ($! == EPERM) {
	    Travertine::twarn "don't have permission to signal $cf->{pid}";
	    return undef;
	} elsif ($! == ESRCH) {
	    # looks like pid is gone, read stat
	    my $fh = 
		IO::File->new("<$REMOTE_STAT_DIR/ChildFunc.$cf->{pid}.stat");
	    if (!$fh) {
		Travertine::twarn "no status file for $cf->{pid}";
		return undef;
	    }
	    while (<$fh>) {
		$retval .= $_;
	    }
	    close $fh;
	    unlink $cf->{stat};
	}  else {
	    Travertine::twarn "unknown error signalling $cf->{pid}";
	    return undef;
	}

    } else {
	# errorenous process!
	Travertine::tdie "child func ref doesn't have fh or stat file!";
    }

    my $val = eval { no strict; eval $retval; die $@ if ($@); $datum; };
    #Travertine::tinfo " >> $retval";
    #Travertine::tinfo " >> $val";
    if ($@) {
	Travertine::twarn 
	    "failed to eval serialized return value ($retval): $@"
    }

    return $val;
}


###############################################################################
# class SSHTunnel

package Travertine::SSHTunnel;
use Socket;
use IPC::Open3;
use Data::Dumper;
use Fcntl qw(F_GETFL F_SETFL);
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;

use overload q("") => sub { 
    my $self = shift;
    return "(SSHTunnel host=$self->{host} user=$self->{user})";
};

# Usage:
#    Travertine::SSHTunnel->new($user, $host, [@options])
# 
# Args:
#    $user - username of login (assume password is loaded into ssh-agent)
#    $host - hostname of login
#    @options - a string of options to pass to ssh
#
# Returns:
#    sshTunnel on success
#    undef on error
#
# Notes:
#    If the tunnel is required to fall back on to SSH Protocol 1, *each*
#    new command executed will require a new connection, so it will be
#    relatively show compared to SSH Protocol 2. However, the authentication
#    process in SSH 2 is slower (usually), so there is a trade-off (if you
#    execute 5 or more different commands, SSH2 is probably faster).
sub new($$$%) {
    my $cls = shift;
    my $user = shift;
    my $host = shift;
    my @uopts = @_;
    
    my @opts = (@SSH_DEFAULT_OPTS, @uopts);
    # user options override defaults
    #foreach my $k (%uopts) {
    #	$opts{$k} = $uopts{$k};
    #}

    my $sshTunnel = {};
    
    # two-levels of eval to avoid race conditions
#    eval {
#	local $SIG{ALRM} = sub { die "alarm clock restart" };
#	alarm $SSH_CONNECT_TIMEOUT;
#	eval {
	    # XXX PERL IPC F*CKING SUCKS, THERE IS NO F*CKING WAY 
	    # TO PASS FILEHANDLES AND SOCKETS AROUND WITHOUT SEGFAULTING
	    # THE F*CKING PROCESS!!! SO F*CK IT! :(
	    #$sshTunnel->{ssh} = Net::SSH::Perl->new($host, %opts);
	    $sshTunnel->{ssh} = 'dummy';
	    $sshTunnel->{opts} = \@opts;
#	};
#	alarm 0;
#
#	if ($@ && $@ =~ /alarm clock restart/) {
#	    die "ssh connect timed-out";
#	} elsif ($@) {
#	    die $@; # propagate error
#	}
#    };
#    alarm 0;
#    if ($@) {
#	Travertine::twarn $@;
#	return undef;
#    }

    # XXX SEE ABOVE
    #if (!$sshTunnel->{ssh}) {
    #	return undef;
    #}
    #$sshTunnel->{ssh}->login($user);
    $sshTunnel->{host} = $host;
    $sshTunnel->{user} = $user;

    return bless $sshTunnel, "Travertine::SSHTunnel";
}

# Usage:
#    sshTunnel->close()
#
# Args:
#    sshTunnel - the tunnel to close
#
# Returns:
#    Void. 
sub close($)
{
    my $ssh = shift;
    Travertine::tdie "not an sshTunnel" 
	if ref($ssh) ne 'Travertine::SSHTunnel';

    # XXX SEE ABOVE
    #close $ssh->{ssh}->sock();
}

sub _setnonblocking($)
{
    my $handle = shift;
    my $flags = '';
    fcntl($handle, F_GETFL, $flags) or return undef;
    $flags |= O_NONBLOCK;
    fcntl($handle, F_SETFL, $flags) or return undef;

    return "0";
}

# Usage:
#    sshTunnel->exec($cmd [, $stdin [, $timeout]])
#
# Args:
#    sshTunnel - tunnel open to the remote host
#    $cmd - the command to run on the remote host
#    $stdin - pass this to the stdin of the cmd
#    $timeout = optional command timeout
#
# Returns:
#    ($stdout, $stderr, $retval) of the executed command.
#
# Notes:
#    This creates a remote process.
sub exec($$@)
{
    my $ssh = shift;
    Travertine::tdie "not an sshTunnel" 
	if ref($ssh) ne 'Travertine::SSHTunnel';

    my $cmd = shift;
    my $stdin = shift;
    my $timeout = shift;
    $timeout = 0 if !$timeout;

    my @args = ($cmd);
    if ($stdin) {
	push @args, $stdin;
    }
    
    if ($DEBUG_PRINT_CMDS > 1) {
	if ($cmd =~ /^zcat -f/) { 
	    Travertine::tinfo ("[35m$ssh->{host}[m Transmitting script");
	} elsif ($cmd =~ /^perl \/tmp/) { 
	    Travertine::tinfo ("[35m$ssh->{host}[m Executing script");
	} else {	    
	    Travertine::tinfo("ssh [35m$ssh->{host}[m $cmd");
	}
    }

    my @ret;
    # two-levels of eval to avoid race conditions
#    eval {
#	local $SIG{ALRM} = sub { die "alarm clock restart" };
#	alarm $timeout;
#	eval {
#	    # XXX See above
#	    #@ret = $ssh->{ssh}->cmd(@args);
#
#	    # Instead just do it all in one shot :P
#	    my $tunnel;
#	    eval {
#		alarm $SSH_CONNECT_TIMEOUT;
#		$tunnel = Net::SSH::Perl->new($ssh->{host}, %{$ssh->{opts}});
#		die "couldn't open ssh to $ssh->{host}" if !$tunnel;
#		$tunnel->login($ssh->{user});
#	    };
#	    alarm 0;
#	    if ($@) {
#		# propagate
#	        die $@;
#	    }
#	    
#	    @ret = $tunnel->cmd(@args);
#	    #close($tunnel->sock());
#	    # XXX End
#	};
#	alarm 0;
#	
#	if ($@ && $@ =~ /alarm clock restart/) {
#	    die "ssh exec timed-out";
#	} elsif ($@) {
#	    die $@; # propagate error
#	}
#    };
#    alarm 0;
#    if ( $@ ) {
#	Travertine::twarn "exec failed on $ssh: $@";
#	return undef;
#    }

    my($out, $err, $exit);
    my $pid;

    $SIG{PIPE} = sub {};
    eval {
	local $SIG{ALRM} = sub { die "alarm clock restart" };
	alarm $timeout;
	eval { # ============================================================ #

	    my @cmd = ("ssh", @{$ssh->{opts}},
		       "$ssh->{user}\@$ssh->{host}", $cmd);
	    if (defined $ENV{TRAVERTINE_SSHTUNNELHOST}) {
		@cmd = map { $_ = quotemeta $_; $_; } @cmd;
		@cmd = ("ssh", @{$ssh->{opts}},
			$ENV{TRAVERTINE_SSHTUNNELHOST}, @cmd);
	    }
	    #Travertine::tinfo "@cmd";
	    
	    # XXX: My god this is such a HACK. I hate perl. :P
	    my($writer, $reader);
	    local *errout = IO::File->new_tmpfile;
	    $pid = open3($writer, $reader, ">&errout", @cmd);
	    if ($pid < 1) {
		Travertine::twarn "open3 of ssh tunnel failed: $!";
		  return (undef, undef, -1);
	      }
	    if (defined $stdin) {
		print $writer $stdin;
	    }
	    close($writer);

	    if (0) {
# ASHWIN: add support for reading stdout/stderr of the ssh
# tunnel immediately...

		_setnonblocking(\*errout);

		my $sel = new IO::Select();
		$sel->add(\*errout);	    

		my $we;

		while (($we = waitpid($pid, &POSIX::WNOHANG())) == 0) {
		    my $buf = "";
		    if ($sel->can_read(0.5)) {		    
			sysread($reader, $buf, 1024);
			print STDERR $buf;
			$err .= $buf;
		    }		    
		}

		if ($we < 0) { 
		    Travertine::twarn "waitpid error: $!";
		    return (undef, undef, -1);
		}

		Travertine::tinfo "ssh tunnel has ended....";
	    }
	    else {
		waitpid($pid, 0);
	    }
	    
	    $exit = $? >> 8;

	    if (0) {

	    }
	    else {
		while (<$reader>) {
		    $out .= $_;
		}
		close($reader);

		seek errout, 0, 0;
		while (<errout>) {
		    $err .= $_;
		}
	    }
	    close(errout);

	};     # ============================================================ #
	alarm 0;
	if ($@) {
	    # propagate
	    die $@;
	}
    };
    alarm 0;
    $SIG{PIPE} = undef;
    if ( $@ ) {
	Travertine::twarn "exec failed on $ssh: $@";
	if ($pid > 0) {
	    kill 9, $pid;
	}
	return undef;
    }

    return ($out, $err, $exit);
}

###############################################################################
# class RemoteDaemon

package Travertine::RemoteDaemon;
use Socket;
use Data::Dumper;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;

use overload q("") => sub { 
    my $self = shift;
    return "(RemoteDaemon ssh=$self->{ssh} uid=$self->{uid} pid=$self->{pid} cmd={$self->{cmd}} log=$self->{log} time=\"" . scalar(localtime($self->{time})) . "\")";
};

# Usage:
#     rdRef->serialize()
#
# Args:
#     rdRef - a reference to the remote daemon
#
# Returns:
#     A string form of the reference that can be written to disk.
sub serialize($) {
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';

    my $cmd = Travertine::encode_str($ref->{cmd});
    my $log = Travertine::encode_str($ref->{log});

    return "$ref->{host}:$ref->{user}:$ref->{uid}:$ref->{pid}:$cmd:$log:$ref->{time}";
}

# Usage:
#     Travertine::RemoteDaemon::deserialize($string [, sshTunnel])
#
# Args:
#     $string - the serialized reference to the remote daemon
#     sshTunnel - an optional sshTunnel to use in the reference;
#     if not supplied, will create a new one.
#
# Returns:
#     A reference to the remote daemon WITH an open ssh connection.
sub deserialize($@) {
    my $str = shift;
    my ($ssh) = @_;
    Travertine::tdie "not an sshTunnel" if $ssh && ref($ssh) ne 'Travertine::SSHTunnel';
    my ($host, $user, $uid, $pid, $cmd, $log, $time) = split(/:/, $str);
    if (!$host || !$user || $uid !~ /^\d+$/ || $pid !~ /^\d+$/) {
	Travertine::twarn "invalid string";
	  return undef;
      }
	

    $ssh = defined $ssh ? $ssh : 
	Travertine::SSHTunnel->new($user, $host);
    if (!$ssh) {
	return undef;
    }

    my $ret = {
	'uid'  => $uid,
	'host' => $host,
	'user' => $user,
	'pid'  => $pid,
	'cmd'  => Travertine::decode_str($cmd),
	'ssh'  => $ssh,
	'log'  => Travertine::decode_str($log),
	'time' => $time,
    };

    return bless $ret, "Travertine::RemoteDaemon";
}

# Usage:
#     Travertine::RemoteDaemon->new(sshTunnel, $cmd [, %opts])
#
# Args:
#     sshTunnel - the open tunnel to the remote host
#     $cmd - the command to execute on the remote host (within screen)
#     %opts - options
#        -title - title of screen session (default: $hostname)
#        -log   - output log file path
#
# Returns:
#     An rdRef to the newly started process. undef on error. If the process
#     started within screen dies before we can grab the pid, then the pid
#     of the returned rdRef == -1.
#
# Notes:
#     This function returns as soon as the detached screen process is
#     executed.
sub new($$$%)
{
    my $cls = shift;
    my $ssh = shift;
    Travertine::tdie "not an sshTunnel" 
	if ref($ssh) ne 'Travertine::SSHTunnel';
    my $cmd = shift;
    my %opts = @_;

    my $uid = int(rand(0xFFFFFFFF));
    my $ret = {
	'uid'    => $uid,
	'host'   => $ssh->{host},
	'user'   => $ssh->{user},
	'pid'    => undef,
	'cmd'    => $cmd,
	'ssh'    => $ssh,
	'log'    => undef,
	'time'   => time(),
    };
    $ret = bless $ret, "Travertine::RemoteDaemon";
    
    $ret->{log} = defined $opts{-log} ? $opts{-log} : 
	"$LOG_DEFAULT_DIR/$ret->{user}.$ret->{host}.$ret->{uid}.log";

    my $title = defined $opts{-title} ? $opts{-title} : $ret->{host};
    
    # run detached screen instance with session name = uid
    $cmd =~ s/\\/\\\\/g;
    $cmd =~ s/\"/\\\"/g;
    my $run = "screen -t \"$title\" -S $ret->{uid} -d -m sh -c \"$cmd 2>&1 | tee $ret->{log}\"";

    my ($out, $err, $stat) = $ssh->exec($run);
    if ($stat) {
	Travertine::twarn 
	    "remote screen exec failed (exit status: $stat):\n" .
	    Travertine::thighlight($err);
	return undef;
    }

    # get the pid of the screen process
    my $getpid = "sh -c 'screen -ls | grep $uid'";
    my ($out, $err, $exit) = $ssh->exec($getpid);
    if ($out =~ /(\d+)\.$uid/) {
	$ret->{pid} = $1;
    } else {
	Travertine::twarn 
	    "couldn't get pid for remote screen (output: $out) " .
	    "$ret->{user}:$ret->{host}:$uid (exit status: $exit):\n" .
	    Travertine::thighlight($err);
	Travertine::twarn
	    "... tail of the process's log:\n" . 
	    Travertine::thighlight($ret->tailLog(10));
	$ret->{pid} = -1;
    }
    
    return $ret;
}

# Usage:
#     rdRef->DaemonStop()
#
# Args:
#     rdRef - the remote daemon to stop
#
# Returns:
#     undef on success; error message on error.
#
# Notes:
#     This does NOT close the sshTunnel in the ref.
sub stop($@)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';

    my $sig = shift;
    $sig = 15 if (!defined $sig);

    my $kill = "kill -$sig $ref->{pid}";
    my (undef, $err, $retval) = $ref->{ssh}->exec($kill);
    if ($retval) {
	return $err;
    }

    return undef;
}

sub isAlive($)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';
    if ($ref->{pid} < 0) {
	return 0;
    }
    my $ret = Travertine::ExecRemoteFunc($ref->{ssh},
					 sub {
					     my $pid = shift;
					     
					     if (kill 0, $pid) {
						 return 1;
					     } elsif ($! == EPERM) {
						 return 2;
					     } elsif ($! == ESRCH) {
						 return 0;
					     } else {
						 return 3;
					     }
					 }, [$ref->{pid}]);

    if ($ret == 2) {
	Travertine::twarn "don't have permission to signal $ref->{pid}";
    } elsif ($ret == 3) {
	Travertine::twarn "error trying to signal $ref->{pid}";
    }
    
    return $ret;
}

sub _ProcLog($$)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';
    my $cmd = shift;
    
    if (!$ref->{log}) {
	Travertine::twarn("ref doesn't have a log; ignoring");
	return(undef, 1);
    }
    
    my ($out, $err, $ret) = 
	$ref->{ssh}->exec("sh -c '$cmd $ref->{log} | gzip -f'");

    if ($ret) {
	Travertine::twarn("error '$cmd $ref->{log}':\n" .
			  Travertine::thighlight($err));
    }
    return (Compress::Zlib::memGunzip($out), $ret);
}

sub headLog($$)
{
    my $ref = shift;
    my $lines = shift;
    if ($lines <= 0) {
	return (_ProcLog($ref, "cat"))[0];
    } else {
	return (_ProcLog($ref, "head -$lines"))[0];
    }
}

sub tailLog($$)
{
    my $ref = shift;
    my $lines = shift;
    if ($lines <= 0) {
	return (_ProcLog($ref, "cat"))[0];
    } else {
	return (_ProcLog($ref, "tail -$lines"))[0];
    }
}

sub removeLog($)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';
    my $ret = (_ProcLog($ref, "rm -f"))[1];
    $ref->{log} = undef;
    return !$ret;
}

sub _RemoteDaemonAttachTermFunc($)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';

    return "ssh -oProtocol=1,2 -oStrictHostKeyChecking=no -oForwardX11=no -t -l $ref->{user} $ref->{host} screen -r $ref->{pid}.$ref->{uid}";
}

# Usage:
#     rdRef->attachTerm()
#
# Args:
#     rdRef - remote daemon to attach to
sub attachToTerm($)
{
    use Term::ReadKey;

    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';

    ReadMode('raw');
    Travertine::psystem(_RemoteDaemonAttachTermFunc($ref));
    ReadMode('restore');
}

# Usage:
#     rdRef->attachXterm([%opts])
#
# Args:
#     rdRef - remote daemon to attach to
#     %opts - options
#         -title = string. window title (default: hostname)
sub attachToXterm($%)
{
    my $ref = shift;
    Travertine::tdie "not an rdRef" if ref($ref) ne 'Travertine::RemoteDaemon';

    my %opts = @_;

    my $title = $opts{-title};
    $title = "$ref->{user}\@$ref->{host}:$ref->{uid}" if !defined $title;

    my $cmd = _RemoteDaemonAttachTermFunc($ref);
    my $run = "xterm -T \"$title\" -sl 5000 -e $cmd";

    tinfo ("Attaching to xterm: $run") if ($DEBUG_PRINT_CMDS > 1);
    system("$run &") and 
	Travertine::tdie ("can't invoke xterm: $!");
}

# Usage:
#      Travertine::RemoteDaemon::AttachGTerm(\@rdRegs [, %opts])
#
# Args:
#     rdRef - remote daemon to attach to
#     %opts - options
#         -titles = \@string. window tab titles (default: hostnames)
sub AttachToGterm($%)
{
    my @refs = @{shift()};
    if (@refs == 0) { return; }
    foreach my $ref (@refs) {
	Travertine::tdie "not an rdRef" 
	    if ref($ref) ne 'Travertine::RemoteDaemon';
    }
    my %opts = @_;

    my $tref = $opts{-titles};
    my @titles;
    if ($tref) {
	@titles = @$tref;
    }
    for (my $i=0; $i<@refs; $i++) {
	$titles[$i] = "$refs[$i]->{user}\@$refs[$i]->{host}:$refs[$i]->{uid}"
	    if !defined $titles[$i];
    }
    
    my $cmd = _RemoteDaemonAttachTermFunc( shift @refs );
    my $title = shift @titles;
    
    my $args = "--command=\"$cmd\" --title \"$title\" ";
    for (my $i=0; $i<@refs; $i++) {
	($title, $cmd) = ($titles[$i], _RemoteDaemonAttachTermFunc($refs[$i]));
	$args .= 
	    "--tab-with-profile=Default " .
	    "--command=\"$cmd\" " .
	    "--title \"$title\" ";
    }
    tinfo ("Attaching to gterm: $args") if ($DEBUG_PRINT_CMDS > 1);
    system("gnome-terminal $args &") and 
	Travertine::tdie ("can't invoke gterm: $!");
}

###############################################################################
# class RemoteBgFunc

package Travertine::RemoteBgFunc;
use Socket;
use Data::Dumper;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;

use overload q("") => sub { 
    my $self = shift;
    return "(RemoteBgFunc ssh=$self->{ssh} cfs=$self->{cfs} log=$self->{log} time=\"" . scalar(localtime($self->{time})) . "\")";
};

# Usage:
#    Travertine::RemoteBgFunc->new($childFuncString, sshTunnel, $logname)
sub new($$$$)
{
    my ($cls, $cfs, $ssh, $log) = @_;

    my $rb = {
	'cfs'  => $cfs,
	'ssh'  => $ssh,
	'log'  => $log,
	'time' => time(),
    };
    
    return bless $rb, "Travertine::RemoteBgFunc";
}

# Usage:
#    rbRef->serialize()
sub serialize($)
{
    my $rb = shift;
    Travertine::tdie "not an rbRef" if ref($rb) ne 'Travertine::RemoteBgFunc';

    my $cfs = Travertine::encode_str($rb->{cfs});
    my $log = Travertine::encode_str($rb->{log});
    
    return "$rb->{ssh}->{host}:$rb->{ssh}->{user}:$cfs:$log:$rb->{time}";
}

# Usage:
#    Travertine::RemoteBgFunc::deserialize()
sub deserialize($@) {
    my $str = shift;
    my ($ssh) = @_;
    Travertine::tdie "not an sshTunnel" if $ssh && ref($ssh) ne 'Travertine::SSHTunnel';
    my ($host, $user, $cfs, $log, $time) = split(/:/, $str);
    if (!$host || !$user || !defined $cfs) {
	Travertine::twarn "invalid string";
	  return undef;
      }

    $ssh = defined $ssh ? $ssh : 
	Travertine::SSHTunnel->new($user, $host);
    if (!$ssh) {
	return undef;
    }

    my $ret = {
	'cfs'  => Travertine::decode_str($cfs),
	'ssh'  => $ssh,
	'log'  => Travertine::decode_str($log),
	'time' => $time,
    };

    return bless $ret, "Travertine::RemoteBgFunc";
}

# Usage:
#    rbRef->stop()
#
# Args:
#    rbRef - the remote background func to stop
#
# Returns:
#    1 if successful, 0 otherwise.
sub stop($)
{
    my $rb = shift;
    Travertine::tdie "not an rbRef" if ref($rb) ne 'Travertine::RemoteBgFunc';
    
    my ($out, $err, $ret) =
	Travertine::ExecRemoteFunc($rb->{ssh}, sub {
	    my $cfs = shift;
	    my $cf  = Travertine::ChildFunc::deserialize($cfs);
	    return defined $cf->stop() ? 0 : 1;
	}, [$rb->{cfs}]);
    
    if ($ret) {
	Travertine::twarn "Stop rbRef failed:\n" . 
	    Travertine::thighlight($err);
	return undef;
    } else {
	if ($err) {
	    print STDERR "$err";
	}
	return 1;
    }
}

# Usage:
#    rbRef->wait()
#
# Args:
#    rbRef - the remote background func to wait on
#
# Returns:
#    The value returned by the remote background process.
#    Will print error message and return undef on error.
#
# Notes:
#    This waits for the return of a remote func that was started
#    via ExecRemoteFunc (with the -background => 1 option).
sub wait($)
{
    my $rb = shift;
    Travertine::tdie "not an rbRef" if ref($rb) ne 'Travertine::RemoteBgFunc';

    my ($out, $err, $ret) =
	Travertine::ExecRemoteFunc($rb->{ssh}, sub {
	    my $cfs = shift;
	    my $cf  = Travertine::ChildFunc::deserialize($cfs);
	    my $ret = $cf->wait();

	    my $dumper = Data::Dumper->new([$ret]);
	    $dumper->Terse(1);
	    $dumper->Indent(0);
	    
	    print STDOUT ( $dumper->Dump() );
	    
	    return 0;
	}, [$rb->{cfs}]);
    
    if ($ret) {
	Travertine::twarn "wait on rbRef failed:\n" .
	    Travertine::thighlight($err);
	return undef;
    } else {
	if ($err) {
	    print STDERR "$err";
	}
	return eval $out;
    }
}

sub isAlive($)
{
    my $ref = shift;
    Travertine::tdie "not an rbRef" if ref($ref) ne 'Travertine::RemoteBgFunc';

    my ($out, $err, $ret) =
	Travertine::ExecRemoteFunc($ref->{ssh}, sub {
	    my $cfs = shift;
	    my $cf  = Travertine::ChildFunc::deserialize($cfs);
	    return $cf->isAlive();
	}, [$ref->{cfs}]);

    return $ret;
}


sub _ProcLog($$)
{
    my $ref = shift;
    Travertine::tdie "not an rbRef" if ref($ref) ne 'Travertine::RemoteBgFunc';
    my $cmd = shift;
    
    if (!$ref->{log}) {
	Travertine::twarn("ref doesn't have a log; ignoring");
	return(undef, 1);
    }

    my ($out, $err, $ret) = 
	$ref->{ssh}->exec("sh -c '$cmd $ref->{log} | gzip -f'");

    if ($ret) {
	Travertine::twarn("error '$cmd $ref->{log}':\n" . 
			  Travertine::thighlight($err));
    }
    return (Compress::Zlib::memGunzip($out), $ret);
}

sub headLog($$)
{
    my $ref = shift;
    my $lines = shift;
    if ($lines <= 0) {
	return (_ProcLog($ref, "cat"))[0];
    } else {
	return (_ProcLog($ref, "head -$lines"))[0];
    }
}

sub tailLog($$)
{
    my $ref = shift;
    my $lines = shift;
    if ($lines <= 0) {
	return (_ProcLog($ref, "cat"))[0];
    } else {
	return (_ProcLog($ref, "tail -$lines"))[0];
    }
}

sub removeLog($)
{
    my $ref = shift;
    Travertine::tdie "not an rbRef" if ref($ref) ne 'Travertine::RemoteBgFunc';
    my $ret = (_ProcLog($ref, "rm -f"))[1];
    $ref->{log} = undef;
    return !$ret;
}

###############################################################################
# General Functions

package Travertine;
use POSIX ":sys_wait_h";

sub ParallelExec0($@)
{
    my $maxparallel = shift @_;
    my @cfs;

    # don't auto-reap
    my $OLD_SIGCHLD = $SIG{CHLD};
    $SIG{CHLD} = undef;

    my %pids;

    my $todo = scalar @_ > $maxparallel ? $maxparallel : scalar @_;
    
    my $i;
    for ($i=0; $i<$todo; $i++) {
	tdie "not [\&func, \@args]: $_[$i]" if ref($_[$i]) ne 'ARRAY';
	my $func = shift @{$_[$i]};
	tdie "not a function: $func" if ref($func) ne 'CODE';
	$cfs[$i] = Travertine::ChildFunc->new($func, $_[$i]); #, -thread => 1);
	$pids{$cfs[$i]->{pid}} = $i;
    }
    
    my @ret;

    my $pid;

    # XXX: There is a race condition here if two dudes are reaping
    # children at the same time. Assuming we don't use threads, that
    # shouldn't be possible, I think
    while ( ($pid = waitpid(-1, 0)) >= 0 ) {
	if (defined $pids{$pid}) {
	    # when one process completes...
	    $ret[$pids{$pid}] = $cfs[$pids{$pid}]->wait();
	    delete $pids{$pid};
	    # start off one more
	    if ($i < @_) {
		my $func = shift @{$_[$i]};
		tdie "not a function: $func" if ref($func) ne 'CODE';
		$cfs[$i] = Travertine::ChildFunc->new($func, $_[$i]);
		$pids{$cfs[$i]->{pid}} = $i;
		++$i;
	    }
	}
	last if (keys(%pids) == 0);
    }
    
    #for (my $i=0; $i<@cfs; $i++) {
    #	$ret[$i] = $cfs[$i]->wait();
    #}
    
    # restore old sigchld handler
    $SIG{CHLD} = $OLD_SIGCHLD;
    
    return @ret;
}

# Usage:
#    ParallelExec(@exec_array)
#
# Args:
#    @exec_array - an array of execution elements. Each execution element is
#    a tuple of [\&func, $arg1, $arg2, ..., $argN], which means run &func with
#    ($arg1, ..., $argN).
#
# Returns:
#    An array of references to the return values. E.G., the return value of
#    the first func is $result[0]. The functions MUST return scalars (or
#    rather, they will be called in scalar context).
#
# Notes:
#    This function forks a bunch of processes to run all the given functions
#    in parallel. Each child process is isolated from the others.
sub ParallelExec(@)
{
    my @cfs;

    # don't auto-reap
    #my $OLD_SIGCHLD = $SIG{CHLD};
    #$SIG{CHLD} = undef;
    
    for (my $i=0; $i<@_; $i++) {
	tdie "not [\&func, \@args]: $_[$i]" if ref($_[$i]) ne 'ARRAY';
	my $func = shift @{$_[$i]};
	tdie "not a function: $func" if ref($func) ne 'CODE';
	$cfs[$i] = Travertine::ChildFunc->new($func, $_[$i]); #, -thread => 1);
    }
    
    my @ret;
    
    for (my $i=0; $i<@cfs; $i++) {
	$ret[$i] = $cfs[$i]->wait();
    }
    
    # restore old sigchld handler
    #$SIG{CHLD} = $OLD_SIGCHLD;
    
    return @ret;
}

# Usage:
#    ParallelExec2(\&func, @args_array)
#
# Args:
#    \&func - the function to execute
#    @args_array - an array of argument array refs. E.G.,
#    ([$arg11,...], [$arg21,...], ...)
#
# Returns:
#    An array of references to the return values. E.G., the return value of
#    the first func is $result[0]. The functions MUST return scalars.
#
# Notes:
#    Same as ParallelExec, but run the same function on a set of argument 
#    lists.
sub ParallelExec2($@) {
    my $func = shift;
    my @args = @_;

    foreach my $args (@args) {
	if (ref($args) ne 'ARRAY') {
	    # assume only 1 arg per function
	    $args = [$func, $args];
	} else {
	    # multiple args per function
	    unshift @$args, $func;
	}
    }
    return ParallelExec(@args);
}

# Usage:
#    ParallelExec3($maxparallel, \&func, @args_array)
#
# Args:
#    $maxparallel - maximum parallel functions in flight
#    \&func - the function to execute
#    @args_array - arguments to each execution of the function (see above) 
#
# Returns:
#    An array of references to the return values of each execution.
#
# Notes:
#    This function keeps $maxparallel functions in flight at any given time.
sub ParallelExec3($$@)
{
    my $maxparallel = shift;
    my $func        = shift;
    my @args        = @_;

    foreach my $args (@args) {
	if (ref($args) ne 'ARRAY') {
	    # assume only 1 arg per function
	    $args = [$func, $args];
	} else {
	    # multiple args per function
	    unshift @$args, $func;
	}
    }
    return ParallelExec0($maxparallel, @args);
}

# Usage:
#    ParallelExec4($maxparallel, \&func, @\@args_array)
#
# Args:
#    $maxparallel - maximum parallel functions at one time
#    \&func - the function to execute
#    @\@args_array - arguments to each execution of the function (see above) 
#
# Returns:
#    An array of references to the return values of each execution.
#
# Notes:
#    This process waits until 1 batch has completed before moving onto
#    the next batch of $maxparallel functions. 
sub ParallelExec4($$@)
{
    my $maxparallel = shift;
    my $func        = shift;
    my @args        = @_;

    my @res;

    for (my $i=0; $i<@args; $i += $maxparallel) {
    	my $begin = $i;
	my $end   = $i+$maxparallel-1 > $#args ? 
	    $#args : $i+$maxparallel-1;

	push @res, ParallelExec2($func, @args[$begin..$end]);
    }

    return @res;
}

sub _MakeRemoteDumper($$$)
{
    my $ssh  = shift;
    tdie "not an sshTunnel" if ref($ssh) ne 'Travertine::SSHTunnel';

    my @vals = $_[0] ? @{shift()} : ();
    my @keys = $_[0] ? @{shift()} : ();

    my $node = $ssh->{host};
    push @keys, 'node';
    push @vals, \$node;

    # Create the perl script stub
    push @keys, map { "$_" } keys %REMOTEFUNC_GLOBAL_EXPORT;
    push @vals, (map { $REMOTEFUNC_GLOBAL_EXPORT{$_} } 
		 (keys %REMOTEFUNC_GLOBAL_EXPORT));
    
    my $d = Data::Dumper->new(\@vals, \@keys);
    $d->Deparse(1);

    return $d;
}

# Usage:
#    ExecRemoteFunc(sshTunnel, \&func, \@args [, %opts])
#
# Args:
#    sshTunnel - the sshTunnel to execute over
#    \&func - the function stub to run on the remote host
#    \@args - the arguments to pass to the function
#    %opts  - options:
#       -timeout    = X - if normal mode, timeout the command after X secs
#       -daemon     = 1 - run function in daemon mode (see RemoteDaemon*)
#       -background = 1 - execute in forked bg process and return.
#       -log        = string - output log if run as daemon or in background
#       -rotate     = X - rotate log every X lines (only with rb funcs)
#       -title      = title of screen session if in daemon mode
#       -print      = 1 - if in normal mode, also print the stdout/err output
#
# Returns:
#    - If in normal mode, this returns ($stdout, $stderr, $ret), where
#      $stdout is the output generated on stdout, $stderr is the output
#      generated on $stderr and $ret is the return value of function
#      interpreted as a value [0-255].
#    - If in daemon mode, the rdRef to the daemon started.
#    - If in background mode, rbRef to the background process
#    - undef on error.
#
# Notes:
#    This is the new version of 'rsystem'. Remember that you CAN NOT use
#    any global variables/references in the function that are not in
#    %REMOTEFUNC_GLOBAL_EXPORT. The additional globals that are added
#    are $node (the hostname as refered to by the sshTunnel) and
#    $func (the function you passed in).
sub ExecRemoteFunc($$$%)
{
    my $ssh  = shift;
    tdie "not an sshTunnel" if ref($ssh) ne 'Travertine::SSHTunnel';
    my $func = shift;
    tdie "not an \&func" if ref($func) ne 'CODE';

    my @args = @{shift()};
    my %opts = @_;

    my @dvals = ($func, \@args);
    my @dkeys = ('func', 'args');

    my $d     = _MakeRemoteDumper($ssh, \@dvals, \@dkeys);

    my $file  = "RemoteFunc." . (int(rand(0xFFFFFFFF))) . ".stub";
    my $lfile = "$LOCAL_STUB_DIR/$file";
    my $rfile = "$REMOTE_STUB_DIR/$file";

    my $perl = '';
    $perl .= "#\n";
    $perl .= "# host : $ssh->{host}\n";
    $perl .= "# time : " . scalar(localtime()) . "\n";
    $perl .= "# vers : $VERSION\n";
    $perl .= "#\n\n";

    $perl .= "require 5.6.1;\n";
    $perl .= "package Travertine;\n";
    foreach my $mod (@REMOTEFUNC_MODULE_EXPORT) {
	$perl .= "use $mod;\n";
    }

    my @keys = $d->Names();
    my @vals = $d->Values();
    for (my $i=0; $i<@keys; $i++) {
	my $key = $keys[$i];
	if (ref($vals[$i]) eq 'CODE') {
	    $perl .= "sub $key { \$$key\->(\@_) }\n";
	}
    }
    $perl .= $d->Dump;
    for (my $i=0; $i<@keys; $i++) {
	my $key = $keys[$i];
	if (ref($vals[$i]) eq 'CODE') {
	    next;
	} elsif (ref($vals[$i]) eq 'SCALAR') {
	    $perl .= "our \$$key = \$\$$key;\n";
	} elsif (ref($vals[$i]) eq 'ARRAY') {
	    $perl .= "our \@$key = \@\$$key;\n";
	} elsif (ref($vals[$i]) eq 'HASH') {
	    $perl .= "our \%$key = \%\$$key;\n";
	} elsif (ref($vals[$i]) eq 'REF') {
	    $perl .= "our \$$key = \$\$$key;\n";
	} else {
	    twarn "unknown reference type exported: $key=" . ref($vals[$i]);
	}
    }

    if (!$opts{-background}) {
	# run normally
	$perl .= 'select STDERR; $| = 1; select STDOUT; $| = 1;' . "\n";
	$perl .= "my \$value = &func(\@args);\n";
    } else {
	my $log = defined $opts{-log} ? $opts{-log} : '/dev/null';
	my $rotate = $opts{-rotate};

	if ($log eq 'tmp') { 
	    $perl .= 'my ($fh, $tmpfn) = tempfile(DIR => "/tmp");'. "\n\n";
	}

	# run as detached forked process
	$perl .= 'my $value;' . "\n";
	if ($log ne 'tmp') 
	{
	    $perl .= 'my $cf = Travertine::ChildFunc->new(\&func, [@args], ' .
		    '-detach => 1, -log => ' . "'$log'" . 
		    ($rotate?", -rotate => $rotate":"") . ");\n";
	}
	else {
	    $perl .= 'my $cf = Travertine::ChildFunc->new(\&func, [@args], ' . 
	            '-detach => 1, -log => $tmpfn);' . "\n";
	}

	$perl .= 'if (defined $cf) { ' . "\n";
	$perl .= '   print STDOUT $cf->serialize();' . "\n";
	$perl .= '   $value = 0; ' . "\n";
	$perl .= '} else { ' . "\n";
	$perl .= '   print STDERR "could not create childFunc: $!\n";' . "\n";
	$perl .= '   $value = 1; ' . "\n";
	$perl .= "}\n";
    }
    # remove myself before starting
    $perl .= "BEGIN { unlink '$rfile'; }\n";
    $perl .= "exit \$value;\n";

    if ($DEBUG_SAVE_STUBS) {
	# save a copy locally
	psystem("mkdir -p $LOCAL_STUB_DIR") if ! -d $LOCAL_STUB_DIR;
	open(STUB, ">$lfile") || twarn("can't create $lfile locally");
	print STUB $perl;
	close(STUB);
    }

    #my $gzip = Compress::Zlib::memGzip($perl);

    # Copy it over to the remote host
    my ($out, $err, $ret) = 
	$ssh->exec("zcat -f > $rfile", Compress::Zlib::memGzip($perl), $opts{-timeout});
    if ($ret) {
	twarn "copy of stub to remote host $ssh failed:\n" .
	    Travertine::thighlight($err);
	return undef;
    }
    
    # Execute it in a remote perl process
    if ($opts{-daemon}) {
	my %o;
	$o{-log} = $opts{-log} if $opts{-log};
	$o{-title} = $opts{-title} if $opts{-title};
	my $ref = Travertine::RemoteDaemon->new($ssh, "perl $rfile", %o);
	tdie("bad ref") if !$ref;
	return $ref;
    } else {
	($out, $err, $ret) = $ssh->exec("perl $rfile", undef, $opts{-timeout});

	if (!$opts{-background}) {
	    # if not executed in background
	    if ($opts{-print}) {
		print STDOUT $out;
		print STDERR $err;
	    }

	    return ($out, $err, $ret);
	} else {
	    if ($ret) {
		twarn "exec remote bg func $ssh failed:" .
		    Travertine::thighlight($err);
		return undef;
	    }

	    chomp $out;
	    if (!$out) {
		twarn "exec remote bg func $ssh gave no output: $err";
		return undef;
	    }

	    my $cf = Travertine::ChildFunc::deserialize($out);
	    return Travertine::RemoteBgFunc->new($out, $ssh, $cf->{log});
	}
    }
}

# Usage:
#     rexec(sshTunnel, \&func, \@args [, %opts])
#
# Notes:
#     Alias for ExecRemoteFunc.
#
sub rexec($$$%)
{
    return ExecRemoteFunc($_[0], $_[1], $_[2], @_[3..$#_]);
}

# Usage:
#     rsystem($user, $host, &func, @args)
#
# Notes:
#     Legacy function.
sub rsystem($$$@)
{
    my ($user, $host, $func, @args) = @_;

    my $ssh = Travertine::SSHTunnel->new($user, $host);
    if (!$ssh) {
	twarn "can't open sshTunnel to $user\@$host";
	return undef;
    }

    return (ExecRemoteFunc($ssh, $func, \@args, -print => 1, -timeout => 600))[2];
}

# Usage:
#     rsystem2($user, $host, $func, \@args, %opts)
# 
# Notes: 
#     Similar to rsystem, but with the following differences:
#       - \@args instead of @args
#       - func() can return arbitrary values as its return type 
#       - if ($opts {-immedop} == true) output is available immediately on STDERR
#       
#         XXX !BROKEN! we use 'tail' to read the remote log. depending on when
#         we start executing the 'tail' process we may miss some part of the
#         output. instead maybe we should write the log to a unix-domain
#         socket or pipe, and read from that instead.
#
sub rsystem2($$$$%) 
{
    my ($user, $host, $func, $pargs, %opts) = @_;
    my $ssh = Travertine::SSHTunnel->new($user, $host);
    if (!$ssh) { 
	twarn "can't open sshTunnel to $user\@$host";
	return undef;
    }

    $opts{-background} = 1;
    if ($opts{-immedop}) {
	$opts{-log} = "tmp";    # overwrite;
    }
    
    my $rbref = ExecRemoteFunc($ssh, $func, $pargs, %opts);
    my $pid;	

    if ($opts{-immedop}) {
	$pid = fork();
	if ($pid > 0) { 
            # tinfo "invoked child to get log info ";
	}
	else {
	    our $fh;
	    open $fh, "ssh -n -t -t -q $user\@$host tail -f $rbref->{log} | ";

	    sub routine { 
	    # fucking ssh wont kill the remote process even if 
	    # I sent a signal locally...  update: -t -t would do 
	    # it, but i am lazy to rewrite now.

	    # tinfo "killing remote tail process";

		my $kc = "ssh -n $user\@$host " . 'ps ax \| grep tail \| grep ' .  $rbref->{log};
		$kc .= ' \| grep -v grep \| awk \\\'\{ print \$1 \}\\\' \| xargs kill -9 ';
		
		psystem "$kc";
		close $fh;
		
                # be safe
		local $SIG{HUP} = 'IGNORE';
                # tinfo "trying to kill our child ssh process";
		kill HUP => -$$;
		exit;
	    }
	    local $SIG{HUP} = \&routine;
	    
	    while (<$fh>) {
		print STDERR;
	    }
	    close $fh;

	    exit; # should not ever come here
	}
    }
    my $retval = $rbref->wait();

    if ($opts{-immedop}) {
	sleep 1;
	kill 1, $pid; 
	waitpid $pid, 0;
    }
    
    return $retval;
}

# Usage:
#    SubToText($coderef, $name)
#
# Returns:
#    the textual representation of the code; utility routine
sub SubToText ($$) {
    my $sub = shift;
    my $name = shift;

    my $d = new Data::Dumper ([$sub], [$name]);
    $d->Deparse (1);
    
    my @lines = split (/\n/, $d->Dump ());
    pop @lines;
    return @lines;
}

# Usage:
#    CombineFuncs($funca, $funcb)
#
# Returns:
#    "concatenates" the subroutines funca, funcb and returns
#    the new subroutine which does funca(@_) and funcb(@_) ..
#    the individual subroutines can modify @_ all they want..
#
sub CombineFuncs($$) { 
    my ($fa, $fb) = @_;
    my ($hdra, @texta) = SubToText ($fa, "fa");
    my ($hdrb, @textb) = SubToText ($fb, "fb");

    my $ret = "\$__my_new_func = sub {\n";
    $ret .= "\tmy \@__store_args = \@_;\n";
    $ret .= "\t{\n";
    $ret .= join "\n", @texta;
    $ret .= "\t}\n";

    $ret .= "\n";
    $ret .= "\t\@_ = \@__store_args;\n";
    $ret .= "\t{\n";
    $ret .= join "\n", @textb;
    $ret .= "\t}\n";

    $ret .= "\n};\n";    

    my $val = eval { no strict; eval $ret; die $@ if ($@); $__my_new_func; };
    die "could not combine funcs ($@)" if $@;
    return $val;
}

# Usage:
#     ssh($user, $host [, %opts])
#
# Notes:
#     Alias for new Travertine::SSHTunnel.
sub ssh($$%)
{
    return Travertine::SSHTunnel->new(@_);
}

# Usage:
#     cfork(\&func, \@args [, %opts])
#
# Notes:
#     Alias for new Travertine::ChildFunc.
sub cfork($$%)
{
    return Travertine::ChildFunc->new(@_);
}

###############################################################################

=start
# Usage:
#     ParallelRsyncPull(\@sources, $target, $srcdir [, $maxparallel])
#
# Args:
#     \@sources - Where to pull from [ "user1\@host1", "user2\@host2", ... ].
#     $target   - Where to pull to "user\@host:/path/to/target"
#     $srcdir   - Directory on sources to pull.
#     $maxparallel - maximum rsyncs to run in parallel
sub ParallelRsyncPull($$$@) {
    my $source  = shift;
    my $target  = shift;
    my $dir     = shift;
    my $maxparallel = shift;
    $maxparallel = 10 if !$maxparallel;

    my @logins = @$target;

    for (my $i=0; $i<@logins; $i += $maxparallel) {
	my $begin = $i;
	my $end   = $i+$maxparallel-1 > $#logins ? 
	    $#logins : $i+$maxparallel-1;
	
	ParallelExec2(sub {
	    my $pref = shift;
	    $pref =~ s/:.*$//;
	    my ($user, $host) = split(/@/, $pref);
	
	    rsystem($user, $host, sub {
		my ($pref, $dir, $target) = @_;
		my $stat = psystem("rsync -e ssh -azb $dir/* $target 1>&2");

		if ($stat) {
		    twarn "rsync from $pref:$dir failed!";
		}
	    }, $pref, $dir, $target);

	    tinfo "rync to $pref complete";
	}, @logins[$begin..$end]);
    }
}

# Usage:
#     ParallelRsyncPush(\@sources, $target, $srcdir [, $maxparallel])
#
# Args:
#     $source   - Where to push from "user\@host:/path/to/source"
#     \@targets - Where to push to [ "user1\@host1", "user2\@host2", ... ].
#     $tgtdir   - Directory on targets to push to
#     $maxparallel - maximum rsyncs to run in parallel
sub ParallelRsyncPush($$$@) {
    my $source  = shift;
    my $target  = shift;
    my $dir     = shift;
    my $maxparallel = shift;
    $maxparallel = 10 if !$maxparallel;

    my @logins = @$source;

    my ($user, $host, $sdir) = ($source =~ /^(.*)\@(.*):(.*)$);

    rsystem($user, $host, sub {
	my $sdir = shift;
	my $dir  = shift;
	my $maxparallel = shift;
	my @logins = @_;
	
	for (my $i=0; $i<@logins; $i += $maxparallel) {
	    my $begin = $i;
	    my $end   = $i+$maxparallel-1 > $#logins ? 
		$#logins : $i+$maxparallel-1;

	    ParallelExec2(sub {
		my $pref = shift;
		$pref =~ s/:.*$//;

		my $stat = psystem("rsync -e ssh -azb $sdir/* $pref:$dir 1>&2");

		if ($stat) {
		    twarn "rsync to $pref:$dir failed!";
		}

		tinfo "rync to $pref complete";
	    });
	}
    }, $sdir, $dir, $maxparallel, @logins);
}
=cut

1
;
