#
# Mercury and Colyseus Software Distribution 
# 
# Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
#               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
#                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USA
#
###############################################################################
#
# File: TLite.pm 
# Version: $Id: TLite.pm 3112 2006-07-16 18:39:25Z ashu $
# 
# A lightweight utility version of Travertine. 
# 
# Error Handling
# ==============
#
# Undef usually means effor. Warning and fatal messages are printed out as
# follows (tinfo, twarn, tdie, respectively):
#
# [II] informational message
#
# [EE] warning message
#           stacktrace...
#
# [FF] fatal message
#           stacktrace...
#
# Messages in blue background is the output on STDERR on the remote host.
#
# Example: ChildFunc
# ==================
#
# Suppose we have the following declarations:
#
#   use TLite;
#
#   sub a { print "a @_\n"; sleep 10; return 1; }     # regular func
#   sub b { print "b @_\n"; sleep 10; return [3,4]; } # regular func
#   sub d { while (1) { print "d @_\n"; sleep 1; } }  # daemon func
#
# To run &a(1,2) in a new forked process, and then wait for its return value:
#
#   my $cf = TLite::ChildFunc->new(\&a, [1,2]);
#   ...
#   my $retval = $cf->wait(); # $ret == 1
# 
# To run &a(1,2) and &b(1,2) and wait for both to complete:
#
#   my @retvals = ParallelExec([\&a,1,2], [\&b,1,2]);
#
# If you pass the -detach flag when creating a new TLite::ChildFunc, the
# child process will detach from the parent and you can wait for it in another
# perl script even if the parent has exited:
#
#   my $cf = TLite::ChildFunc->new(\&a, [1,2], -detach => 1);
#   print SAVE $cf->serialize();
#   exit 0;
#
#   ...
#
#   my $cf = TLite::ChildFunc::deserialize(<SAVE>);
#   my $retval = $cf->wait();
#
#
# Environment Configuration Variables
# ===================================
#
# TRAVERTINE_LOGDIR          = default log dir on remote hosts
# TRAVERTINE_STUBDIR         = dir to copy perl stubs to on remote hosts
# TRAVERTINE_LOCALSTUBDIR    = local dir to save perl stubs if enabled
# TRAVERTINE_STATDIR         = dir to save remote process status
#
# TRAVERTINE_SHOWSTACKTRACE  = show stacktrace with warnings and errors if true
# TRAVERTINE_COLORSTACKTRACE = colorize stacktraces if true
# TRAVERTINE_PRINTCMDS       = print all shell or ssh commands executed
# TRAVERTINE_SAVESTUBS       = save stubs in LOCALSTUBDIR executed remotely
#
# TRAVERTINE_SSHTIMEOUT      = timeout on ssh connect (currently unused)
# TRAVERTINE_SSHTUNNELHOST   = host to tunnel all ssh commands through
# TRAVERTINE_SSHVERSION      = default ssh protocol version (e.g., '2,1')
#
###############################################################################

package TLite;
require 5.6.1;
require Exporter;

# standard libraries/pragmas
use strict;
#use Thread;
use Socket;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;
use Fcntl qw(F_GETFL F_SETFL);
use IO::Select;
use Carp;
use Carp qw(cluck);

our @ISA    = ("Exporter");
our @EXPORT = qw(psystem ppsystem pbacktick rerror famsymsg rsystem rsystem2 tinfo twarn tdie thighlight
		 ParallelExec ExecRemoteFunc
		 ParallelExec0 ParallelExec2 ParallelExec3 ParallelExec4,
		 rexec ssh cfork);
our @CARP_NOT = qw(tinfo twarn tdie);

###############################################################################
# Environment Config:

# default directory where output logs are created (on local as well as remote)
# this directory must *already* exist!
our $LOG_DEFAULT_DIR = defined $ENV{TRAVERTINE_LOGDIR} ? 
    $ENV{TRAVERTINE_LOGDIR} : "/tmp";

# where remote perl script stubs are copied to before executed
# this directory must *already* exist!
our $REMOTE_STUB_DIR = defined $ENV{TRAVERTINE_STUBDIR} ? 
    $ENV{TRAVERTINE_STUBDIR} : "/tmp";

# where remote perl script stubs are saved locally (if debug enabled)
# make sure this is *different* from $REMOTE_STUB_DIR, otherwise behavior
# when executing stuff on localhost will be screwed up
# this directory will be created if it doesn't exist
our $LOCAL_STUB_DIR = defined $ENV{TRAVERTINE_LOCALSTUBDIR} ? 
    $ENV{TRAVERTINE_LOCALSTUBDIR} : "/tmp/stubs";

# where the return/exit status of exiting remote processes is written to
# this directory must *already* exist!
our $REMOTE_STAT_DIR = defined $ENV{TRAVERTINE_STATDIR} ? 
    $ENV{TRAVERTINE_STATDIR} : "/tmp";

# if true, show a stack trace with warnings and fatal errors
our $DEBUG_SHOW_STACKTRACE = defined $ENV{TRAVERTINE_SHOWSTACKTRACE} ?
    $ENV{TRAVERTINE_SHOWSTACKTRACE} : 0;

# if true, colorize the strack traces to make them a bit easier to read
our $DEBUG_COLOR_STACKTRACE = defined $ENV{TRAVERTINE_COLORSTACKTRACE} ?
    $ENV{TRAVERTINE_COLORSTACKTRACE} : 1;

# if true, then print all shell commands executed locally or via ssh-tunnel
our $DEBUG_PRINT_CMDS = defined $ENV{TRAVERTINE_PRINTCMDS} ? 
    $ENV{TRAVERTINE_PRINTCMDS} : 0;

# if true, then save a copy of all perl script stubs in LOCAL_STUB_DIR
# (never saved in remote REMOTE_STUB_DIR)
our $DEBUG_SAVE_STUBS = defined $ENV{TRAVERTINE_SAVESTUBS} ? 
    $ENV{TRAVERTINE_SAVESTUBS} : 0;

# connection timeout for initiated ssh connections (in sec)
our $SSH_CONNECT_TIMEOUT = defined $ENV{TRAVERTINE_SSHTIMEOUT} ?
    $ENV{TRAVERTINE_SSHTIMEOUT} : 10;

# tunnel all ssh connections through this machine (i.e., instead of
# 'ssh host cmd' do 'ssh tunnel ssh host cmd'. None if undef.
our $SSH_TUNNEL_HOST = defined $ENV{TRAVERTINE_SSHTUNNELHOST} ?
    $ENV{TRAVERTINE_SSHTUNNELHOST} : undef;

# default ssh protocol version
our $SSH_PROTO_VERSION = defined $ENV{TRAVERTINE_SSHVERSION} ?
    $ENV{TRAVERTINE_SSHVERSION} : '2,1'; # planetlab requires 2,1

###############################################################################
# Options:

our $VERSION = '$Id: TLite.pm 3112 2006-07-16 18:39:25Z ashu $';

# Ssh options passed to Net::SSH::Perl
#our %SSH_DEFAULT_OPTS = ( 'protocol' => '1,2',
#			  'options'  => [ "StrictHostKeyChecking no",
#					  "ForwardX11 no",
#					  "ForwardAgent yes" ] );
our @SSH_DEFAULT_OPTS = ( "-oProtocol=$SSH_PROTO_VERSION", 
			  '-oStrictHostKeyChecking=no',
			  '-oForwardX11=no',
			  '-oForwardAgent=yes' );

# Modules available for remote functions
our @REMOTEFUNC_MODULE_EXPORT = ( #"Thread",
				  "Socket",
				  "IO::File",
				  "Data::Dumper",
				  "Errno qw(EAGAIN ESRCH EPERM)",
				  "POSIX qw(:sys_wait_h)",
				  "POSIX qw(strftime)",
				  "File::Temp qw/tempfile tempdir/",
				  "Carp",
				  "Carp qw(cluck)" );
				  
# Functions exported and available in ExecRemoteFunc. NOTE: Ensure that any
# function in this list only uses modules that are declared above!
# Do not put functions in here if they depend on non-standard modules.
# Each key should be the name of the desired variable in the remote script
# Each val should be a reference to the desired value the variable is
# initialized to (e.g., $foo = '10' should be ( 'foo' => \$foo ) );
our %REMOTEFUNC_GLOBAL_EXPORT = ( 'LOG_DEFAULT_DIR'             => 
				  \$LOG_DEFAULT_DIR,
				  'REMOTE_STAT_DIR'             => 
				  \$REMOTE_STAT_DIR,
				  'REMOTE_STUB_DIR'             => 
				  \$REMOTE_STUB_DIR,
				  'LOCAL_STUB_DIR'              =>
				  \$LOCAL_STUB_DIR,
				  'DEBUG_COLOR_STACKTRACE'     => 
				  \$DEBUG_COLOR_STACKTRACE,
				  'DEBUG_PRINT_CMDS'            => 
				  \$DEBUG_PRINT_CMDS,
				  'TLite::fancymsg'        => 
				  \&TLite::fancymsg,
				  'TLite::rerror'          => 
				  \&TLite::rerror,
				  'TLite::psystem'         => 
				  \&TLite::psystem,
				  'TLite::ppsystem'         => 
				  \&TLite::ppsystem,
				  'TLite::pbacktick'         => 
				  \&TLite::pbacktick,
				  'TLite::tcolorize'       => 
				  \&TLite::tcolorize,
				  'TLite::thighlight'       => 
				  \&TLite::thighlight,
				  'TLite::tinfo'           => 
				  \&TLite::tinfo,
				  'TLite::twarn'           => 
				  \&TLite::twarn,
				  'TLite::tdie'            => 
				  \&TLite::tdie,
				  'TLite::encode_str'      => 
				  \&TLite::encode_str,
				  'TLite::decode_str'      => 
				  \&TLite::decode_str,
				  'TLite::ssh'             => 
				  \&TLite::ssh,
				  'TLite::cfork'           => 
				  \&TLite::cfork,
				  'TLite::ChildFunc::new'  => 
				  \&TLite::ChildFunc::new,
				  'TLite::ChildFunc::isAlive' => 
				  \&TLite::ChildFunc::isAlive,
				  'TLite::ChildFunc::stop' => 
				  \&TLite::ChildFunc::stop,
				  'TLite::ChildFunc::wait' => 
				  \&TLite::ChildFunc::wait,
				  'TLite::ChildFunc::serialize'   => 
				  \&TLite::ChildFunc::serialize,
				  'TLite::ChildFunc::deserialize' => 
				  \&TLite::ChildFunc::deserialize,
				  'TLite::ParallelExec'    => 
				  \&TLite::ParallelExec,
				  'TLite::ParallelExec0'   => 
				  \&TLite::ParallelExec0,
				  'TLite::ParallelExec2'   => 
				  \&TLite::ParallelExec2,
				  'TLite::ParallelExec3'   => 
				  \&TLite::ParallelExec3,
				  'TLite::ParallelExec4'   => 
				  \&TLite::ParallelExec4);

###############################################################################
# Data structures:

# A function that is executed as a local child process. We can later wait on
# this process and obtain the function's return value. You may specify that
# the function be "detached" in which case you can even wait on the function
# later after the parent process has exited.
#
# childFunc = { 
#     pid  => pid of forked child process
#     fd   => readable/writable end of pipe (only if non-detached)
#     stat => status file where result is written (only if detached)
#     time => time the proc was started (unix time)
# }

###############################################################################
# General functions:

# Usage:
#    psystem($cmd)
#
# Notes:
#    Run system() on a command and possibly echo it to the screen.
sub psystem {
    tinfo("$_[0]") if $DEBUG_PRINT_CMDS;
    return system($_[0]);
};

# Usage:
#    ppsystem($cmd)
#
# Notes:
#    Run system() on a command and _always_ echo it to the screen.
sub ppsystem {
    tinfo ("$_[0]");
    return system ($_[0]);
}

# Usage:
#    ptick($cmd)
#
# Notes:
#    Run `command` (possibly echoing on screen) and return the
#    return-value.
sub pbacktick {
    tinfo("$_[0]") if $DEBUG_PRINT_CMDS;
    return `$_[0]`;
}

# deprecated!
sub rerror {
    my $msg = shift;
    our $node;

    print STDERR "\n[31mERROR @ [35m$node[m: $msg\n";
    sleep(500);
    exit 4;
}

# deprecated!
sub fancymsg {
    my $msg = shift;
    our $node;

    print STDERR "\n[34mMSG @ [35m$node[m: $msg\n";
    exit 4;
}

# colorize stack trace
sub tcolorize($)
{
    my @lines = split(/(\n)/, shift());

    my $skip = 0;

    foreach my $line (@lines) {
	# don't over color parts that are already colored
	$skip = 1 if ($line =~ /\[\d*(;\d+)?m/);
	$skip = 0 if ($line =~ /\[m/);
	next if $skip;

	# XXX: This might gobble up some lines we don't want it to...
	if ($line =~ /^\t([\w_:]+)\((.*)\) called at (.*) line (\d+)$/) {
	    my ($func, $args, $file, $lineno) = ($1, $2, $3, $4);
	    $line = "\t[36m$func[m($args) called at $file line $lineno";
	}
    }

    return join('', @lines);
}

# colorize string with background
sub thighlight($@)
{
    my ($str, $fgcolor, $bgcolor) = @_;
    $bgcolor = 44 if !defined $bgcolor; # blue
    $fgcolor = 36 if !defined $fgcolor; # white

    my @lines = split(/(\n)/, $str);
    map { $_ =~ s/(\[\d+(;\d+)?m)/\[m$1/g; 
	  $_ =~ s/(\[m)/$1\[${fgcolor};${bgcolor}m/g; 
	  $_} @lines;
    unshift @lines, "[${fgcolor};${bgcolor}m";
    if ($lines[$#lines] =~ /(.*)\n$/) {
	$lines[$#lines] = "$1[m\n";
    } else {
	push @lines, "[m";
    }
    return join('', @lines);
}

# Usage:
#    tinfo(@msgs)
#
# Notes:
#    Display an informational message.
sub tinfo(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "\@[35m$node[m ";
    }
    
    print STDERR ("[30;42m[II][m $host", @_, "\n");
}

# Usage:
#    twarn(@msgs)
#
# Notes:
#    Display an warning message and stacktrace.
sub twarn(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "[35m$node[m ";
    }
    if ($DEBUG_SHOW_STACKTRACE) {
	my $msg = Carp::longmess("[30;41m[EE][m $host", @_);
	$msg = tcolorize $msg if $DEBUG_COLOR_STACKTRACE;
	print STDERR $msg;
    } else {
	print STDERR ("[30;41m[EE][m $host", @_, "\n");
    }
}

# Usage:
#    tdie(@msgs)
#
# Notes:
#    Display an fatal message and stacktrace.
sub tdie(@)
{
    our $node;
    my $host;
    if (!$node) {
	$host = "";
    } else {
	$host = "[35m$node[m ";
    }

    if ($DEBUG_SHOW_STACKTRACE) {
	my $msg = Carp::longmess("[30;41m[FF][m $host", @_);
	$msg = tcolorize $msg if $DEBUG_COLOR_STACKTRACE;
	print STDERR $msg;
    } else {
	print STDERR ("[30;41m[FF][m $host", @_, "\n");
    }
    exit 255;
}

sub encode_str {
    my $str = shift;

    $str =~ s/(.)/ sprintf "%02X", unpack("C", $1) /eg;
    return $str;
}

sub decode_str {
    my $str = shift;

    $str =~ s/(..)/ pack("C", hex $1) /eg;
    return $str;
}

###############################################################################
# class ChildFunc

package TLite::ChildFunc;
#use Thread;
use Socket;
use Errno qw(EAGAIN ESRCH EPERM);
use IO::File;

use overload q("") => sub { 
    my $self = shift;
    #my $thread = ($self->{thread} ? 1 : 0);
    my $detached = ($self->{stat} ? 1 : 0);
    #if ($thread) {
    #	return "(ChildFunc thread=$self->{thread})";
    #}
    return "(ChildFunc pid=$self->{pid} detached=$detached" .
	($detached?" stat=$self->{stat}":"") . " time=\"" . 
	scalar(localtime($self->{time})) . "\")";
};

# Usage:
#    TLite::ChildFunc->new(\&func, \@args [, %opts])
#
# Args:
#    \&func - function to execute
#    \@args - arguments to execute with
#    %opts  - options:
#       -thread = 1. run in a thread instead of a process. this allows you to
#                 return non-trivial datastructures like filehandles and 
#                 sockets, but you can't detach from this. DISABLED!
#       -detach = 1. don't open a pipe, but write the result to REMOTE_STAT_DIR
#                 and don't die on SIGHUP. A detached childFunc will keep
#                 running after it has parent has exited and you can still
#                 use WaitChildFunc on it later.
#       -log    = string. If detached, then write stdout/stderr to this file.
#                 (otherwise they will be closed for detached childFuncs)
#       -rotate = int. rotate the output log every x lines.
#
# Returns:
#    childFunc on success, undef on fork() failure;
#
# Notes:
#    Run a function in a child process. Returns immediately; use
#    WaitChildFunc() to "join" the child process and retrieve the return
#    value. Assumes the function is called in scalar context.
sub new($$$%) {
    my $cls  = shift;
    my $func = shift;
    TLite::tdie "not a \&func" if ref($func) ne 'CODE';

    my @args = @{shift()};
    my %opts = @_;

#    if ($opts{-thread}) {
#	my $thread = Thread->new($func, @args);
#
#	my $ret = {
#	    'pid'    => undef,
#	    'fd'     => undef,
#	    'stat'   => undef,
#	    'thread' => $thread,
#	    'time'   => time(),
#	};
#	return bless $ret, "TLite::ChildFunc";
#    }
    
    my $retries = 5;

    my ($childfh, $parentfh);
    if (!$opts{-detach}) {
	socketpair($childfh, $parentfh, AF_UNIX, SOCK_STREAM, PF_UNSPEC);
	TLite::tdie "can't open pipes" if !$childfh || !$parentfh;
    }

    my $out;
    if ($opts{-detach} && defined $opts{-log}) {
	# make sure we can create the output log
	my $log = $opts{-log};
	if ($opts{-rotate}) {
	    TLite::twarn "bad -rotate option $opts{-rotate}" if 
		!($opts{-rotate} > 0);
	      my $dumper = Data::Dumper->new([\&TLite::ChildFunc::logrotate], ['logrotate']);
	      $dumper->Indent(0);
	      my $prog = $dumper->Dump();
	      open(L, ">$REMOTE_STUB_DIR/lr.$$.pl") or
		  TLite::twarn "couldn't create $REMOTE_STUB_DIR/lr.$$.pl script: $!" && return undef;
	      print L "use IO::File;\n";
	      print L "unlink '$REMOTE_STUB_DIR/lr.$$.pl';\n";
	      print L <<'EOT';
sub logrotate {
    my ($LOGNAME, $MAXLINES) = @_;
    my $lines = 0;
    my $start_time = time();
    my $end_time;
    my $log = new IO::File(">$LOGNAME");
    autoflush $log 1;
    die "can't create $LOGNAME" if !$log;
    close STDOUT;
    close STDERR;
    my $pid = fork();
    if ($pid > 0) {
	exit 0;
    } elsif (!defined $pid) {
	die "fork failed!";
    }
    # child
    $SIG{HUP} = 'IGNORE';
    while(<STDIN>) {
	++$lines;
	print $log $_;
	
	if ($lines >= $MAXLINES) {
	    close($log);
	    $end_time = time();
	    
	    system("mv -f $LOGNAME $LOGNAME.$start_time-$end_time");
	    
	    $start_time = time();
	    $log = new IO::File(">$LOGNAME");
	    autoflush $log 1;
	    die "can't create $LOGNAME" if !$log;
	    $lines = 0;
	}
    }
}
EOT
	      print L "logrotate('$log', $opts{-rotate});\n";
	      close(L);
	      $out = IO::File->new("| perl $REMOTE_STUB_DIR/lr.$$.pl");
	  } else {
	      $out = IO::File->new(">$log");
	  }
	if (!$out) {
	    TLite::twarn "couldn't create childFunc output log $log: $!";
	    return undef;
	}
    }

    while ($retries-- > 0) {
	my $pid = fork();
	
	if ($pid > 0) {
	    # parent
	    close($parentfh) if !$opts{-detach};
	    my $ret = {
		'pid'  => $pid,
		'fd'   => !$opts{-detach} ? $childfh : undef,
		'stat' => ($opts{-detach} ? 
			   "$REMOTE_STAT_DIR/ChildFunc.$pid.stat" : undef),
		'time' => time(),
		'log'  => $opts{-log},
	    };
	    return bless $ret, "TLite::ChildFunc";
	} elsif (defined $pid) {
	    # child
	    if (!$opts{-detach}) {
		close($childfh);
	    } else {
		$SIG{HUP} = 'IGNORE';
		close STDIN;
		if ($out) {
		    STDOUT->fdopen($out, 'w');
		    STDERR->fdopen($out, 'w');
		} else {
		    # if we don't redirect, then close them
		    close(STDOUT);
		    close(STDERR);
		}

		$parentfh = 
		    IO::File->new(">$REMOTE_STAT_DIR/ChildFunc.$$.stat");
		TLite::tdie "can't create stat file for $$: $!" 
		    if !$parentfh;
	    }
	    my $ret = (&{$func}(@args));
	
	    my $dumper = Data::Dumper->new([$ret], ['datum']);
	    #$dumper->Terse(1);
	    $dumper->Purity(1);
	    $dumper->Indent(0);
	    my $text = $dumper->Dump();
	    
	    print $parentfh ( $text );
	    close($parentfh);
	    exit 0;
	} elsif ($! == EAGAIN) {
	    sleep 5;
	    next;
	} else {
	    # bad fork error
	    TLite::twarn "fork failed: $!";
	    return undef;
	}
    }
}

# Usage:
#     childFunc->serialize()
#
# Returns:
#     String representation of a detached childFunc.
sub serialize($)
{
    my $cf = shift;
    TLite::tdie "not a childFunc" if ref($cf) ne 'TLite::ChildFunc';
    #TLite::tdie "can't serialize a thread" if $cf->{thread};

    if ($cf->{fd} || !$cf->{stat}) {
	TLite::tdie "can't seriliaze non-detached childFunc!";
    }

    return "$cf->{pid}==$cf->{time}==$cf->{log}";
}

# Usage:
#    TLite::ChildFunc::deserialize($string)
#
# Returns:
#    The detached childFunc the string represents.
sub deserialize($)
{
    my $str = shift;
    if ($str !~ /^\d+==\d+==/) {
	TLite::twarn "invalid string";
	  return undef;
      }
    my ($pid, $time, $log) = split(/==/, $str);

    my $cf = {
	'pid'  => $pid,
	'fd'   => undef,
	'stat' => "$REMOTE_STAT_DIR/ChildFunc.$str.stat",
	'time' => $time,
	'log'  => $log,
    };
    return bless $cf, "TLite::ChildFunc";
}

# Usage:
#     childFunc->isAlive()
#
# Returns:
#     1 if yes or unknown, 0 if no
sub isAlive($)
{
    my $ref = shift;
    TLite::tdie "not a childFUnc" if ref($ref) ne 'TLite::ChildFunc';
    if ($ref->{pid} < 0) {
	return 0;
    }
    
    my $ret;
    if (kill 0, $ref->{pid}) {
	$ret = 1;
    } elsif ($! == EPERM) {
	$ret = 2;
    } elsif ($! == ESRCH) {
	$ret = 0;
    } else {
	$ret = 3;
    }

    if ($ret == 2) {
	TLite::twarn "don't have permission to signal $ref->{pid}";
    } elsif ($ret == 3) {
	TLite::twarn "error trying to signal $ref->{pid}";
    }
    
    return $ret;
}

# Usage:
#     childFunc->stop([signal])
#
# Returns:
#     1 on successful signal, undef otherwise
sub stop($@)
{
    my $cf = shift;
    TLite::tdie "not a childFunc" if ref($cf) ne 'TLite::ChildFunc';
    #TLite::tdie "can't stop a thread" if $cf->{thread};

    my $sig = shift;
    $sig = 15 if !defined $sig;

    my $ret;

    if ( (kill $sig, $cf->{pid}) > 0 ) {
	$ret = 1;
    } elsif ($! == EPERM) {
	TLite::twarn "don't have permission to signal $cf->{pid}";
	$ret = undef;
    } elsif ($! == ESRCH) {
	# ok -- not found, means it is dead
	$ret = 1;
    } else {
	TLite::twarn "unknown error signalling $cf->{pid}";
	$ret = undef;
    }

    if (defined $cf->{fd}) {
	close $cf->{fd};
    } elsif (defined $cf->{stat}) {
	unlink $cf->{stat};
    }

    return $ret;
}

# Usage:
#   childFunc->wait()
#
# Args:
#    childFunc - reference to the childFunc created by CreateChildFunc
#
# Returns:
#    return value of the function executed in childFunc. OPEN FILEHANDLES
#    AND SOCKETS WILL NOT BE RETURNED CORRECTLY!!!
#
# Notes:
sub wait($) {
    my $cf = shift;
    TLite::tdie "not a childFunc" if ref($cf) ne 'TLite::ChildFunc';

    my $retval;

#    if (defined $cf->{thread}) {
#	return $cf->{thread}->join();
#    } elsif 
    if (defined $cf->{fd}) {
	# non-detached process...
	my $pid = waitpid $cf->{pid}, 0;
	if ($pid != $cf->{pid} && $pid != -1) {
	    # possibly already waited for it, so let -1 be ok...
	    TLite::tdie 
		"waitpid failed! returned $pid, but expected pid $cf->{pid}";
	}
    
	my $fh = $cf->{fd};
	while(<$fh>) {
	    $retval .= $_;
	}

	close($fh);
    } elsif (defined $cf->{stat}) {
	# detached process...
	# the target might be a child, in which case, try to reap it
	waitpid $cf->{pid}, 0;
	while (kill 0, $cf->{pid}) {
	    # child is still alive
	    sleep 1;
	}
	if ($! == EPERM) {
	    TLite::twarn "don't have permission to signal $cf->{pid}";
	    return undef;
	} elsif ($! == ESRCH) {
	    # looks like pid is gone, read stat
	    my $fh = 
		IO::File->new("<$REMOTE_STAT_DIR/ChildFunc.$cf->{pid}.stat");
	    if (!$fh) {
		TLite::twarn "no status file for $cf->{pid}";
		return undef;
	    }
	    while (<$fh>) {
		$retval .= $_;
	    }
	    close $fh;
	    unlink $cf->{stat};
	}  else {
	    TLite::twarn "unknown error signalling $cf->{pid}";
	    return undef;
	}

    } else {
	# errorenous process!
	TLite::tdie "child func ref doesn't have fh or stat file!";
    }

    my $val = eval { no strict; eval $retval; die $@ if ($@); $datum; };
    #TLite::tinfo " >> $retval";
    #TLite::tinfo " >> $val";
    if ($@) {
	TLite::twarn 
	    "failed to eval serialized return value ($retval): $@"
    }

    return $val;
}

###############################################################################
# General Functions

package TLite;
use POSIX ":sys_wait_h";

sub ParallelExec0($@)
{
    my $maxparallel = shift @_;
    my @cfs;

    # don't auto-reap
    my $OLD_SIGCHLD = $SIG{CHLD};
    $SIG{CHLD} = undef;

    my %pids;

    my $todo = scalar @_ > $maxparallel ? $maxparallel : scalar @_;
    
    my $i;
    for ($i=0; $i<$todo; $i++) {
	tdie "not [\&func, \@args]: $_[$i]" if ref($_[$i]) ne 'ARRAY';
	my $func = shift @{$_[$i]};
	tdie "not a function: $func" if ref($func) ne 'CODE';
	$cfs[$i] = TLite::ChildFunc->new($func, $_[$i]); #, -thread => 1);
	$pids{$cfs[$i]->{pid}} = $i;
    }
    
    my @ret;

    my $pid;

    # XXX: There is a race condition here if two dudes are reaping
    # children at the same time. Assuming we don't use threads, that
    # shouldn't be possible, I think
    while ( ($pid = waitpid(-1, 0)) >= 0 ) {
	if (defined $pids{$pid}) {
	    # when one process completes...
	    $ret[$pids{$pid}] = $cfs[$pids{$pid}]->wait();
	    delete $pids{$pid};
	    # start off one more
	    if ($i < @_) {
		my $func = shift @{$_[$i]};
		tdie "not a function: $func" if ref($func) ne 'CODE';
		$cfs[$i] = TLite::ChildFunc->new($func, $_[$i]);
		$pids{$cfs[$i]->{pid}} = $i;
		++$i;
	    }
	}
	last if (keys(%pids) == 0);
    }
    
    #for (my $i=0; $i<@cfs; $i++) {
    #	$ret[$i] = $cfs[$i]->wait();
    #}
    
    # restore old sigchld handler
    $SIG{CHLD} = $OLD_SIGCHLD;
    
    return @ret;
}

# Usage:
#    ParallelExec(@exec_array)
#
# Args:
#    @exec_array - an array of execution elements. Each execution element is
#    a tuple of [\&func, $arg1, $arg2, ..., $argN], which means run &func with
#    ($arg1, ..., $argN).
#
# Returns:
#    An array of references to the return values. E.G., the return value of
#    the first func is $result[0]. The functions MUST return scalars (or
#    rather, they will be called in scalar context).
#
# Notes:
#    This function forks a bunch of processes to run all the given functions
#    in parallel. Each child process is isolated from the others.
sub ParallelExec(@)
{
    my @cfs;

    # don't auto-reap
    #my $OLD_SIGCHLD = $SIG{CHLD};
    #$SIG{CHLD} = undef;
    
    for (my $i=0; $i<@_; $i++) {
	tdie "not [\&func, \@args]: $_[$i]" if ref($_[$i]) ne 'ARRAY';
	my $func = shift @{$_[$i]};
	tdie "not a function: $func" if ref($func) ne 'CODE';
	$cfs[$i] = TLite::ChildFunc->new($func, $_[$i]); #, -thread => 1);
    }
    
    my @ret;
    
    for (my $i=0; $i<@cfs; $i++) {
	$ret[$i] = $cfs[$i]->wait();
    }
    
    # restore old sigchld handler
    #$SIG{CHLD} = $OLD_SIGCHLD;
    
    return @ret;
}

# Usage:
#    ParallelExec2(\&func, @args_array)
#
# Args:
#    \&func - the function to execute
#    @args_array - an array of argument array refs. E.G.,
#    ([$arg11,...], [$arg21,...], ...)
#
# Returns:
#    An array of references to the return values. E.G., the return value of
#    the first func is $result[0]. The functions MUST return scalars.
#
# Notes:
#    Same as ParallelExec, but run the same function on a set of argument 
#    lists.
sub ParallelExec2($@) {
    my $func = shift;
    my @args = @_;

    foreach my $args (@args) {
	if (ref($args) ne 'ARRAY') {
	    # assume only 1 arg per function
	    $args = [$func, $args];
	} else {
	    # multiple args per function
	    unshift @$args, $func;
	}
    }
    return ParallelExec(@args);
}

# Usage:
#    ParallelExec3($maxparallel, \&func, @args_array)
#
# Args:
#    $maxparallel - maximum parallel functions in flight
#    \&func - the function to execute
#    @args_array - arguments to each execution of the function (see above) 
#
# Returns:
#    An array of references to the return values of each execution.
#
# Notes:
#    This function keeps $maxparallel functions in flight at any given time.
sub ParallelExec3($$@)
{
    my $maxparallel = shift;
    my $func        = shift;
    my @args        = @_;

    foreach my $args (@args) {
	if (ref($args) ne 'ARRAY') {
	    # assume only 1 arg per function
	    $args = [$func, $args];
	} else {
	    # multiple args per function
	    unshift @$args, $func;
	}
    }
    return ParallelExec0($maxparallel, @args);
}

# Usage:
#    ParallelExec4($maxparallel, \&func, @\@args_array)
#
# Args:
#    $maxparallel - maximum parallel functions at one time
#    \&func - the function to execute
#    @\@args_array - arguments to each execution of the function (see above) 
#
# Returns:
#    An array of references to the return values of each execution.
#
# Notes:
#    This process waits until 1 batch has completed before moving onto
#    the next batch of $maxparallel functions. 
sub ParallelExec4($$@)
{
    my $maxparallel = shift;
    my $func        = shift;
    my @args        = @_;

    my @res;

    for (my $i=0; $i<@args; $i += $maxparallel) {
    	my $begin = $i;
	my $end   = $i+$maxparallel-1 > $#args ? 
	    $#args : $i+$maxparallel-1;

	push @res, ParallelExec2($func, @args[$begin..$end]);
    }

    return @res;
}

# Usage:
#     cfork(\&func, \@args [, %opts])
#
# Notes:
#     Alias for new TLite::ChildFunc.
sub cfork($$%)
{
    return TLite::ChildFunc->new(@_);
}

1
;
