#!/usr/bin/perl
#
# Check the aliveness of daemons in an experiment
#

use strict;
use Travertine;
use Getopt::Std;
use IO::File;
use vars qw($opt_l $opt_t $opt_h $opt_L $opt_d);

our $MAXPARALLEL = 10;

getopts("l:t:L:dh");

if (!$ARGV[0] || $opt_h) {
    print STDERR "usage: Status.pl [options] <exp_file>\n\n";
    print STDERR "       -l rex   only hosts with login matching regexp\n";
    print STDERR "       -t rex   only hosts with title matching regexp\n";
    print STDERR "       -L num   also tail -num lines from the log\n";
    print STDERR "       -d       remote daemons instead of background processes\n";
    print STDERR "       -h       this help message\n";
    exit 0;
}

our $INPUT = *STDIN;
our $LOGIN_REX = defined($opt_l) ? $opt_l : undef;
our $TITLE_REX = defined($opt_t) ? $opt_t : undef;
our $TAILLOG = $opt_L;
our $BGFUNC = !defined $opt_d;

if ($ARGV[0] ne '-') {
    $INPUT = new IO::File("<$ARGV[0]");
    tdie "can't open $ARGV[0]: $!" if !$INPUT;
}

our @triples;

while (<$INPUT>) {
    chomp $_;
    my ($login, $title, $ser) = ($_ =~ /^([^\t]+)\t'([^\t]+)'\t([^\t]+)$/);
    tdie "bad line: $_" if !defined $login or !defined $title or !$ser;

    next if ($LOGIN_REX && $login !~ $LOGIN_REX);
    next if ($TITLE_REX && $title !~ $TITLE_REX);

    my $ref;
    
    if (!$BGFUNC) {
	$ref = Travertine::RemoteDaemon::deserialize($ser);
    } else {
	$ref = Travertine::RemoteBgFunc::deserialize($ser);
    }

    tdie "can't deserilize ref on line: $_" if !defined $ref;

    push @triples, [ $login, $title, $ref ];
}

if (@triples == 0) {
    tdie "no matching logins/titles";
}

our @ret = ParallelExec3($MAXPARALLEL, sub {
    my ($login, $title, $ref) = @_;
    return $ref->isAlive();
}, @triples);

for (my $i=0; $i<@ret; $i++) {
    if (!$ret[$i]) {
	twarn "[ $triples[$i]->[1] ] is dead!";
    } else {
	tinfo "[ $triples[$i]->[1] ] is alive!";
    }
    if ($TAILLOG) {
	print STDERR thighlight($triples[$i]->[2]->tailLog($TAILLOG));
	print STDERR "\n";
    }
}
