#!/usr/bin/perl

use strict;
use Travertine;
use Getopt::Std;
use vars qw($opt_e $opt_p $opt_P $opt_m);

getopts ("e:p:P:m:");

our $COLY_HOME  = "$ENV{HOME}/Colyseus";
our $EXP_FILE   = defined $opt_e ? $opt_e : "";  
our $PASSWORD   = defined $opt_P ? $opt_P : "f7Ughi6l"; #undef;
our $MAXPLAYERS = defined $opt_m ? $opt_m : 4;
our $PORT       = defined $opt_p ? $opt_p : 52153; #30000;

###############################################################################

sub mdie;

my $fs = "font-size: 12px;";

our $th = "td style=\"font-weight: bold; color: white; padding: 3px 2px; background-color: #5555DD\"";
our $td = "td style=\"background-color: #b5edbc\"";

our $EXP_FILE_BLURB = undef;
if ($EXP_FILE eq '-') {
    while (<STDIN>) { $EXP_FILE_BLURB .= $_; }
}
else {    
    mdie "no exp file: $EXP_FILE" if ! -f $EXP_FILE;
    open F, "< $EXP_FILE";
    while (<F>) { $EXP_FILE_BLURB .= $_; }
    close F;
}

our $TmpExpFile = "/tmp/pss.$$.exp";
open G, "> $TmpExpFile";
print G $EXP_FILE_BLURB;
close G;

$ENV{PERL5LIB} .= ":$COLY_HOME/run:$COLY_HOME/run/lib:/usr/local/lib";

sub GetTermExecOutput { 
    my $tcmd = shift;
    my $termExec = "$COLY_HOME/run/TermExec.pl";
    my $cmd = "$termExec -b $PORT -o " . ($PASSWORD ne "" ? " -p $PASSWORD " : "") .
    " $TmpExpFile $tcmd ";

    my $out = `$cmd`;
    my @lines = split(/\n/, $out);
    return @lines;
}


our %ENTINFO = GetEntitiesInfo ();
our %BWINFO  = GetBandwidthInfo ();

my $lboard = GetLeaderBoard ();
my $sstats = GetServerStatistics ();
my $date = localtime;

print <<EOT;
<div style='background: #ffffd4; border: 1px solid #c0c0c0; padding: 5px; font-size: 14px;'>
Latest Colyseus experiment status retrieved at: <strong>$date</strong>
</div>

$lboard

$sstats
EOT

unlink ("$TmpExpFile");

###################################################################################

sub GetServerStatistics {
    my $html = "";

    my $html = <<EOT;
<div class='servstats'>
<h3>Server Statistics</strong></h3>
  <table style='$fs; width: 80%;' border=0>
    <tr><$th>Statistic</td><$th colspan="2">Value</td></tr>
EOT

    foreach my $s (keys %ENTINFO) {
	my $info = $ENTINFO {$s};
	
	my $name;
	my $online;
	my $nplayers = scalar @{$info->{stats}};

	if ($info->{alive}) {
	    $online = "yes";
	    $name = $info->{host};
	} else {
	    $online = "no";
	    $name = "<span style=\"color: red\">$info->{host}</span>\n";
	}
	
	if ($online eq "no") { 
	    $html .= "<tr><$td>$name</td><$td colspan='2'>Not online</td></tr>\n";
	    next;
	}

	$html .= "<tr><td style='background-color: #fad163;' colspan='3'>$name</td></tr>\n";

	my ($ro, $rp, $rm, $po, $pp, $pm);
	$rp = $info->{repstats}->{player};
	$rm = $info->{repstats}->{missile};
	$ro = $info->{repstats}->{item};

	$pp = $info->{pristats}->{player};
	$pm = $info->{pristats}->{missile};
	$po = $info->{pristats}->{item};

	my $mstyle = 'padding-left: 10px;';
	$html .= <<EOT;
<tr>
   <$td>Replica statistics</td>
   <$td colspan="2">$rp players, $rm missiles, $ro others</td>
</tr>
<tr>
   <$td>Primary statistics</td>
   <$td colspan="2">$pp players, $pm missiles, $po others</td>
</tr>
EOT
	if (defined $BWINFO {$s} and exists $BWINFO {$s}->{"OM:inbound"}) {
	    foreach my $d ("inbound", "outbound") {
		foreach my $t ("OM", "MERC_SUB", "MERC_PUB", "MERC_MATCH") {
		    my $pkts = $BWINFO {$s}->{"$t:$d"}->[0];
		    my $bytes = $BWINFO {$s}->{"$t:$d"}->[1];

		    my $kbps = sprintf ("%.2f", $bytes * 8 / 1000.0);
		    $html .= <<EOT;
<tr>
   <td>$d $t</td>
   <td>$pkts pkts</td>
   <td>$kbps kbps</td>
</tr>
EOT
		}
		$html .= "<tr><td colspan='3' style='border: 1px solid #c0c0c0;'></td></tr>\n";
	    }
	}
    }
    $html .= <<EOT;
  </table>
</div>
EOT

    return $html;
}

sub getWeight {
    my $x = shift;
    if ($x->[0] =~ /Bot/) { 
	return $x->[1];
    }
    else {
	return $x->[1] + 1000;
    }
}

sub sortPlayer {
    return getWeight ($b) <=> getWeight ($a);
}

sub GetLeaderBoard {
    my @players = ();
    foreach (keys %ENTINFO) { 
	push @players, @{$ENTINFO{$_}->{stats}};
    }

    @players = sort sortPlayer @players;
    
    my $html = <<EOT;
<div style='padding: 20px 0px;' >

<div class='servstats' style='float: right;'>
<h3>Server List</h3>
  <table style='$fs; width: 310px;' border=0>
    <tr><$th>Server Address</td><$th>Online</td><$th>Players/Max</td></tr>
EOT
    foreach my $s (keys %ENTINFO) {
	my $info = $ENTINFO {$s};

	my $name;
	my $online;
	my $nplayers = scalar @{$info->{stats}};

	if ($info->{alive}) {
	    $online = "yes";
	    $name = $info->{host};
	} else {
	    $online = "no";
	    $name = "<span style=\"color: red\">$info->{host}</span>\n";
	}

	print STDERR "$name\t$online\n";
	$html .= "<tr><$td>$name</td><$td>$online</td><$td>$nplayers/$info->{maxplayers}</td></tr>\n";
    }
    $html .= <<EOT;
  </table>
</div>
EOT

    $html .= "<div class='leaderboard'>\n";
    $html .= "<h3>Leaderboard</h3>\n";
    $html .= "<table style=\"$fs; width: 310px;\" border=0><tr><$th>Rank</td><$th>Player</td><$th>Score</td></tr>\n";

    my $i = 0;
    foreach my $p (@players) {
	last if $i++ >= 10;
	$html .= "<tr><$td>$i</td><$td>$p->[0]</td><$td>$p->[1]</td></tr>\n";
    }

    $html .= <<EOT;
  </table>
</div>

</div>
<div style='clear: both;'/>
EOT
    return $html;
}

sub mdie {
    my @err = @_;

    print "<html><head></head><body>\n";
    print "<p style=\"$fs; font-size: 110%; color: red;\" align=\"center\">Internal Error: Please <a href=\"mailto:jeffpang+colyseus-demo-status\@cs.cmu.edu\">contact us</a> so we can fix the problem.</p>\n";
    print "</body></html>\n";

    unlink ("$TmpExpFile");
    die @_;
}

sub GetEntitiesInfo {
    my %info = ();

    my $curr = undef;
    my @lines = GetTermExecOutput ("GE P P");

    foreach my $line (@lines) {
	chomp $line;

	if ($line =~ /^[\w\d_-]+\@([\w\d_\-\.]+):([\w\d_\-\.]+)\s+(.*)/) {
	    my ($host, $iface) = ($1, $2);
	    my $title = $3;

	    $info {$title} = { 
		host => $host, iface => $iface,
		repstats => {}, pristats => {},
		alive => 1,
		maxplayers => 0,
		stats => []
	    };
	    $curr = $title;
	}
	if ($line =~ /maxplayers=(\d+) numreplicas=(\d+) (\d+) (\d+) numprimaries=(\d+) (\d+) (\d+)/) { 
	    my $h = $info {$curr}->{repstats};

	    mdie "bad format: $line" if !$curr;

	    $info {$curr}->{maxplayers} = $1;

	    $h->{player} = $2;
	    $h->{missile} = $3;
	    $h->{item} = $4;

	    $h = $info {$curr}->{pristats};
	    $h->{player} = $5;
	    $h->{missile} = $6;
	    $h->{item} = $7;
	}

	if ($line =~ /edict:.* class=(\w+) .*name=([^\t ]+) .*score=([-\d]+)/) {
	    my ($class, $name, $score) = ($1, $2, $3);

	    mdie "bad format: $line" if !$curr;

	    my $arr = $info {$curr}->{stats};
	    push @{$arr}, [ $name, $score ];
	} 
	elsif ($line =~ /NO RESPONSE/) {
	    mdie "bad format: $line" if !$curr;

	    $info {$curr}->{alive} = 0;  # dead;
	}
    }
    mdie "no servers! @lines" if scalar (keys %info) == 0;
    return %info;
}

sub GetBandwidthInfo {
    my %info  = ();

    my $curr = undef;
    my @lines = GetTermExecOutput ("BW");

    foreach my $line  (@lines) {
	chomp $line;

	if ($line =~ /^[\w\d_-]+\@([\w\d_\-\.]+):([\w\d_\-\.]+)\s+(.*)/) {
	    my ($host, $iface) = ($1, $2);
	    my $title = $3;

	    $info {$title} = { 
		host => $host, iface => $iface,
	    };
	    foreach my $t ("OM", "MERC_SUB", "MERC_PUB", "MERC_MATCH") {
		foreach my $d ("inbound", "outbound") {
		    $info {$title}->{"$t:$d"} = [];
		}
	    }

	    $curr = $title;
	}

	elsif ($line =~ /OK:/) {

	    mdie "bad format: $line" if !$curr;

	}
	elsif ($line =~ /(\w+) (\w+) (\d+) pkts, (\d+) bytes/) { 
	    my ($type, $inout, $pkts, $bytes) = ($1, $2, $3, $4);

	    mdie "bad format: $line" if !$curr;
	    $info {$curr}->{"$type:$inout"} = [ $pkts, $bytes ];
	}

    }

    mdie "no servers! @lines" if scalar (keys %info) == 0;
    return %info;
}

