package Options;
# Author: Ashwin Bharambe   <ashu@cs.cmu.edu>    Copyright (C) 2006. 
require 5.6.1;
require Exporter;

our $VERSION = '$Id: Options.pm 3198 2006-10-17 18:04:53Z ashu $';

# standard libraries/pragmas
use strict;
use Carp;

our @ISA    = ("Exporter");
our @EXPORT = qw(OPT_NOARG OPT_INT OPT_STR OPT_FLT OPT_CHR OPT_BOOL ProcessOptions PrintOptionValues PrintUsage);

our $VERBOSE = 0;

use constant OPT_NOARG => 0x01;
use constant OPT_INT   => 0x02;
use constant OPT_STR   => 0x04;
use constant OPT_FLT   => 0x08;
use constant OPT_CHR   => 0x10;
use constant OPT_BOOL  => 0x20;

sub owarn(@) {
    print STDERR ("[30;41m[EE][m ", @_, "\n");
}

sub PrintUsage($) {
    return _PrintUsage(ConvertOptTable($_[0]));
}

sub PrintOptionValues($) {
    return _PrintOptionValues(ConvertOptTable($_[0]));
}

# each option type has the form: "
sub _PrintUsage($) {
    my $optbl = shift;

    print STDERR "available options:\n";
    my $maxlen = 0;
    foreach my $opt (@$optbl) {
	$maxlen = length($opt->{longword}) if length($opt->{longword}) > $maxlen;
    }
    
    $maxlen += 2;
    foreach my $opt (@$optbl) {
	print STDERR sprintf "\t-[32m%s[m,--[33m%s[m:", $opt->{shortletter}, $opt->{longword};
	print STDERR " "x($maxlen - length($opt->{longword}));
	print STDERR $opt->{comment}, " [[4;37m", $opt->{default_val},
	"[m]\n";
    }
    print STDERR "\n";
}

sub _PrintOptionValues($) {
    my $optbl = shift;
    my $maxlen = 0;
    foreach my $opt (@$optbl) {
	$maxlen = length($opt->{longword}) if length($opt->{longword}) > $maxlen;
    }
    print STDERR "Current option values...\n"; 
    foreach my $opt (@$optbl) {
	print STDERR sprintf ("\t-%s,--%s:", $opt->{shortletter},
		$opt->{longword});
	print STDERR " "x($maxlen - length($opt->{longword}));
	if ($opt->{flags} & OPT_BOOL) {
	    my $v = ${$opt->{varptr}};
	    print STDERR "[35m", ($v eq "1" ? "true" : "false"), "[m\n";
	}
	else {
	    print STDERR "[35m", ${$opt->{varptr}}, "[m\n";
	}
    }
}

sub _AssignDefaults ($) { 
    my $optbl = shift;
    foreach my $opt (@$optbl) {
	_AssignValue($opt, $opt->{default_val});
    }
}

sub _AssignValue ($$) {
    my $opt = shift;
    my $val = shift;

    # will this work?
    ${$opt->{varptr}} = $val;
}

# arguments are: 
#  (1) option_table = ( [option_desc_map1], [option_desc_map2], ... )
#  (2) reference to argv
#  (3) other options for processing
#
# returns the unprocessed options
sub _ProcessOptions($$%) {
    my $optbl = shift;
    my $pargv = shift;
    my %conf  = @_;
    my @rargv = ();
    
    _AssignDefaults($optbl);
    while (defined ($_ = shift @$pargv)) {
	if ($_ eq '-h' or $_ eq '--help' or $_ eq '--h') {
	    _PrintUsage($optbl);
	    exit 0;
	}
	
	my $found = 0;
	foreach my $opt (@$optbl) { 
	    my $l = "--" . $opt->{longword};
	    my $s = "-" . $opt->{shortletter};
	    
	    if ($_ eq $l or $_ eq $s) { 
		$found = 1;
		if ($opt->{flags} & OPT_NOARG) {
		    _AssignValue($opt, ($opt->{val_to_set} ?
				$opt->{val_to_set} : undef));
		}
		else {
		    $_ = shift @$pargv;
		    die "optval not defined for option $l ($s)" if !defined $_;
		    _AssignValue($opt, $_);
		}
		last;
	    }

	}

	if (!$found) {
	    owarn "unknown option $_" if $conf{-complain};
	    push @rargv, $_;
	}
    }

    return @rargv;
}

sub ConvertOptTable($) {
    my $optbl = shift;
    my $noptbl = [];
    
    foreach my $opt (@$optbl) { 
	my $nopt = {};

	$nopt->{shortletter} = $opt->[0];
	$nopt->{longword} = $opt->[1];
	$nopt->{flags} = $opt->[2];
	$nopt->{comment} = $opt->[3];
	$nopt->{varptr} = $opt->[4];
	$nopt->{default_val} = $opt->[5];
	$nopt->{val_to_set} = $opt->[6];

	push @$noptbl, $nopt;
    }
    return $noptbl;
}

sub ProcessOptions ($$%) {
    my $optbl = shift;
    my $pargv = shift;
    my %conf = @_;
    
    return _ProcessOptions(ConvertOptTable($optbl), $pargv, %conf);
}

# utility function for common boolean options
#  assumes default_val = undef
#  and     val_to_set  = "1"
#
sub Boolean($$$$) {
    return [$_[0], $_[1], OPT_BOOL | OPT_NOARG, $_[2], $_[3], undef, "1"];
}

sub String($$$$$) {
    return [$_[0], $_[1], OPT_STR, $_[2], $_[3], $_[4], undef];
}

1;
__END__

=head1 NAME

Options - parse command line options a little more flexibly

=head1 SYNOPSIS

Following is an example:

my @options = (
	
	[ "H", "hostname", OPT_STR, "hostname - required on emulab machines", \$HOSTNAME, "127.0.0.1", undef ],
	[ "p", "port", OPT_INT, "mercury port", \$PORT, "7000", undef ],
	[ "m", "dothis", OPT_BOOL | OPT_NOARG, "dooo this", \$DOTHIS, "1", "0" ]
	);

my @argv = split (/\s+/, '-p 9000 --invalid_opt -m -H node0.emulab.net');

ProcessOptions(\@options, \@argv);

ProcessOptions(\@options, \@argv, --complain => 1);

PrintOptionValues(\@options);

=head1 DESCRIPTION

Nothing much. 

=cut
