#!/usr/bin/perl
use strict;
use Travertine;
use Options;
use IO::Socket;

##################################################################
use vars qw($HOSTS $VSERVERS $PASSWORD $BASEPORT $INTERVAL);

my @options = (
    Options::String ("H", "hosts", "comma-separated list of hosts", \$HOSTS, ""),
    Options::String ("v", "vservers", "how many vservers per host", \$VSERVERS, "1"),
    Options::String ("P", "password", "password for the terminal", \$PASSWORD, ""),
    Options::String ("b", "baseport", "base port which vservers start from", \$BASEPORT, "30000"),
    Options::String ("i", "pollinterval", "how many seconds to sleep in between", \$INTERVAL, "2"),
);

@ARGV = ProcessOptions (\@options, \@ARGV, -complain => 1);
if ($HOSTS eq "") { 
    tdie "Option -H, --hosts is compulsory. ";
}

our $vizSocket = IO::Socket::INET->new(Proto => 'udp') or die "socket: $@";     
our $vizIP     = inet_aton ("127.0.0.1");
our $vizPort   = 60000;
our $vizPortAddr = sockaddr_in($vizPort, $vizIP);

my @hosts = split (/,/, $HOSTS);

use constant READ_NOTHING => 0x0;
use constant READ_LENGTH => 0x1;
use constant READ_FIRST_DELIM => 0x2;
use constant READ_SECOND_DELIM => 0x3;

# open all terminal sockets. make all o them happy by giving password, etc.
my %handles = ();
my $id = 1;
my @tuples = ();
foreach my $host (@hosts) { 
    for (my $i = 0; $i < $VSERVERS; $i++) { 
	my $port = $BASEPORT + $i;
	my $sock = openTerminal ($host, $port);
	$handles {"$h::$i"} = $sock;

	push @tuples, [ $sock, $id ];
	$id ++;
    }
}

our $MAX_PARALLEL = 10;

while (1) {

    foreach my $x (@tuples) {
	my $sock = $x->[0];
	my $id = $x->[1];

	getVisInfo ($sock, $id);
    }

    my $sleep = $INTERVAL;
    tinfo "Sleeping for $sleep seconds ";
    sleep ($sleep);
}

sub sendData {
    my $id = shift;
    my $buf = shift;

    my $header = pack ("CC", 0xff, $id);
    $buf = $header . $buf;

    send ($vizSocket, $buf, 0, $vizPortAddr) or die ("cannot send to localhost ($vizPort): $!");
}

sub openTerminal {
    my $host = shift;
    my $port = shift;

    my $sock = IO::Socket::INET->new(PeerAddr => $host,
				     PeerPort => $port,
				     Proto    => 'tcp',
				     Type     => SOCK_STREAM,
				     Timeout  => 20);

    if (!$sock) {
	twarn "can't open socket to $host:$port";
	return undef;
    }

    tinfo "Connected successfully to $host:$port";

    if ($PASSWORD ne "") {
	print $sock "$PASSWORD\n";
    }
    # read off the message banner;
    my $banner = read_line($sock, undef);

    if ($banner !~ /Colyseus Terminal/) {
	chomp $banner;
	$banner =~ s/^Password: //;
	$sock->close();
	twarn "$host:$port closed socket, it said: $banner";
	return undef;
    }

    return $sock;
}

sub read_line($$) {
    my $fh = shift;
    my $timeout = shift;
    $timeout = 30 if (!defined $timeout);
    my $res = undef;

    eval {
        local $SIG{ALRM} = sub { die "alarm clock restart" };
        alarm $timeout;
        eval {
            $res = <$fh>;
            $res =~ s/\r\n/\n/g;
        };
        alarm 0;

        if ($@ && $@ =~ /alarm clock restart/) {
            die "timeout failure";
        }
    };
    alarm 0;

    if ($@ && $@ =~ /timeout failure/) {
        twarn "timed out waiting for response";
        return undef;
    }
    return $res;
}

sub getVisInfo {
    my $sock = shift;
    my $id   = shift;

    my $state = READ_NOTHING;
    my $length = undef;
    my $buf = "";
    my $junk = "";

    print $sock "GETVIS\n";

    while ($junk = read_line($sock, undef)) {
	last if $junk =~ /^(Password: )?==BEGIN_RESP/;
    }

    while (<$sock>) {
	next if (/OK/);
	last if (/==END_RESP/);

	if ($state == READ_NOTHING) {
	    if (/^length=(\d+)/) {
		$state = READ_LENGTH;
		$length = $1;
	    }
	}
	elsif ($state == READ_LENGTH) {
	    if (/^=======/) {
		$state = READ_FIRST_DELIM;
		tdie "No length here" if (!defined $length or $length <= 0);

		my $nread = read ($sock, $buf, $length);
		tdie "Not enough to read" if ($nread < $length);

		sendData ($id, $buf);
		read ($sock, $junk, 1);   # read the newline;
	    }
	}
	elsif ($state == READ_FIRST_DELIM) {
	    if (/^=========/) {
		$state = READ_NOTHING;
	    }
	}
    }
}

sub execCommand {
    my $host = shift;
    my $sock = shift;
    my $cmd = shift;

    print $sock "$cmd\n";
    my $line;
    my $resp;
    while ($line = read_line($sock, undef)) {
	next if $line =~ /^(Password: )?==BEGIN_RESP/;
	last if $line =~ /^==END_RESP/;
	$resp .= $line;
    }
    if ($line !~ /^==END_RESP/) {
	twarn "$host: resp did not end with ==END_RESP:\n" . thighlight($resp);
    } else {
	$resp .= "";
    }

    return $resp;
}

