#!/usr/bin/perl
#
# Installs Colyseus and Quake on remote nodes
#
# The installation proceeds as follows:
#
# (1) A local copy of the 'distribution' is made in $tmp by
#     sym-linking files from $LOCAL_TOPDIR.
# (2) A pre-install procedure is run on all the remote hosts.
# (3) A sync-procedure is run locally to synchronize the latest local 
#     distribution to each remote host.
# (4) A post-install procedure is run on all the remote hosts.
# (5) [Optional] The local copy of the distribution is removed.

use strict;
use lib "$ENV{HOME}/Colyseus/run";
use lib "$ENV{HOME}/Colyseus/run/lib";
use Travertine;
use Options;

###############################################################################
# Options

# Function to execute locally to synchronize files to a remote host
# The function is passed ($local_dist_dir, $remote_host_login, $remote_topdir)
# The function should FOLLOW sym-links
#
our $sync = sub {
    my ($ldir, $login, $rdir) = @_;
    my $ret = psystem("rsync -rtLz --exclude '.svn*' -e ssh -v $ldir/* $login:$rdir/ | egrep -v '/\$'");
    twarn "rsync to $login failed (see above messages)" if $ret;
};

#
# Temp directory where we build the rsync dist locally
#
our $tmp = "/tmp/colquake.dist";
 
###############################################################################

our $APP_CONF = shift @ARGV;
Usage() if (!$APP_CONF);

our @APP_INST_DIST;
our $APP_INST_PRE;
our $APP_INST_POST;

tdie "bad conf file: '$APP_CONF'" if !-f $APP_CONF;
no strict;
require "$APP_CONF";
use strict;

if (!defined @APP_INST_DIST) { 
    tdie "variable \@APP_INST_DIST not defined; nothing to install?";
}
if (!defined $APP_INST_PRE) { 
    tdie "subroutine \$APP_INST_PRE not defined";
}
if (!defined $APP_INST_POST) {
    tdie "subroutine \$APP_INST_POST not defined";
}

use vars qw($LOCAL_TOPDIR $TOPDIR $CLEAN $MAXPARALLEL $PRINT $INST_LIST $opt_Z $opt_r $VERBOSE $RSYNC_OPTS);

my @options = (
    Options::String ('l', 'localdir', 'local top directory (pubsub dir)', \$LOCAL_TOPDIR, $ENV{HOME}),
    Options::String ('t', 'topdir', 'install in this directory remotely', \$TOPDIR, ""),
    Options::Boolean ('c', 'clean', 'cleanup local tmp dist after finished', \$CLEAN),
    Options::String ('p', 'maxpar', 'maximum parallel executions in flight (def=10)', \$MAXPARALLEL, 10),
    Options::String ('i', 'instlist', 'install the specified files only', \$INST_LIST, ""),
    Options::Boolean ('/', 'print',   'list all the files going to be installed', \$PRINT),
    Options::Boolean ('Z', 'nocompress', "don't use compression in rsync", \$opt_Z),
    Options::Boolean ('r', 'recdeps', 'include myself + deps in install list', \$opt_r),
    Options::Boolean ('v', 'verbose', 'verbose output', \$VERBOSE)
);

@ARGV = ProcessOptions (\@options, \@ARGV, -complain => 0);
if ($PRINT) {
    print_dist ();
    exit 0;
}

if (@ARGV == 0) {
    Usage();
}

sub Usage () {
    print STDERR "usage: Install.pl AppConf.pl [options] [user1\@]host1 [user2\@]host2 ...\n\n";
    PrintUsage (\@options);
    exit 1;
}

tdie "need -t topdir!" unless ($TOPDIR ne "");

if ($opt_Z) {
    $RSYNC_OPTS = "-rtL";
} else {
    $RSYNC_OPTS = "-rtLz";
}

if ($opt_r) {
    # xxx: what about conf files that aren't in Colyseus/run?
    #my $conf   = `basename $APP_CONF`; chomp $conf;
    my @MYSELF = ( "Colyseus/run/Install.pl",
		   "Colyseus/run/Travertine.pm",
		   "Colyseus/run/Options.pm",
		   "Colyseus/run/lib-*.tar.gz",
		   "Colyseus/run/*Conf.pl"
		    );
    #push @MYSELF, $APP_CONF if $APP_CONF !~ m|/|;
    my %dist = map { $_ => 1 } @APP_INST_DIST;
    foreach my $p (@MYSELF) {
	if (!$dist{$p}) {
	    push @APP_INST_DIST, $p;
	}
    }
    # xxx: should also untar the libs in post-install...
}

our @logins = @ARGV;
@logins = map { if ($_ !~ /\@/) { $_="$ENV{USER}\@$_" } else { $_ } } @logins;

###############################################################################

tinfo "* making current dist";
make_dist();

tinfo "* running pre-install";
do_all($APP_INST_PRE);

tinfo "* performing sync";
do_sync();

tinfo "* running post-install";
do_all($APP_INST_POST);

if ($CLEAN) {
    tinfo "* cleaning up tmp dist";
    psystem("rm -rf $tmp");
}

###############################################################################

sub print_dist { 
    foreach my $glob (@APP_INST_DIST) {
	print STDERR $glob, "\n";
    }
}

sub make_dist {
    my $ret = 0;

    $ret ||= psystem "rm -rf $tmp";
    
    if ($INST_LIST ne "") { 
	@APP_INST_DIST = split (/,/, $INST_LIST);
    }

    foreach my $glob (@APP_INST_DIST) {
	my $dirname = $glob;
	$dirname =~ s/[^\/]*$//;
	$ret ||= psystem "mkdir -p $tmp/$dirname";
	$ret ||= psystem "cp -sHLrf $LOCAL_TOPDIR/$glob $tmp/$dirname";

	tdie "failed: $!" if $ret;
    }
}

sub do_all {
    my $func   = shift;
    return if !$func;

    ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	tinfo "$mach started." if $VERBOSE;
	rsystem($user, $mach, $func, $TOPDIR, $user);
	tinfo "$mach completed." if $VERBOSE;
    }, @logins);
}

sub do_sync {
    tdie "no sync function defined" if !$sync;
    ParallelExec3($MAXPARALLEL, sub {
	my $login = $_[1];
	my ($user, $mach) = split(/\@/, $login);
	tinfo "$mach started." if $VERBOSE;
	my $ret = &{$sync}(@_);
	tinfo "$mach completed." if $VERBOSE;
	return $ret;
    }, map { [$tmp, $_, $TOPDIR] } @logins);
}
