#!/usr/bin/perl
#
# $Id: BTInstall.pl 3203 2006-10-18 23:06:38Z ashu $
#    
##########################################################################
#
# If gethostname gives a stupid localhost.localdomain on your machine,
# be sure to provide the -H option...  
#
# Installs Colyseus and Quake on remote nodes. Modification of the Install.pl 
# script. This one pushes the software using BitTorrent. 
#
# The installation proceeds as follows:
#
# (1) A local copy of the 'distribution' is made in $tmp by
#     sym-linking files from $LOCAL_TOPDIR. A .torrent file is 
#     made from this. 
# (2) The tracker and seed-node are started locally. 
# (3) A pre-install procedure is run on all the remote hosts. The 
#     remote nodes should have the bittorrent rpm installed.
#     The .torrent file as well as one additional python script 
#     is rsync'ed to all the nodes. 
# (4) The btpull.py script is started at each remote host, which pulls the
#     software using bittorrent.
# (5) A post-install procedure is run on all the remote hosts.
# (6) [Optional] The local copy of the distribution is removed.

use strict;
use Travertine;

###############################################################################
# Options

# We have to execute this in addition to the normal application _PRE function

our $pre = sub {
    # ps behaves badly sometimes, so have to resort to killall :(
    psystem("killall -9 python >/dev/null 2>&1");
};

our $APP_CONF = shift @ARGV;
Usage() if (!$APP_CONF);

our @APP_INST_DIST;
our $APP_INST_PRE;
our $APP_INST_POST;

tdie "bad conf file: '$APP_CONF'" if !-f $APP_CONF;
no strict;
require "$APP_CONF";
use strict;

if (!defined @APP_INST_DIST) { 
    tdie "variable \@APP_INST_DIST not defined; nothing to install?";
}
if (!defined $APP_INST_PRE) { 
    tdie "subroutine \$APP_INST_PRE not defined";
}
if (!defined $APP_INST_POST) {
    tdie "subroutine \$APP_INST_POST not defined";
}

###############################################################################

use vars qw($LOCAL_TOPDIR $TOPDIR $CLEAN $MAXPARALLEL $PRINT $INST_LIST $VERBOSE $HOSTNAME $DST_TMP_DIR $opt_k $opt_P $opt_s $opt_S);

my @options = (
    Options::String ('l', 'localdir', 'local top directory (pubsub dir)', \$LOCAL_TOPDIR, $ENV{HOME}),
    Options::String ('t', 'topdir', 'install in this directory remotely', \$TOPDIR, ""),
    Options::String ('T', 'tmpdir', 'temp directory at the remote host (def=/tmp)', \$DST_TMP_DIR, "/tmp"),
    Options::Boolean ('c', 'clean', 'cleanup local tmp dist after finished', \$CLEAN),
    Options::String ('p', 'maxpar', 'maximum parallel executions in flight (def=10)', \$MAXPARALLEL, 10),
    Options::String ('i', 'instlist', 'install the specified files only', \$INST_LIST, ""),
    Options::String ('H', 'hostname', 'hostname to use for ourselves', \$HOSTNAME, `hostname -i`),
    Options::Boolean ('k', 'killremote', 'kill remote bt instances only', \$opt_k),
    Options::Boolean ('P', 'skip-preinstall', 'dont do pre-install', \$opt_P),
    Options::Boolean ('s', 'skip-btmakefile', 'dont make the distribution. just sync.', \$opt_s),
    Options::Boolean ('S', 'skip-sync', 'dont even sync', \$opt_S),
    Options::Boolean ('/', 'print',   'list all the files going to be installed', \$PRINT),
    Options::Boolean ('v', 'verbose', 'verbose output', \$VERBOSE)
);

@ARGV = ProcessOptions (\@options, \@ARGV, -complain => 0);
if ($PRINT) {
    print_dist ();
    exit 0;
}

if (@ARGV == 0) {
    Usage();
}

sub Usage() {
    print STDERR "usage: BTInstall.pl AppConf.pl [options] [user1\@]host1 [user2\@]host2 ...\n\n";
    PrintUsage (\@options);
    exit 1;
}

### these are BTInstall specific
chomp $HOSTNAME;

# pick a random tracker port so that old instances are unlikely to contact us
# XXX really should pass all the hosts to `btsource' so it knows exactly which
#     ones it should wait for and ignore the rest.
our $TRACKER_PORT = 55000 + int(rand(10000));
our $TRACKER_URL = "http://$HOSTNAME:$TRACKER_PORT/announce";

tdie "need -t topdir!" unless ($TOPDIR ne "");

## Temp directory where we build the rsync dist locally
our $tmp = `basename $TOPDIR`;
chomp $tmp;
$tmp = "/tmp/$tmp";

our @logins = @ARGV;
@logins = map { if ($_ !~ /\@/) { $_="$ENV{USER}\@$_" } else { $_ } } @logins;

###############################################################################
tinfo "* checking if bittorrent is installed";
my $ret = psystem "python -c 'from BitTorrent.track import track;' 2>/dev/null ";
if ($ret) {     
    twarn "--BitTorrent modules not found! Install the bittorrent rpm or \nset PYTHONPATH to the location of the BitTorrent/ module--";
    exit 1;
}

if ($opt_k) {
    tinfo "* killing remote bt clients";
    do_all($pre);
    exit 0;
}

if (! ($opt_S || $opt_s)) {
    tinfo "* making current dist";
    make_dist();
}

if (!$opt_P) {
    tinfo "* running pre-install";
    # save duplicate ssh connections;
    my $pre2 = Travertine::CombineFuncs($APP_INST_PRE, $pre); 
    do_all($pre2);
}

our $sync;

if (!$opt_S) {
    tinfo "* starting tracker";
    my $trpid = start_tracker_seed(scalar @logins);

    tinfo "* performing sync";
    # This function pushes the relevant bittorrent scripts to the node
    # along with the .torrent file. This function runs locally so we 
    # can rely on global variables
    #
    $sync = sub {
	my $login = shift;
	
	# Assume this is running in the directory containing btpull.py
	my $ret = psystem("rsync -rtLz -e ssh btpull.py $tmp.torrent $login:$DST_TMP_DIR/");
	twarn "rsync to $login failed (see above messages)" if $ret;
    };
    
    do_sync();
    
    tinfo "* starting bittorrent processes on the remote hosts";
    
    my @rbrefs = ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	
	my $ssh = Travertine::SSHTunnel->new($user, $mach);
	if (!$ssh) {
	    twarn "can't open sshTunnel to $user\@$mach";
	    return undef;
	}
	
	my $torrent = `basename $tmp`; chomp $torrent;
	$torrent = "$DST_TMP_DIR/$torrent.torrent";
	
	my @args = ($DST_TMP_DIR, $HOSTNAME, $torrent, $TOPDIR);
	return ExecRemoteFunc($ssh, \&bt_start, \@args, -background => 1, -log => "$DST_TMP_DIR/btinstall.log");
    }, @logins);
    
    tinfo "* waiting for the nodes to finish downloading";
    waitpid($trpid, 0); # wait for the tracker process to end!
}

sub bt_start {
    my ($tmpdir, $watchdog, $torrent, $destdir) = @_;
    my $pid = 0;
    
    $destdir = `dirname $destdir`;
    chomp $destdir;
     # XXX TODO: just do the rsync here. saves ssh connections.
    exec "$tmpdir/btpull.py $watchdog --minport 55000 --maxport 55009 --max_uploads 15 --responsefile $torrent --saveas $destdir";
}

=start
# try to use only one ssh connection here... 

# now kill the background btpull.py processes
tinfo "* stopping remote bittorrent processes";
ParallelExec3($MAXPARALLEL, sub {
    my $r = shift;
    $r->stop(); 
}, @rbrefs);
=cut

tinfo "* running post-install";
my $post = sub {
    psystem("killall -9 python >/dev/null 2>&1");
};

my $post2 = Travertine::CombineFuncs($APP_INST_POST, $post); 
do_all($post2);

if ($CLEAN) {
    tinfo "* cleaning up tmp dist";
    psystem("rm -rf $tmp");
}

###############################################################################
sub start_tracker_seed {
    my $ndownloaders = shift;

    my $cmd = "./btsource.py -p $TRACKER_PORT -T $tmp -n $ndownloaders";

    my $pid = fork();
    if ($pid > 0) {
	return $pid;
    }
    else {
	 # psystem "xterm -T \"Source Node\" -e bash -c \"$cmd\""; 
	psystem "$cmd";
	exit();
    }
}

sub print_dist { 
    foreach my $glob (@APP_INST_DIST) {
	print STDERR $glob, "\n";
    }
}

sub make_dist {
    my $ret = 0;

    psystem "rm -rf $tmp";
    if ($INST_LIST ne "") { 
	@APP_INST_DIST = split (/,/, $INST_LIST);
    }

    foreach my $glob (@APP_INST_DIST) {
	my $dirname = $glob;
	$dirname =~ s/[^\/]*$//;
	$ret ||= psystem "mkdir -p $tmp/$dirname";
	$ret ||= psystem "cp -sHLrf $LOCAL_TOPDIR/$glob $tmp/$dirname";
	
	tdie "failed: $!" if $ret;
    }
    
    psystem "btmakemetafile $tmp $TRACKER_URL --target $tmp.torrent";
}

sub do_all {
    my $func   = shift;
    return if !$func;

    ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	# tinfo "$mach started." if $VERBOSE;
	rsystem($user, $mach, $func, $TOPDIR, $user);
	# tinfo "$mach completed." if $VERBOSE;
    }, @logins);
}


sub do_sync {
    tdie "no sync function defined" if !$sync;
    ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	# tinfo "$mach started." if $VERBOSE;
	my $ret = &{$sync}($login);
	# tinfo "$mach completed." if $VERBOSE;
	return $ret;
    }, @logins);
}
