#!/usr/bin/perl
#
# Scaling comparison between us and client-server and broadcast.
#
# Usage: ScalingComparison.pl [options] dat_dir data_points > output
#
# E.G.: ScalingComparison.pl jan2005/nodes/scaling 5,10,20,30,40
#
# Required files in dat_dir: nodes.dat, csbc.dat
#
# Plots mean + 95% error bar (95% is NOT percentile of average but the
# average of 95 percentile bwidths on individual servers)
#
###############################################################################

use strict;
use Getopt::Std;
use vars qw($opt_p $opt_g);

getopts("g:");

my @GPLOT_OPTS = (defined $opt_g ? split(/,/, $opt_g) : ());

my $dir = shift @ARGV;
my @nodes = split(/,/, $ARGV[0]);
die if !@nodes;

my $orig = `pwd`; chomp $orig;
chdir $dir or die "can't change to directory: $dir";

# get data for merc --- grep TOTAL:OUT
my @mercs = `grep TOTAL:OUT:MEAN nodes.dat | awk '{ print \$2, \$3 }'`; chomp (@mercs);
my @cs    = `grep OUT:MEAN client-server.dat | awk '{ print \$2 }'`; chomp (@cs);
my @bcast = `grep OUT:MEAN bcast.dat | awk '{ print \$2 }'`; chomp (@bcast);

my @mercs95 = `grep TOTAL:OUT:95% nodes.dat | awk '{ print \$2 }'`; chomp (@mercs95);
my @cs95    = `grep OUT:95% client-server.dat | awk '{ print \$2 }'`; chomp (@cs95);
my @bcast95 = `grep OUT:95% bcast.dat | awk '{ print \$2 }'`; chomp (@bcast95);

die "bad points" if (!@mercs || !@cs || !@bcast);

open F, ">/tmp/dat";
foreach my $i (0..$#nodes) {
    my ($mean, $std) = split(/\s+/, $mercs[$i]);
    print F "$nodes[$i] $mean $mean $mercs95[$i] $std\n";
}
print F "\n\n";
foreach my $i (0..$#nodes) {
	print F "$nodes[$i] $cs[$i] $cs[$i] $cs95[$i]\n";
}
print F "\n\n";
foreach my $i (0..$#nodes) {
	print F "$nodes[$i] $bcast[$i] $bcast[$i] $bcast95[$i]\n";
}
close F;

###############################################################################

chdir $orig;

my $gpl = <<EOT;
set terminal postscript eps 20
set xlabel "Number of nodes"
set ylabel "Per-node outgoing bandwidth (kbps)"
set key t l
set size 0.8,0.8
set logscale y
set border 31 lw 1.4
set grid ytics lw 2 
set yrange [10:100000]
EOT

if (@GPLOT_OPTS) {
    foreach my $opt (@GPLOT_OPTS) {
	$gpl .= "$opt\n";
    }
}

$gpl .= <<EOT;

plot \\
    '/tmp/dat' index 0 using 1:(\$1*\$2) title 'colyseus-aggregate' w lp lw 2.5 ps 1 pt 4 lt 4, \\
    '/tmp/dat' index 0 using 1:2:(\$1*\$3):(\$1*\$4) notitle w yerrorbars lw 1.5 ps 1 pt 4 lt 4, \\
    '/tmp/dat' index 1 title 'single-server' w lp lw 2.5 ps 1 pt 7 lt 2, \\
    '/tmp/dat' index 1 notitle w yerrorbars lw 1.5 ps 1 pt 7 lt 2, \\
    '/tmp/dat' index 2 title 'broadcast' w lp lw 2.5 ps 1 pt 9 lt 3, \\
    '/tmp/dat' index 2 notitle w yerrorbars lw 1.5 ps 1 pt 9 lt 3, \\
    '/tmp/dat' index 0 title 'colyseus' w lp lw 2.5 ps 1 pt 5 lt 1, \\
    '/tmp/dat' index 0 notitle w yerrorbars lw 1.5 ps 1 pt 5 lt 1, \\
    '/tmp/dat' index 0 using 1:2:2:(\$2+\$5) notitle w yerrorbars lw 5 ps 1 pt 5 lt 1
EOT

open G, "| gnuplot";
print G $gpl;
close G;
