////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************

    Terminal.cpp

    Debugging terminal that can be used to log into the manager and inspect
    stuff online.

	begin		   : October 10, 2003
	copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )
 
***************************************************************************/

#include <Colyseus.h>
#include <om/Manager.h>
#include <om/Terminal.h>
#include <util/Benchmark.h>

typedef map<string, guid_t> StrGUIDMap;
typedef StrGUIDMap::iterator StrGUIDMapIter;
typedef map<guid_t, string, less_GUID> StringMap;
typedef StringMap::iterator StringMapIter;

static char *extract_str(char *buf, int *len);
static char **extract_args(char *buf);
static void CheckTimeouts(struct timeval *now);
static void Cleanup(int sig);

///////////////////////////////////////////////////////////////////////////////

static InputHandler g_DefaultHandlers[] = {
    { "VER",    "V",  TerminalWorker::handleVersion, "",
      "Show the version of libcolyseus" },
    { "HELP",   "H",  TerminalWorker::handleHelp,   "",
      "Show this help message" },
    { "MIGRATE",  "M",  TerminalWorker::handleMigrate,  "<ID> <SID>",
      "Migrate the primary <ID> to the node <SID>" },
    { "SHOW",   "S",  TerminalWorker::handleShow,   "<ID>",
      "Show the object <ID>" },
    { "STORE",  "O",  TerminalWorker::handleStore,  "", 
      "Show the entire object store" },
    { "PENDING",  "P",  TerminalWorker::handlePending,  "", 
      "Show the entire pending store" },
    { "GRAPH",  "G", TerminalWorker::handleGraph, "",
      "Dump the interest-graph in format <src>,<tgt>,<link_time>" },
    { "GRAPHVIS", "GV", TerminalWorker::handleGraphVis, "",
      "Dump the interest-graph and display in GraphVis" },
    { "OBJINFO",  "OI", TerminalWorker::handleObjectInfo, "",
      "Dump the object info map" },
    { "LOAD",   "L", TerminalWorker::handleLoad, "",
      "Obtain the load value of the node" },
    { "LOADMAP","LM", TerminalWorker::handleLoadMap, "",
      "Dump the load map of the node in format <tgt>,<toLow>,<toHigh>" },
    { "BENCHMARK","BM", TerminalWorker::handleBenchmark, "",
      "Dump benchmark numbers" },
    { "ROTATE","R", TerminalWorker::handleRotate, "",
      "Rotate all measurement logs" },
    { "CONN","CO", TerminalWorker::handleReplicaConn, "",
      "Dump the replica connections" },
    { "DELGUIDS","DEL", TerminalWorker::handleDelGUIDs, "",
      "Dump the deleted GUIDs list" },

    { "SUBLIST","SUB", TerminalWorker::handleSubList, "",
      "Dump the subscriptions stored at this node" },
    { "PUBLIST","PUB", TerminalWorker::handlePubList, "",
      "Dump the publications stored at this node" },

    { "QUIT","Q", TerminalWorker::handleQuit, "",
      "Close connection" },
    { NULL, NULL, NULL, NULL }
};

static InputHandlers  g_Handlers;
static Manager       *g_Manager;

///////////////////////////////////////////////////////////////////////////////

Terminal::Terminal(uint16 port, Manager *manager) 
{ 
    g_Manager = manager;
	
    try {
	m_LocalSock = ServerSocket(port);
    } catch (SocketException ex) {
	Debug::die("can't create socket: %s", ex.description().c_str() );
    }

    InputHandler *ih = g_DefaultHandlers;
    while (ih->cmd) {
	g_Handlers.push_back(ih);
	++ih;
    }
    manager->m_Game->InstallTerminalHandlers(&g_Handlers);

    // FIXME: Search for duplication for command names here 
    // in reality the API should be structured in a different 
    // way, perhaps. like: handlers->Add (handler), so the
    //  Add method can take care of checking for duplication.

    for (int i=0; i<THREAD_POOL_SIZE; i++) {
	pool[i] = NULL;
    }

    DBG << "Terminal started listening on port " << port << endl;
}

Terminal::~Terminal()
{ 
    m_LocalSock.Close();
}

void Terminal::Run()
{
    while (1) {
	ServerSocket sock;
		
	DBG << "accepting new connections..." << endl;
		
	try {
	    m_LocalSock.Accept(sock);
	    sock.SetNonBlocking(true);
	} catch (SocketException ex) {
	    WARN << "accept on local socket failed: " 
		 << ex.description() << "\n";
	    continue;
	}
	DBG << "accepted new client on socket: " << sock.GetSocket() << endl;
		
	TerminalWorker *worker = NULL;
		
	// find a thread from the thread pool
	for (int i=0; i<THREAD_POOL_SIZE; i++) {
	    if (pool[i] != NULL && pool[i]->Finished()) {
		//delete pool[i];
		//pool[i] = NULL;
		worker = pool[i];
		DBG << "restarting worker thread on socket " 
		    << sock.GetSocket() << "\n";
		worker->Restart(sock);
		break;
	    }

			
	    if (pool[i] == NULL) {
		pool[i] = new TerminalWorker(sock);
		worker = pool[i];
		DBG << "starting worker thread on socket " 
		    << sock.GetSocket() << "\n";
		worker->Start();
		break;
	    }
	}
	if (worker == NULL) {
	    // not found! -- kill the connection
	    // ignore errors here -- we're killing this client anyway
	    DBG << "too many clients connected, closing socket " 
		<< sock.GetSocket() << "\n";
	    sock << RESP_TOO_MANY_CLIENTS;
	    sock.Close();
	    continue;
	}
    }
}

TerminalWorker::TerminalWorker(ServerSocket& sock) : 
    m_Sock(sock), m_Finished(false)
{

}

TerminalWorker::~TerminalWorker()
{
    ASSERT(Finished());
}

void TerminalWorker::Restart(ServerSocket& sock)
{
    m_Sock = sock;
    
    m_Sleep.Acquire();
    m_Sleep.Signal();
    m_Sleep.Release();
}

void TerminalWorker::Run()
{
    
    while (true) {
	_Run();
	m_Sleep.Acquire();
	m_Finished = true;
	m_Sleep.Wait();
	m_Finished = false;
	m_Sleep.Release();
    }
}

void TerminalWorker::_Run()
{
    char msg[MAX_MSG_SIZE];
    int pos = 0;

    // some very minimal security...
    if (strcmp(ManagerParams::TERMINAL_PASSWORD, "")) {
	string passwd("");

	fd_set d;
	FD_ZERO(&d);
	FD_SET(m_Sock.GetSocket(), &d);

	int ret;
	TimeVal start, now, to;
	OS::GetCurrentTime(&start);

	m_Sock << "Password: ";
	do {
	    OS::GetCurrentTime(&now);
	    sint64 left = (start + 15000) - now;
	    if (left < 0) break;
	    to.tv_sec  = left/MSEC_IN_SEC;
	    to.tv_usec = 0;
	    while (true) {
		ret = select(m_Sock.GetSocket()+1, &d, 0, 0, &to);
		if (ret >= 0 || errno != EINTR && errno != EAGAIN)
		    break;
	    }
	    if (!FD_ISSET(m_Sock.GetSocket(), &d))
		break;
	    
	    ret = read( m_Sock.GetSocket(), msg, MAX_MSG_SIZE-1 );
	    if (ret == 0) break;
	    msg[ret] = '\0';
	    passwd.append( msg );
	} while (( passwd.length() < 1 || 
		   passwd[passwd.length()-1] != '\n' ) &&
		 ( passwd.length() < MAX_MSG_SIZE ));
	
	while ( passwd.length() > 0 &&
		(passwd[passwd.length()-1] == '\n' ||
		 passwd[passwd.length()-1] == '\r') ) {
	    passwd = passwd.substr(0,passwd.length()-1);
	}
	if (strcmp(passwd.c_str(), ManagerParams::TERMINAL_PASSWORD)) {
	    m_Sock << "Access denied.\n";
	    m_Sock.Close();
	    return;
	}
    }
    
    m_Sock << "Colyseus Terminal for "
	   << g_LocalSID.ToString() << "." << "\n";

    while (1) {
	int ret = read(m_Sock.GetSocket(), &msg[pos], MAX_MSG_SIZE-pos);
	if (ret == 0) {
	    DBG << "client closed Socket " << m_Sock.GetSocket() << "\n";
	    // EOF
	    break;
	} else if (ret < 0) {
	    if (errno == EINTR || errno == EAGAIN) {
		// interrupted, try again...
		continue;
	    } else {
		// some other error, don't really want to recover
		WARN << "read error on client Socket "
		     << m_Sock.GetSocket() << ": " 
		     << strerror(errno) << "\n";
		break;
	    }
	} else {
	    bool quit = false;

	    pos += ret;
	    while (1) {
		char *str = extract_str(msg, &pos);
		if (str == NULL)
		    break;
		
		DBG << "req: " << str << "\n";
		
		char **args = extract_args(str);
		char *res;

		m_Sock << "==BEGIN_RESP" << "\n";
		if (! args[0] || args[0] == '\0') {
		    m_Sock << "ERROR: no command provided" << "\n";
		} else {
		    bool done = false;
					
		    for (int i=0; i < (int)g_Handlers.size(); i++) {
			if ( !strcasecmp(args[0], g_Handlers[i]->cmd) ||
			     !strcasecmp(args[0], g_Handlers[i]->abbrev) ) {
							
			    stringstream out;

			    g_Manager->Lock();
			    g_Handlers[i]->handler(out, args);
			    g_Manager->Unlock();

			    m_Sock << out.str();
							
			    done = true;
			    break;
			}
		    }
					
		    if (!done) {
			m_Sock << "ERROR: unknown command: " 
			       << args[0] << "\n";
		    }
		}
		m_Sock << "==END_RESP" << "\n";
				
		// special case for exiting
		if (args[0] &&
		    (!strcasecmp(args[0], "QUIT") ||
		     !strcasecmp(args[0], "Q"))) {
		    quit = true;
		}
	    }

	    if (quit) break;
	}
    }

    m_Sock.Close();
}


///////////////////////////////////////////////////////////////////////////////

#define NOT_ENOUGH_ARGS(x) out << "ERROR: need " << x << " arguments" << "\n"
#define DOES_NOT_EXIST(x) out << "ERROR: object does not exist" << "\n";

void TerminalWorker::handleVersion(ostream& out, char **args) {
    out << "libcolyseus " << g_ColyseusVersion 
	<< " (" << g_ColyseusBuildTime << ")" << "\n";
    return;
}

void TerminalWorker::handleHelp(ostream& out, char **args) {
    out << "Definitions:" << "\n";
    out << "  <ID>       = a name you gave an object or its GUID" << "\n";
    out << "  <attr_key> = a string (no spaces) for an attribute" << "\n";
    out << "  <attr_val> = a string <type>:<value> string for value" << "\n";
    out << "  <type>     = {int,char,byte,real,string,guid,ID}" << "\n";
    out << "  <guid>     = <ip>:<port>:<localOID>, where <ip> is in dotted-tail format" << "\n";
    out << "  <op>       = {==,!=,<,<=,>,>=,*}, where * means 'any'" << "\n";

    out << "\n" << "Commands:" << "\n";
	
    for (int i=0; i < (int)g_Handlers.size(); i++) {
	out << "  [" << g_Handlers[i]->abbrev << "] "
	    << g_Handlers[i]->cmd << " " 
	    << g_Handlers[i]->usage  << "\n";
	out << "    " << g_Handlers[i]->help << "\n";
    }
}

void TerminalWorker::handleMigrate(ostream& out, char **args)
{
    if (args[1] == NULL || args[2] == NULL) {
	NOT_ENOUGH_ARGS(2);
	return;
    }

    GObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    if (obj->IsReplica()) {
	out << "ERROR: can't migrate a replica: " 
	    << obj->GetGUID() << "\n";
	return;
    }

    sid_t sid = parse_sid(args[2]);
    if (sid == g_Manager->GetSID()) {
	out << "ERROR: can't migrate object to ourselves" << "\n";
	return;
    }
    if (sid == SID_NONE) {
	out << "ERROR: bad sid value: " << args[2] << "\n";
	return;
    }

    g_Manager->Migrate(obj, sid);
	
    out << "OK: started migration for guid=" << obj->GetGUID() 
	<< " to " << sid << "\n";
    return;
}

void TerminalWorker::handleShow(ostream& out, char **args)
{
    if (args[1] == NULL) {
	NOT_ENOUGH_ARGS(1);
	return;
    }

    GObject *obj = find_obj(args[1]);
    if (obj == NULL) {
	DOES_NOT_EXIST(args[1]);
	return;
    }

    out << "OK: " << obj << "\n";
}

void TerminalWorker::handleStore(ostream& out, char **args)
{
    GObject *obj;
    g_Manager->GetObjectStore()->Begin();
    while ( (obj = g_Manager->GetObjectStore()->Next()) != NULL ) {
	out << "OK: " << obj << "\n";
    }
}

void TerminalWorker::handlePending(ostream& out, char **args)
{
    GObject *obj;
    g_Manager->GetPendingStore()->Begin();
    while ( (obj = g_Manager->GetPendingStore()->Next()) != NULL ) {
	out << "OK: " << obj << "\n";
    }
}

void TerminalWorker::handleGraph(ostream& out, char **args)
{
    const char *output = g_Manager->DumpInterestGraph();
    out << output;
    return;
}

void TerminalWorker::handleLoad(ostream& out, char **args)
{
    TimeVal now;
    OS::GetCurrentTime(&now);

    out << g_Manager->GetLoad(now) << "\n";
    return;
}

void TerminalWorker::handleLoadMap(ostream& out, char **args)
{
    const char *output = g_Manager->DumpNodeLoadMap();
    out << output;
    return;
}

void TerminalWorker::handleObjectInfo(ostream& out, char **args)
{
    const char *output = g_Manager->DumpObjectInfo();
    out << output;
    return;
}

#define TEMP_DOT "/tmp/TestApp.graphvis.tmp.dot"
#define TEMP_PS  "/tmp/TestApp.graphvis.tmp.ps"

void TerminalWorker::handleGraphVis(ostream& out, char **args)
{
    const char *output = g_Manager->DumpInterestGraphToGraphViz();

    FILE *tmp = fopen(TEMP_DOT, "w");
    if (!tmp) {
	out << "ERROR: couldn't open temp graph vis file: " 
	    << TEMP_DOT << "\n";
    }
    fprintf(tmp, output);
    fclose(tmp);
    out << "OK: generating postscript..." << "\n";
    system("neato -Tps " TEMP_DOT " > " TEMP_PS);
    out << "OK: displaying; close window to continue..." << "\n";
    system("gv " TEMP_PS);

    return;
}

void TerminalWorker::handleBenchmark(ostream& out, char **args)
{
    Benchmark::print(out);
    return;
}

void TerminalWorker::handleRotate(ostream& out, char **args)
{
    if (!g_MeasurementParams.enabled)
	out << "ERROR: measurement is not enabled" << endl;
    else {
	if (ROTATE_ALL()) {
	    out << "OK: done" << endl;
	} else {
	    out << "ERROR: failed!" << endl;
	}
    }
    return;
}

void TerminalWorker::handleReplicaConn(ostream& out, char **args)
{
    const char *output = g_Manager->DumpReplicaConnections();
    out << output;
    return;
}

void TerminalWorker::handleDelGUIDs(ostream& out, char **args)
{
    const char *output = g_Manager->DumpDeletedGUIDs();
    out << output;
    return;
}

void TerminalWorker::handleSubList(ostream& out, char **args)
{
    g_Manager->m_PubSubRouter->PrintSubscriptionList(out);
    return;
}

void TerminalWorker::handlePubList(ostream& out, char **args)
{
    g_Manager->m_PubSubRouter->PrintPublicationList(out);
    return;
}

void TerminalWorker::handleQuit(ostream& out, char **args)
{
    out << "OK: closing connection\n";
    return;
}

///////////////////////////////////////////////////////////////////////////////

GObject *TerminalWorker::find_obj(char *name)
{
    string key(name);
    guid_t guid;

    guid = parse_guid(name);
    GObject *obj = g_Manager->GetObjectStore()->Find(guid);
    return obj;
}

/*
  int TerminalWorker::parse_val(ostream& out, Value *val, char *str)
  {
  char *type   = strtok(str, ":");
  char *value  = strtok(NULL, " "); // until end

  if (type == NULL || value == NULL) {
  out << "ERROR: invalid value string: " << str << "\n";
  return 0;
  }

  if ( !strcasecmp(type, "i") ||
  !strcasecmp(type, "int") ||
  !strcasecmp(type, "integer") ) {

  *val = Value( (uint32)atoi(value) );

  } else if ( !strcasecmp(type, "c") ||
  !strcasecmp(type, "char") ) {

  *val = Value( (char)value[0] );

  } else if ( !strcasecmp(type, "b") ||
  !strcasecmp(type, "byte") ) {

  *val = Value( (char)atoi(value) );

  } else if ( !strcasecmp(type, "f") ||
  !strcasecmp(type, "float") ||
  !strcasecmp(type, "real") ) {

  *val = Value( (float)atof(value) );

  } else if ( !strcasecmp(type, "s") ||
  !strcasecmp(type, "string") ||
  !strcasecmp(type, "str") ) {

  string s(value);
  *val = Value( s );

  } else if ( !strcasecmp(type, "g") ||
  !strcasecmp(type, "guid") ) {

  guid_t guid = parse_guid(value);
  if (guid == GUID_NONE) {
  out << "ERROR: invalid guid=" << value << "\n";
  return 0;
  }
  if (g_Manager->GetObjectStore()->Find(guid) == NULL) {
  out << "ERROR: guid " << value << " wasn't found in ObjectStore"
  << "\n";
  return 0;
  }
  *val = Value( guid );

  } else {
  out << "ERROR: invalid value type: " << type << "\n";
  return 0;
  }

  return 1;
  }
*/

guid_t TerminalWorker::parse_guid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    char *localOID = strtok(NULL, ":");
    if (ip == NULL || port == NULL || localOID == NULL) {
	return GUID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    uint32 _localOID = (uint32)atoi(localOID);
    if (_ip == INADDR_NONE || _port == 0) {
	return GUID_NONE;
    }

    guid_t guid(_ip, _port, _localOID);

    return guid;
}

sid_t TerminalWorker::parse_sid(char *str)
{
    char *ip   = strtok(str, ":");
    char *port = strtok(NULL, ":");
    if (ip == NULL || port == NULL) {
	return SID_NONE;
    }
    uint32 _ip   = inet_addr(ip);
    uint16 _port = (uint16)atoi(port);
    if (_ip == INADDR_NONE || _port == 0) {
	return SID_NONE;
    }

    sid_t sid(_ip, _port);

    return sid;
}

///////////////////////////////////////////////////////////////////////////////

static char str_buf[512];
static char *arg_buf[512];

char *extract_str(char *buf, int *len) 
{
    int i;
    bool found = 0;

    // parse one command terminated by a newline into the static buf
    for (i=0; i<*len; i++) {
	str_buf[i] = buf[i];
	if (str_buf[i] == '\n') {
	    str_buf[i] = '\0';
	    // if client writes \r\n, then need to get rid of two chars
	    if (str_buf[i-1] == '\r')
		str_buf[i-1] = '\0';
	    found = 1;
	    i++;
	    break;
	}
    }

    // if there is stuff left in the buffer, move it to the front
    int j = 0;
    for ( ; i<*len; i++, j++) {
	buf[j] = buf[i];
    }
    *len = j;
    
    if (found) {
	return str_buf;
    } else {
	return NULL;
    }
}

char **extract_args(char *buf) {
    int i;
    char *last = buf;
    int index = 0;

    // separate command string into an array of string args
    for (i=0; i<512; i++) {
	bool end = (buf[i] == '\0');
		
	if (end || isspace(buf[i])) {
	    buf[i] = '\0';
	    if (last < buf+i) {
		arg_buf[index] = last;
		++index;
		if (index > 512)
		    break;
	    }
	    last = buf + i + 1;
	}
		
	if (end) break;
    }
    arg_buf[index] = NULL;
	
    return arg_buf;
}

///////////////////////////////////////////////////////////////////////////////
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
