////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  ReplicaConnection.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

 ***************************************************************************/

#include <om/ReplicaConnection.h>
#include <om/Manager.h>
#include <om/GInterest.h>
#include <om/OMMessage.h>
#include <om/TestLogs.h>
#include <mercury/ObjectLogs.h>
#include <util/Utils.h>

///////////////////////////////////////////////////////////////////////////////
///// REPLICA CONNECTIONS

void ReplicaConnectionSender::RegisterGUID(GUID guid, 
	bool isDelete,
	bool isNew,
	bool isProactive,
	GInterest *reg) 
{
    if (m_Deltas.find(guid) == m_Deltas.end()) {
	m_Deltas[guid] = DeltaInfo();
    }
    if (isDelete) {
	m_Deltas[guid].isDelete = true;
	m_Deltas[guid].isNew    = false;
    }
    m_Deltas[guid].Refresh();
    if (isNew)
	m_Deltas[guid].isNew = true;
    if (isProactive)
	m_Deltas[guid].isProactive = true;

    ///// MEASUREMENT
    if (g_MeasurementParams.enabled) {
	if (reg && reg->newRegister && reg->nonce != 0) {
	    ASSERT(isNew);
	    DiscoveryLatEntry ent(DiscoveryLatEntry::REG_RECV, 
		    1, reg->nonce);
	    LOG(DiscoveryLatLog, ent, reg->recvTime); // use recv time
	    m_Deltas[guid].nonce = reg->nonce;
	}
    }
    ///// MEASUREMENT
}

void ReplicaConnectionSender::UnRegisterGUID(GUID guid) {
    m_Deltas.erase(guid);
}

void ReplicaConnectionSender::GarbageCollectRegistrations() {
    for (DeltaMapIter it = m_Deltas.begin(); it != m_Deltas.end(); ) {
	if (it->second.TimedOut()) {
	    DeltaMapIter oit = it;
	    oit++;
	    m_Deltas.erase(it->first);
	    it = oit;
	} else {
	    it++;
	}
    }
}

MsgAppUpdate *ReplicaConnectionSender::CreateUpdate(bool onlyNew) 
{
    TimeVal now;
    OS::GetCurrentTime(&now);

    bool doSend = true;

    // MEASUREMENT
    {
	m_Stats.window += m_Outstanding.size();
	m_Stats.samples++;
	m_Stats.Record(now);
    }

    CheckTimeout();

    // If we are not maintaining any replicas on the remote node, then
    // don't transmit an update.
    // We don't have to do retransmissions at this point because this
    // condition implies the remote node is not interested in ANY objects
    // we are maintaining; hence any updates lost to it are irrelevant
    // (if it becomes interested in them again, we will start delivering
    // them in order).
    if (m_Deltas.size() == 0) {
	return NULL;
    }

    // Maintain an appropriate send rate; xxx how to handle onlyNew packets?
    if (!onlyNew && m_LastSendTime + m_SendInterval > now) {
	doSend = false;
    }
    if (!onlyNew) NOTE(UPDATE_SEND_INTERVAL, m_SendInterval);

    uint32 cumulative  = m_LastSeqNoDropped == 0 ? 1 :
	m_LastSeqNoSent + 2 - m_LastSeqNoDropped;

    if (onlyNew && cumulative > 1) {
	// if dropped packets, can't send an "only-new" packet because
	// the receiver needs a "full" packet to patch the missing stuff first
	//
	// XXX FIXME: This shouldn't be necessary since the receiver *can*
	// incorporate new objects regardless of changes to other objects
	return NULL;
    }

    // XXX TODO: Should drop the connection actually... too many cumulative
    // drops or outstanding packets... means other side is down or their
    // connection state was corrupted...
    if (m_Outstanding.size() > DELTA_WINDOW_SIZE ||
	    cumulative > 255) {
	WARN << "(" << m_SID << ") too many outstanding or dropped packets;"
	    << " outstanding=" << m_Outstanding.size() 
	    << " cumulative=" << cumulative
	    << " reseting connection next round: " 
	    << this << endl;
	ResetConnection();
	cumulative = 1;
    }

    MsgAppUpdate *msg = new MsgAppUpdate(onlyNew);
    ObjectStore *store = m_Manager->GetGame()->GetObjectStore();

    uint32 minsize = msg->GetLength();
    uint32 size = minsize;

    // XXX TODO: we should apply some sort of priorities for updates like TNL
    // but right now just use random ordering to ensure every object is 
    // serviced eventually...
    GUIDVec todo;
    todo.reserve(m_Deltas.size());
    for (DeltaMapIter it = m_Deltas.begin(); it != m_Deltas.end(); it++)
	todo.push_back(it->first);
    permute<GUID>(todo);

    // statistics
    uint32 n = 0, d = 0, u = 0;

    // Generate the deltas for primaries
    for (uint32 i=0; i<todo.size(); i++) {
	DeltaMapIter it = m_Deltas.find(todo[i]);
	GUID it_guid = it->first;

	ASSERT(it != m_Deltas.end());

	if (onlyNew && !it->second.isNew) {
	    continue;
	}

	bool dirty = false;
	GUpdate *update = new GUpdate(false, it_guid);

	GObject *obj = store->Find(it_guid);

	///// MEASUREMENT
	if (g_MeasurementParams.enabled) {
	    if (it->second.nonce != 0) {
		ASSERT(it->second.isNew || it->second.isDelete);
		OBJECT_LOG_DO( update->nonce = it->second.nonce );
	    }
	}
	///// MEASUREMENT

	if (obj == NULL) {
	    if ( !m_Manager->IsDeleted(it_guid)) {
		// XXX Currently this happens for items which are dropped
		// (but are temporary) since their pubs last 30 seconds
		// (longer than their lifetime)... not sure why we wouldn't
		// have recorxded their deletion record for that long though
		DB(1) << GetSID() << " is registered to an object that "
		    << "we don't have and we didn't delete recently..."
		    << endl;
		continue;
	    }

	    update->isDelete = true;
	    dirty = true;

	    d++;

	    // if the deletion is dropped, then we will reinstantiate this
	    // delta record from the saved message. XXX: careful, `it' will be invalid if "dirty = true" and obj == NULL
	    //
	    m_Deltas.erase(it);
	} else {
	    ASSERT(! obj->IsReplica() );
	    ASSERT(obj->GetSID() == m_Manager->GetSID());

	    // delta info of dropped packets
	    DeltaInfo delta = it->second;
	    // new objects need to have full delta masks
	    if (delta.isNew) {
		//DB(0) << "is new delta: " << it_guid << endl;
		delta.mask = obj->GetInitDeltaMask();

		n++;
	    } else {
		u++;
	    }
	    // latest delta mask
	    delta.mask.Merge( obj->GetDeltaMask() );

	    /*
	       if (obj->IsDirty()) {
	       DB(0) << "object changed: " << it_guid << endl;
	       }
	     */

	    ASSERT(!delta.isDelete);

	    // dirty?
	    if ( delta.IsDirty() ) {
		Packet *buf = new Packet(GUpdate::MAX_UPDATE_SIZE);
		//START(ReplicaConnection::PackUpdate);
		obj->PackUpdate(buf, delta.mask); // pack the delta
		ASSERT(buf->GetUsed () > 0);
		//STOP(ReplicaConnection::PackUpdate);
		update->delta = buf;
		ASSERT(update->delta);
		update->isNew = delta.isNew;
		update->isProactive = delta.isProactive;
		update->mask  = delta.mask;
		dirty = true;
	    }
	}

	// XXX: we should really update the cost estimate even if no one
	// else is interested in the object also, but i don't know of an
	// accurate way to do that (taking into account how often it will
	// be sent in full or delta encoded, etc.) doing this way only may
	// slow down the estimation process.
	//
	// update the estimate delta sizes of this object (if it exists)
	if (obj) {
	    GObjectInfo *info = static_cast<GObjectInfo *>(obj->GetInfo());
	    ASSERT(info);
	    real32 len;
	    GUID guid = it_guid;

	    if (dirty)
		len = guid.GetLength() + update->GetLength();
	    else
		len = 0;
	    // XXX we are missing some overhead for the msg header + acks
	    // and stuff... but i guess we can maybe ignore that and not
	    // distribute it amoung the object costs?
	    real32 Bps = len/((real32)m_Manager->GetGame()->GetSendInterval()/(real32)MSEC_IN_SEC);

	    // TEST DEBUGGING LOG
	    if (g_MeasurementParams.enabled &&
		    ENABLE_COSTESTIMATEACCURACYLOG) {
		CostEstimateAccuracyEntry e;
		e.guid = guid;
		e.estimate = info->GetCostEstimate()->GetEstimate();
		e.actual = Bps;

		LOG(CostEstimateAccuracyLog, e);
	    }

	    info->GetCostEstimate()->Update( Bps );
	}

	// Merge with total update if possible
	if (dirty && size < (uint32)ManagerParams::MAX_UPDATE_MSG_SIZE) {
	    GUID guid = it_guid;           // dirty could be 'true' because obj was deleted...

	    size += guid.GetLength() + update->GetLength();

	    if (!doSend || size > (uint32)ManagerParams::MAX_UPDATE_MSG_SIZE) {
		// either hit rate limit and have to wait until next
		// frame to send or ...
		// oops! Update got too big! Can't include any more
		// deltas in it, so remember that these things are still
		// dirty and need to be transmitted next time
		/*
		   WARN << "couldn't fit all updates in one update packet to "
		   << GetSID() << " (num objs: " << todo.size() 
		   << "); delaying rest until next time" << endl;
		 */

		if (guid.GetLength() + update->GetLength() > 
			ManagerParams::MAX_UPDATE_MSG_SIZE - minsize) {
		    WARN << "update for obj is too big to fit in a packet! "
			<< "size=" << (guid.GetLength() + update->GetLength())
			<< " maxsize=" << (ManagerParams::MAX_UPDATE_MSG_SIZE - minsize) << " obj=" << obj << ". (you may increase --max-update-size)" << endl;
		}

		if (obj != NULL)
		    it->second.Merge(update);
		delete update;
	    } else {
		if (obj != NULL) { // otherwise we have deleted it->second from m_Deltas - Ashwin [11/01/2006]
		    // clear saved delta info
		    it->second.Clear();
		    OBJECT_LOG_DO( it->second.nonce = 0 );
		}

		// XXX if this message is lost, then we don't record
		// the retransmission... OK? Means we ignore loss
		msg->updates[it_guid] = update;
	    }
	} else {
	    delete update;
	}
    }

    // if an "is-new" message and nothing inside or we hit the rate limit, 
    // then don't send it
    if (onlyNew && msg->updates.size() == 0 || !doSend) {
	delete msg;
	return NULL;
    }

    if (m_LastSeqNoSent == 0) {
	msg->SetResetConnection(true);
    }

    msg->epoch       = m_Epoch;
    msg->updateSeqno = ++m_LastSeqNoSent;
    msg->cumulative  = cumulative;
    msg->timeout     = now + UPDATE_TIMEOUT; 
    m_Outstanding.push_back(msg);
    NOTE(UPDATE_OUTSTANDING, m_Outstanding.size());

    DB(5) << "Created update(" << m_SID << ") size="
	<< msg->GetLength() << ": " << msg << endl;

    // MEASUREMENT
    if (!onlyNew) {
	RepConnSendEntry e(m_SID, m_LastSendTime == TIME_NONE ? 0 : 
		now - m_LastSendTime, size, n, d, u);
	LOG(RepConnSendLog, e);
	m_LastSendTime = now;

	m_Stats.sent++;
    } else {
	m_Stats.async++;
    }

#ifdef UPDATE_TIMESTAMPS
    /// xxx debug
    msg->timestamp = now;
#endif
    return msg;
}

void ReplicaConnectionSender::ResetConnection()
{
    // reset seqno sequence
    m_LastSeqNoSent    = 0;
    m_LastSeqNoAcked   = 0;
    m_LastSeqNoDropped = 0;
    m_Epoch = m_Epoch++; //(m_Epoch == 1 ? 0 : 1);

    ObjectStore *store = m_Manager->GetObjectStore();

    // set all deltas to FULL
    for (DeltaMapIter it = m_Deltas.begin(); it != m_Deltas.end(); it++) {
	if (! it->second.isDelete) {
	    GObject *obj = store->Find(it->first);

	    if (obj) {
		it->second.isNew = true;
		it->second.mask  = obj->GetInitDeltaMask();
	    } else {
		ASSERT(m_Manager->IsDeleted(it->first));

		// we must have deleted the object
		it->second.isNew = false;
		it->second.isDelete = true;
	    }
	}
    }

    // clear outstanding packets
    for (DeltaWindowIter it = m_Outstanding.begin();
	    it != m_Outstanding.end(); it++) {
	delete *it;
    }
    m_Outstanding.clear();
}

void ReplicaConnectionSender::HandleUpdateAck(MsgAppUpdateAck *msg)
{
    uint32 ackSeqno = msg->ackSeqno;

    // XXX TODO: the logic in this function assumes that we always
    // want ordered delivery of updates even in the face of dropped
    // packets. In many cases, this is not likely to be true --
    // e.g., many operations will be commutative so even if one
    // update is dropped, the next one can still be applied while
    // maintaining consistent state (what we are guarding against
    // is changes in 2 fields that might be inconsistent if one is
    // applied without the other). To facilitate this, we need a
    // way to push this back to the GObject interface so that the
    // update can specify which potions of the update must be
    // "ordered" and which should be "as-close-to-up-to-date-as-possible"

    if (msg->epoch != m_Epoch) {
	WARN << "Got ack for epoch " << (int)msg->epoch 
	    << " but we're in epoch " << (int)m_Epoch 
	    << ": " << this << endl;
	return;
    }

    TimeVal now;
    OS::GetCurrentTime(&now);

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    NOTE(HANDLE_UPDATE_ACK_TIME, now - msg->timestamp);
#endif

    // XXX TODO: handle 0-wrapping -- shouldn't happen in any real
    // circumstance since we assume sequence numbers start from 0
    // and the delta rate is approx. 20fps. Hence 31-bits will last
    // several years. Eventually we want to be able to start from
    // non-0 sequence numbers for security and robustness.
    if (ackSeqno < m_LastSeqNoAcked) {
	WARN << GetSID() << " ack'd a delta " << ackSeqno 
	    << " that is before the last ack it sent us " 
	    << m_LastSeqNoAcked << endl;
	return;
    } else if (ackSeqno == m_LastSeqNoAcked) {
	// MEASUREMENT
	{
	    m_Stats.lost++;
	}

	NOTE(UPDATE_DUPACK, 1);

	HandleDrop(ackSeqno);

	/*
	   m_LastSeqNoDropped = ackSeqno+1;

	// this means the next delta we send should be cumulative from
	// m_LastSeqNoDropped until the current state
	while (m_Outstanding.size() > 0) {
	MsgAppUpdate *msg = m_Outstanding.front();
	m_Outstanding.pop_front();
	if (msg->updateSeqno > ackSeqno) {
	for (GUpdateMapIter it = msg->updates.begin();
	it != msg->updates.end(); it++) {
	if (m_Deltas.find(it->first) == m_Deltas.end()) {
	// this implies there are dropped updates for
	// this replica, but the remote node is no longer
	// interested in it, so we don't need to bother trying
	// to deliver them, *unless* this was a deletion 
	// record, in which case it probably was removed by us,
	// and we need to retransmit the deletion. So 
	// reinstantiate the delta record for it.
	if (it->second->isDelete) {
	m_Deltas[it->first] = DeltaInfo();
	m_Deltas[it->first].isDelete = true;
	m_Deltas[it->first].isNew = false;
	}
	} else {
	m_Deltas[it->first].Merge( it->second );
	}
	}
	}
	}
	 */
    } else {
	// we're assuming cumulative acks
	m_LastSeqNoAcked   = ackSeqno;
	// assume everything else got through, until we get a dupack
	m_LastSeqNoDropped = 0;

	while (m_Outstanding.size() > 0) {
	    MsgAppUpdate *umsg = m_Outstanding.front();
	    if (umsg->updateSeqno <= ackSeqno) {
		DBG << "acked(" << m_SID << "): " << umsg << endl;
		m_Outstanding.pop_front();

#ifdef UPDATE_TIMESTAMPS
		// xxx debug
		NOTE(UPDATE_ACKED_TIME, now - umsg->timestamp);
#endif

		// 7/2/2004: unnecessary, we remove these immediately after
		// we send the deletion record
		//for (GUpdateMapIter it = umsg->updates.begin();
		//it != umsg->updates.end(); it++) {
		// once the remote node ack's a delete, we can remove
		// its registration for it
		//if (it->second->isDelete) {
		//DB(0) << "present? " 
		//	  << (m_Deltas.find(it->first) != m_Deltas.end())
		//	  << " deleted: " << it->first << endl;
		//m_Deltas.erase(it->first);
		//}
		//}

		delete umsg;
	    } else {
		break;
	    }
	}

	HandleRecv();
    }
}

void ReplicaConnectionSender::CheckTimeout()
{
    TimeVal now;
    OS::GetCurrentTime(&now);

    bool timedout = false;
    for (DeltaWindowIter it = m_Outstanding.begin(); 
	    it != m_Outstanding.end(); it++) {

	if ((*it)->timeout < now) {
	    timedout = true;
	    break;
	}
    }

    if (timedout) {
	//WARN << "timed out: " << this << endl;
	HandleDrop(0);
    }
}

void ReplicaConnectionSender::HandleDrop(uint32 dupAck)
{
    // if we get multiple dupacks for the same packet, only count
    // one of them as a dropped packet...
    if (!dupAck || dupAck != m_LastSeqNoDropped) {
	// reduce send rate multiplicatively
	m_SendInterval = MIN(m_SendInterval*2, MAX_SEND_INTERVAL);
    }

    m_LastSeqNoDropped = m_LastSeqNoAcked+1;

    // this means the next delta we send should be cumulative from
    // m_LastSeqNoDropped until the current state
    while (m_Outstanding.size() > 0) {
	MsgAppUpdate *msg = m_Outstanding.front();
	m_Outstanding.pop_front();
	if (msg->updateSeqno > m_LastSeqNoAcked) {
	    for (GUpdateMapIter it = msg->updates.begin();
		    it != msg->updates.end(); it++) {
		if (m_Deltas.find(it->first) == m_Deltas.end()) {
		    // this implies there are dropped updates for
		    // this replica, but the remote node is no longer
		    // interested in it, so we don't need to bother trying
		    // to deliver them, *unless* this was a deletion 
		    // record, in which case it probably was removed by us,
		    // and we need to retransmit the deletion. So 
		    // reinstantiate the delta record for it.
		    if (it->second->isDelete) {
			m_Deltas[it->first] = DeltaInfo();
			m_Deltas[it->first].isDelete = true;
			m_Deltas[it->first].isNew = false;
		    }
		} else {
		    m_Deltas[it->first].Merge( it->second );
		}
	    }
	}
	delete msg;
    }
}

void ReplicaConnectionSender::HandleRecv()
{
    // increase send rate additively
    if (m_SendInterval > SEND_INTERVAL_BASE)
	m_SendInterval = MAX(m_SendInterval-SEND_INTERVAL_BASE,
		SEND_INTERVAL_BASE);
}

ostream& operator<<(ostream& out, ReplicaConnectionSender *conn)
{
    out << "(RepConnSender sid=" << conn->GetSID()
	<< " lastAcked=" << conn->m_LastSeqNoAcked
	<< " lastDropped=" << conn->m_LastSeqNoDropped
	<< " lastSent=" << conn->m_LastSeqNoSent
	<< " epoch=" << (int)conn->m_Epoch
	<< " deltas=[";
    for (DeltaMapIter it = conn->GetDeltaMap()->begin();
	    it != conn->GetDeltaMap()->end(); it++) {
	DeltaInfo info = it->second;
	if (it != conn->GetDeltaMap()->begin()) out << ",";
	out << "{" << it->first << "," << &info << "}";
    }
    out << "] numOutstanding=" << conn->m_Outstanding.size() << ")";
    return out;
}

///////////////////////////////////////////////////////////////////////////////

static bool EpochIsLess(byte a, byte b)
{
    // under the assumption that a and b always differ by at most 127,
    // with wrap around, the bigger value is the one "in front" within
    // the 127-unit sized window that covers both values
    if (a > b) {
	return a-b > 127;
    } else if (a < b) {
	return b-a < 128;
    } else {
	return false;
    }
}

void ReplicaConnectionReceiver::HandleUpdate(MsgAppUpdate *msg)
{
    // xxx debug
    TimeVal now;
    OS::GetCurrentTime(&now);
#ifdef UPDATE_TIMESTAMPS
    NOTE(HANDLE_UPDATE_TIME, now - msg->timestamp);
#endif

    bool async = msg->onlyNew;

    // MEASUREMENT
    {
	m_Stats.Record(now);

	if (async)
	    m_Stats.async++;
	else
	    m_Stats.recv++;
    }

    ObjectStore *store = m_Manager->GetGame()->GetObjectStore();
    ObjectStore *pending = m_Manager->GetGame()->GetPendingStore();

    SID    sid         = msg->GetSender();
    byte   epoch       = msg->epoch;
    uint32 updateSeqno = msg->updateSeqno;
    int    cumulative  = (int)msg->cumulative;

    DBG << "called" << endl;

    ASSERT(sid != SID_NONE);
    ASSERT(sid != m_Manager->GetSID());

    // did the sender reset our connection?
    if (msg->IsResetConnection()) {
	if (epoch != 0)
	    WARN << "(" << m_SID 
		<< ") reset the connection with different epoch" 
		<< " epoch=" << (int)epoch << " updateSeqno=" 
		<< updateSeqno << ": " << this << endl;
	ResetConnection(epoch, updateSeqno);
    } else if (EpochIsLess(m_Epoch, epoch)) {
	WARN << "(" << m_SID << ") reset the connection with different epoch" 
	    << " epoch=" << (int)epoch << " updateSeqno=" 
	    << updateSeqno << ": " << this << endl;
	ResetConnection(epoch, updateSeqno);
    } else if (EpochIsLess(epoch, m_Epoch)) {
	WARN << "(" << m_SID << ") update from old epoch=" << (int)epoch
	    << "; dropping: " << this << endl;
	return;
    }

    // XXX TODO: need to handle cases where we don't necessarily need
    // in order delivery of updates (see above)

    // XXX TODO: handle seqno wrap-around
    if (updateSeqno <= m_LastSeqNoAck) {
	// duplicate or out of order packet!
	WARN << "Got update with seqno " << updateSeqno 
	    << " from " << msg->GetSender() 
	    << " which is <= last seqno acked " << m_LastSeqNoAck << endl;

	if (async)
	    m_Stats.async_old++;
	else
	    m_Stats.old++;

	return;
    } else if (updateSeqno - cumulative > m_LastSeqNoAck) {
	// missing packet in the middle!
	DB(1) << "Got update with seqno " << updateSeqno
	    << ", cumulative for " << cumulative << " packets "
	    << " from " << msg->GetSender() 
	    << "; missing packets since last seqno acked was "
	    << m_LastSeqNoAck << endl;

	if (async)
	    m_Stats.async_missing++;
	else
	    m_Stats.missing++;

	return;
    } else {
	m_LastSeqNoAck = updateSeqno;

	SIDMap unresolved;

	// MEASUREMENT
	uint32 n = 0, d = 0, u = 0;

	// ok, apply the delta
	for (GUpdateMapIter it = msg->updates.begin();
		it != msg->updates.end(); it++) {

	    GUpdate *update = it->second;
	    GObject *obj    = store->Find(update->guid);
	    if (obj == NULL)
		obj = pending->Find(update->guid);

	    if (obj != NULL && 
		    ( !obj->IsReplica() ||
		      // we might own the object but it is still in the
		      // process of migrating so it is marked as replica
		      m_Manager->IsIncomingMigration(obj->GetGUID()) ) ||
		    // we deleted this object! shouldn't be getting updates for it!
		    m_Manager->IsDeleted(update->guid) ) {
		WARN << "got delta to object we own; discarding: ";
		if (obj != NULL)
		    cerr << obj->GetGUID();
		cerr << endl;
		continue;
	    }

	    // this update is "new" to us if and only if we don't yet
	    // have a copy of the object. This flag is only for logging
	    // (we only want to log discoveries for "new" objects)
	    bool wasNew = obj == NULL;

	    // MEASUREMENT
	    if (update->isNew)
		n++;
	    else if (update->isDelete)
		d++;
	    else
		u++;

	    // if we already have a copy but get a "new" update, then
	    // just apply the "new" update to the existing object...
	    // Jeff: I want to avoid adding objects that we already
	    // have in the object store... that tends to lead to
	    // consistency issues.
	    if (update->isNew && obj == NULL) {
		ASSERT(update->delta != NULL);

		DBG << "Got isNew delta: " << update << endl;

		GObjectInfo *info;
		GObjectInfoMapIter infop = 
		    m_Manager->GetObjectInfo()->find(update->guid);
		if (infop == m_Manager->GetObjectInfo()->end()) {
		    if (!update->isProactive) {
			// the other guy must have not gotten our unregister
			// message yet...
			DB(1) << "Got a new non-proactive update for we don't "
			    << "have object info for -- that means we "
			    << "aren't interested!" << endl;
			continue;
		    } else {
			// other guy requested that we proactively replicate
			// this object! So do it!
			info = m_Manager->AllocGObjectInfo(update->guid,
				sid, true);
		    }
		} else {
		    info = infop->second;
		}

		DB(1) << "constructing new obj from update: "
		    << update->guid << endl;

		if (g_MeasurementParams.enabled) {
		    ReplicaLifetimeEntry e;
		    e.guid = update->guid;
		    e.type = (char)ReplicaLifetimeEntry::DCREATE;
		    LOG(ReplicaLifetimeLog, e);
		}

		obj = m_Manager->GetGame()->Construct(info,
			update->delta, 
			update->mask,
			&unresolved);
		ASSERT(obj != NULL);
		ASSERT(obj->GetSID() != m_Manager->GetSID());
		ASSERT(obj->IsReplica());
		obj->SetSID(sid);

		{ // just a consistency check
		    GObjectInfoMapIter p = 
			m_Manager->GetObjectInfo()->find(update->guid);
		    ASSERT(p->second == obj->GetInfo());
		    /* Jeff: no need except as a consistency check
		       if (p != m_Manager->m_ObjectInfo.end()) {
		       if (sid != p->second->GetSID())
		       p->second->SetSID( sid );
		       }
		     */
		}

		m_Manager->GetGame()->GetPendingStore()->_ManagerAdd(obj);

		m_Manager->GetObjectArrivals()->insert(pair<GUID,GObject *>(obj->GetGUID(), obj));
		//m_Manager->m_ObjectArrivals[obj->GetGUID()] = obj;

	    } else if (update->isDelete) {
		ASSERT( !m_Manager->IsDeleted(update->guid) );

		DBG << "Got isDelete delta: " << update << endl;

		GObject *obj = store->Find(update->guid);
		if (obj && !obj->IsReplica()) {
		    WARN << "Got a delete for an object we own!" << endl;
		    continue;
		}
		if (m_Manager->IsIncomingMigration(update->guid)) {
		    WARN << "Got a delete for an object migrating to us!" 
			<< endl;
		    continue;
		}

		if (g_MeasurementParams.enabled) {
		    ReplicaLifetimeEntry e;
		    e.guid = update->guid;
		    e.type = (char)ReplicaLifetimeEntry::EDESTROY;
		    LOG(ReplicaLifetimeLog, e);
		}

		// XXX TODO: this shouldn't be happening so often
		// figure out why...
		if (obj || pending->Find(update->guid)) {
		    // signalling a deletion, inform the application
		    DB(1) << "other side deleted one of our replicas: "
			<< update->guid << endl;
		    m_Manager->GetGame()->Destroy(update->guid);
		}

		// remove our registration for it
		// (this is not strictly required since we will eventually
		// garbage collect the entry from not having received any 
		// pubs)
		//
		// Also, this will be auto-called by destroy->delete above
		GObjectInfoMapIter infop = 
		    m_Manager->GetObjectInfo()->find(update->guid);
		if (infop != m_Manager->GetObjectInfo()->end()) {
		    m_Manager->FreeGObjectInfo(infop->second);
		    // Jeff: already called in free
		    //m_Manager->m_ObjectInfo.erase(infop);
		}

		ASSERT(m_Manager->GetObjectInfo()->find(update->guid) ==
			m_Manager->GetObjectInfo()->end());
		ASSERT(m_Manager->FindObject(update->guid) == NULL);

	    } else {
		// this could happen if we get a really delayed
		// delta, or we freeze processing for a long time....
		// shouldn't crash at least
		if (obj == NULL) {
		    DB(1) << "Got regular delta for obj for which we don't have "
			<< "a copy! ignoring..." << endl;
		    return;
		}

		DBG << "Got regular delta: " << update << endl;

		obj->UnpackUpdate(update->delta, update->mask, &unresolved);
		// ignore unresolved self-pointers
		if (unresolved.find(obj->GetGUID()) != unresolved.end()) {
		    unresolved.erase(obj->GetGUID());
		}
		// remember what changed so the application can know
		obj->GetRemoteDeltaMask().Merge(update->mask);

		ASSERT(obj->GetSID() != m_Manager->GetSID());
		ASSERT(obj->IsReplica());
		obj->SetSID(sid);

		DB_DO(0) { // just a consistency check
		    GObjectInfoMapIter p = 
			m_Manager->GetObjectInfo()->find(update->guid);
		    ASSERT(p != m_Manager->GetObjectInfo()->end());
		    ASSERT(p->second == obj->GetInfo());
		    /* Pointless except as a consistency check
		       if (p != m_Manager->m_ObjectInfo.end()) {
		       if (sid != p->second->GetSID())
		       p->second->SetSID( sid );
		       }
		     */
		}
	    }

	    ///// MEASUREMENT
	    if (g_MeasurementParams.enabled) {
		if (wasNew && update->nonce != 0) {
		    DiscoveryLatEntry ent(DiscoveryLatEntry::UPDATE_DONE, 
			    0, update->nonce);
		    LOG(DiscoveryLatLog, ent);
		}
	    }
	    ///// MEASUREMENT

	    m_Manager->Fetch(&unresolved, update->guid);
	}

	// MEASUREMENT
	if (!msg->onlyNew) {
	    RepConnRecvEntry e(m_SID, m_LastRecvTime == TIME_NONE ? 0 : 
		    now - m_LastRecvTime, msg->GetLength(), 
		    n, d, u);
	    LOG(RepConnRecvLog, e);
	    m_LastRecvTime = now;
	}
    }
}

void ReplicaConnectionReceiver::ResetConnection(byte epoch, uint32 updateSeqno)
{	
    m_Epoch        = epoch;
    m_LastSeqNoAck = updateSeqno-1;
}

ostream& operator<<(ostream& out, ReplicaConnectionReceiver *conn)
{
    out << "(RepConnRecver sid=" << conn->GetSID()
	<< " lastAck=" << conn->m_LastSeqNoAck << " epoch=" 
	<< (int)conn->m_Epoch << ")";
    return out;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
