////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  PubSubManager.cpp

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

***************************************************************************/

#include <mercury/ObjectLogs.h>
#include <om/PubSubManager.h>
#include <om/ManagerParams.h>
#include <om/Manager.h>
#include <mercury/Application.h>
#include <om/OMEvent.h>
#include <om/OMInterest.h>
#include <om/OMPubsubStore.h>

class OMApplication : public DummyApp {
public:
    OMApplication() {}
    void EventInterestMatch (const Event *ev, const Interest *in, const IPEndPoint& subscriber) 
    {
	ASSERTDO (ev != NULL, WARN << "ev is NULL! subscriber=" << subscriber);
	ASSERTDO (in != NULL, WARN << "in is NULL! subscriber=" << subscriber);

	const OMInterest *oin = dynamic_cast<const OMInterest *> (in);
	if (oin == NULL)
	    return;

	OMEvent *oev = (OMEvent *) ev;
	oev->AddTags ((OMInterest *) oin);
    }

    PubsubStore *GetPubsubStore (int hubid) { return new OMPubsubStore (); }
};

PubSubManager::PubSubManager(Manager *manager, WANMercuryNode *router)
{
    m_Manager = manager;
    m_Router = router;
    m_App = new OMApplication ();
    m_Router->RegisterApplication (m_App);
}

PubSubManager::~PubSubManager()
{
    // XXX TODO
}

void PubSubManager::RegisterEvent(OMEvent *pub)
{
    //DBG << "called" << endl;

    EventSet *events;
    EventMMapIter p = m_MyEvents.find(pub->GetGUID());

    if ( p == m_MyEvents.end() ) {
	events = new EventSet();
	m_MyEvents[pub->GetGUID()] = events;
    } else {
	events = p->second;
    }

    events->insert(pub);
    m_EventRegQueue.push_back(pub);
}

void PubSubManager::UnregisterEvent(OMEvent *pub)
{
    //DBG << "called" << endl;

    GUID guid = pub->GetGUID();

    EventMMapIter p = m_MyEvents.find(guid);
    if ( p != m_MyEvents.end() ) {
	EventSet *events = p->second;
	events->erase(pub);
	if (events->size() == 0) {
	    // no more! delete it!
	    delete events;
	    m_MyEvents.erase(p);
	}
	m_EventUnRegQueue.push_back(pub);
    } else {
	// nothing todo
    }
}

void PubSubManager::RegisterInterest(OMInterest *in)
{
    //DBG << "called" << endl;

    // all interests must have unique IDs!
    ASSERT(m_MyInterests.find(in->GetGUID()) == m_MyInterests.end());

    m_MyInterests[in->GetGUID()] = in;
    m_InterestRegQueue.push_back(in);
}

void PubSubManager::UnregisterInterest(OMInterest *in)
{
    //DBG << "called" << endl;

    GUID guid = in->GetGUID();

    InterestMapIter p = m_MyInterests.find(guid);
    if ( p != m_MyInterests.end() ) {
	m_InterestUnRegQueue.push_back(p->second);
	m_MyInterests.erase(p);

	InterestIDMapIter first = m_InterestIDMap.lower_bound(guid);
	InterestIDMapIter last  = m_InterestIDMap.upper_bound(guid);
	m_InterestIDMap.erase(first, last);
    } else {
	// nothing todo
    }
}

void PubSubManager::ChangeInterestAssoc(GUIDMap *assoc)
{
    //DBG << "called" << endl;

    if (assoc->size() < 1) {
	return;
    }

    // remove the old interest -> GUID mappings for GUIDs that
    // had mappings change
    //
    // NO! DO NOT DO THIS. Because those subs are still in the
    // system and we might still get delayed matches for them,
    // in which case we "probably" still want to receive them -- right?
    //
    // They will get cleaned up when this sub actually expires or is
    // unregistered.
    /*
      for (InterestIDMapIter it = m_InterestIDMap.begin();
      it != m_InterestIDMap.end(); ) {
      InterestIDMapIter oit = it;
      oit++;

      // XXX WRONG: this should be a range in the multimap?
      GUIDMapIter p = assoc->find(it->second);
      if (p != assoc->end()) {
      m_InterestIDMap.erase(it);
      }

      it = oit;
      }
    */

    // add the new interest -> GUID mappings
    for (GUIDMapIter it = assoc->begin(); it != assoc->end(); it++) {
	pair<GUID,GUID> p(it->second, it->first);

	InterestIDMapIter start = m_InterestIDMap.lower_bound(it->second);
	InterestIDMapIter end   = m_InterestIDMap.upper_bound(it->second);

	// duplicates => inefficiency, so remove them
	bool haveAlready = false;
	for ( ; start != end; start++) {
	    if (start->second == it->first) {
		haveAlready = true;
		break;
	    }
	}
	if (haveAlready)
	    continue;

	m_InterestIDMap.insert(p);
    }
}

EventSet *PubSubManager::GetCurrEvents(const GUID& guid)
{
    //DBG << "called" << endl;

    //START(SEND::PUB::m_MyEvents.find);
    EventMMapIter evp = m_MyEvents.find(guid);
    //STOP(SEND::PUB::m_MyEvents.find);
    if ( evp == m_MyEvents.end() ) {
	return NULL;
    }

    EventSet *events = evp->second;
    ASSERT(events->size() > 0);
    return events;
}

OMEvent *PubSubManager::GetNextLocalMatch()
{
    //DBG << "called" << endl;

    EventMapIter p = m_LocalMatches.begin();
    if (p == m_LocalMatches.end())
	return NULL;
    OMEvent *ev = p->second;
    m_LocalMatches.erase(p);
    return ev;
}

void PubSubManager::AddCachedEvent(OMEvent *ev)
{
    // XXX: The current assumption is that if an object has multiple
    // pubs in the system, either they all contain enough information
    // to facilitate the matching, or only the newest one should be
    // looked at --- so it is sufficient to keep only 1 pub per object

    if (ev->GetLifeTime() < 50) {
	// delete the event if it is expired, or close to expiring
	delete ev;
	return;
    }

    // LifeTime is decremented to the TimeLeft() each time we fetch
    // the event... this means it gets to live a little longer than
    // it should because its TTL will "stand still" while it is being
    // processed, but assume that is ok...
    ev->Refresh( ev->GetLifeTime() );

    EventMapIter p = m_EventCache.find(ev->GetGUID());
    if (p != m_EventCache.end()) {
	OMEvent *old = p->second;

	if (ev->IsNewer(old)) {
	    // got a newer event for GUID, delete the old one
	    delete old;
	    p->second = ev;
	} else {
	    // not newer than old one, so delete it
	    delete ev;
	}
    } else {
	// don't have a copy, so save it
	m_EventCache[ev->GetGUID()] = ev;
    }
}

void PubSubManager::BeginCachedEventIter()
{
    ASSERT( m_EventCacheIter.size() == 0 );

    // transfer cache to iterator

    while (m_EventCache.size() > 0) {
	EventMapIter p = m_EventCache.begin();
	if (p == m_EventCache.end())
	    ASSERT(0); // Unpossible! :)
	OMEvent *ev = p->second;
	m_EventCache.erase(p);

	if (! ev->TimedOut() ) {
	    // set the lifetime to be the TimeLeft so if we add it
	    // again, it will be given the correct TTL
	    ev->SetLifeTime( ev->TimeLeft() );
	    // event hasn't timed out yet! cool!
	    m_EventCacheIter.push_back( ev );
	} else {
	    // event expired, get rid of it
	    delete ev;
	}
    }
}

OMEvent *PubSubManager::GetNextCachedEvent()
{
    if (m_EventCacheIter.size() == 0)
	return NULL;

    OMEvent *ev = m_EventCacheIter.front();
    m_EventCacheIter.pop_front();

    return ev;
}

///////////////////////////////////////////////////////////////////////////////

void PubSubManager::ExpireStaleItems()
{
    //DBG << "called" << endl;

    TimeVal now;
    OS::GetCurrentTime(&now);

    START(MAINTENANCE::ExpireEvents);
    for (EventMMapIter it = m_MyEvents.begin(); 			
	 it != m_MyEvents.end(); /* !!! */) {
	EventMMapIter oit = it;
	oit++;

	EventSet *events = it->second;
	for (EventSetIter sit = events->begin(); 
	     sit != events->end(); /* !!! */) {
	    EventSetIter osit = sit;
	    osit++;

	    if ((*sit)->TimedOut(now)) {
		delete *sit;
		events->erase(sit);
	    }
	    sit = osit;
	}

	if (events->size() == 0) {
	    // nothing left!
	    delete events;
	    m_MyEvents.erase(it);
	}

	it = oit;
    }
    STOP(MAINTENANCE::ExpireEvents);

    START(MAINTENANCE::ExpireInterests);
    for (InterestMapIter it = m_MyInterests.begin(); 
	 it != m_MyInterests.end(); ) {
	InterestMapIter oit = it;
	oit++;

	if (it->second->TimedOut(now)) {
	    InterestIDMapIter first = m_InterestIDMap.lower_bound(it->first);
	    InterestIDMapIter last  = m_InterestIDMap.upper_bound(it->first);
	    m_InterestIDMap.erase(first, last);

	    //DB(0) << "Expiring local interest: " << it->second << endl;

	    delete it->second;
	    m_MyInterests.erase(it);
	}

	it = oit;
    }
    STOP(MAINTENANCE::ExpireInterests);
}

void PubSubManager::MatchLocalItems()
{
    //DBG << "called" << endl;

    uint32 myevents = 0;

    // TODO: This is very inefficient!!!

    // match new interests we sent with our own pubs we have in the system
    for (InterestListIter it = m_InterestRegQueue.begin(); 
	 it != m_InterestRegQueue.end(); it++) {

	OMInterest *in = *it;

	for (EventMMapIter mit = m_MyEvents.begin();
	     mit != m_MyEvents.end(); mit++) {

	    EventSet *events = mit->second;
	    for (EventSetIter sit = events->begin(); 
		 sit != events->end(); sit++) {

		myevents++;

		OMEvent *ev = *sit;

		if (in->Overlaps(ev)) {
		    OMEvent *copy;
		    EventMapIter old = m_LocalMatches.find(ev->GetGUID());
		    if (old == m_LocalMatches.end()) {
			copy = new OMEvent(*ev);
			m_LocalMatches[ev->GetGUID()] = copy;
		    } else {
			copy = old->second;
		    }

		    copy->SetMatched();
		    copy->SetLifeTime( MIN(ev->TimeLeft(), in->LifeTime()) );
		    copy->AddTags(in);
		}
	    }
	}
    }

    if (m_InterestRegQueue.size() > 0)
	myevents /= m_InterestRegQueue.size();

    // match new pubs with subs we have in the system
    for (EventListIter it = m_EventRegQueue.begin();
	 it != m_EventRegQueue.end(); it++) {

	OMEvent *ev = *it;

	for (InterestMapIter mit = m_MyInterests.begin();
	     mit != m_MyInterests.end(); mit++) {
	    OMInterest *in = mit->second;

	    if (in->Overlaps(ev)) {
		OMEvent *copy;
		EventMapIter old = m_LocalMatches.find(ev->GetGUID());
		if (old == m_LocalMatches.end()) {
		    copy = new OMEvent(*ev);
		    m_LocalMatches[ev->GetGUID()] = copy;
		} else {
		    copy = old->second;
		}

		copy->SetMatched();
		copy->SetLifeTime( MIN(ev->LifeTime(), in->TimeLeft()) ); 
		copy->AddTags(in);
	    }
	}
    }

    DB(10) << " iq=" << m_InterestRegQueue.size()
	   << " eq=" << m_EventRegQueue.size()
	   << " me=" << myevents
	   << " mi=" << m_MyInterests.size() << endl;
}

void PubSubManager::Maintenance(uint32 timeout)
{
    // only garbage collect if we have time
    // XXX -- something to prevent starvation!

    // if ( timeout > 0 ) 
    {
	START(PubSubManager::DoWork::ExpireStale);
	ExpireStaleItems();
	STOP(PubSubManager::DoWork::ExpireStale);
    }
}

void PubSubManager::Send(uint32 timeout)
{
    uint64 stop = TIME_STOP_USEC(timeout);

    DBG << "called" << endl;

    START(PubSubManager::DoWork::MercReg);

    for (EventListIter it = m_EventUnRegQueue.begin(); 
	 it != m_EventUnRegQueue.end(); it++) {
	if ( (*it)->TimeLeft() > ManagerParams::PUBSUB_TTL_FUDGE ) {
	    // (*it)->SetUnpublish(true);
	    //m_Router->SendEvent(*it);
	    // XXX TODO: unregistration is broken in merc!
	    // nuked this, only rely on softstate! - Ashwin [07/06/2005]
	}
    }

    for (EventListIter it = m_EventRegQueue.begin(); 
	 it != m_EventRegQueue.end(); it++) {
	//DB(1) << "sending: " << *it << endl;
	m_Router->SendEvent(*it);
	(*it)->Refresh((*it)->GetLifeTime()
		       /*- ManagerParams::PUBSUB_TTL_EXTRA */);

	///// MEASUREMENT
	if (g_MeasurementParams.enabled) {
	    DiscoveryLatEntry ent(DiscoveryLatEntry::PUB_INIT, 
				  0, (*it)->GetNonce());
	    LOG(DiscoveryLatLog, ent);
	}
	///// MEASUREMENT
    }

    for (InterestListIter it = m_InterestUnRegQueue.begin(); 
	 it != m_InterestUnRegQueue.end(); it++) {
	if ( (*it)->TimeLeft() > ManagerParams::PUBSUB_TTL_FUDGE ) {
	    // (*it)->SetUnsubscribe(true);
	    //m_Router->RegisterInterest(*it);
	    // XXX TODO: unregistration is broken in merc!
	    // yah, that's why i am just nuking it - Ashwin [07/06/2005]
	}
    }

    for (InterestListIter it = m_InterestRegQueue.begin(); 
	 it != m_InterestRegQueue.end(); it++) {
	(*it)->AddTag((*it)->GetGUID());
	//DB(1) << "sending: " << *it << endl;

	ASSERT((*it)->GetTags().size() > 0);
	m_Router->RegisterInterest(*it);
	(*it)->Refresh((*it)->GetLifeTime() - ManagerParams::PUBSUB_TTL_EXTRA);

	///// MEASUREMENT
	if (g_MeasurementParams.enabled) {
	    DiscoveryLatEntry ent(DiscoveryLatEntry::SUB_INIT, 
				  0, (*it)->GetNonce());
	    LOG(DiscoveryLatLog, ent);
	}
	///// MEASUREMENT
    }

    STOP(PubSubManager::DoWork::MercReg);

    START(PubSubManager::DoWork::MercSend);
    m_Router->Send ( TIME_LEFT_MSEC(stop) );
    STOP(PubSubManager::DoWork::MercSend);

    START(PubSubManager::DoWork::CleanUp);

    for (EventListIter it = m_EventUnRegQueue.begin(); 
	 it != m_EventUnRegQueue.end(); it++) {
	delete *it;
    }
    for (InterestListIter it = m_InterestUnRegQueue.begin(); 
	 it != m_InterestUnRegQueue.end(); it++) {
	delete *it;
    }

    STOP(PubSubManager::DoWork::CleanUp);

    DBG_DO { 
	DumpInterests(cerr); 
	DumpEvents(cerr);
    }

    // Must match items before we expire stale ones.
    // Why? Because if this function runs real slow then some of the
    // new items we sent may already be expired and deleted!

    START(PubSubManager::DoWork::MatchLocal);
    MatchLocalItems();
    STOP(PubSubManager::DoWork::MatchLocal);

    m_EventUnRegQueue.clear();
    m_EventRegQueue.clear();
    m_InterestUnRegQueue.clear();
    m_InterestRegQueue.clear();
}

///////////////////////////////////////////////////////////////////////////////

void PubSubManager::DumpInterests(ostream& out)
{
    out << "MyInterests: " << endl;
    for (InterestMapIter mit = m_MyInterests.begin();
	 mit != m_MyInterests.end(); mit++) {
	out << mit->first << " -> " << mit->second << " -> (";

	InterestIDMapIter first = m_InterestIDMap.lower_bound(mit->first);
	InterestIDMapIter last  = m_InterestIDMap.upper_bound(mit->first);
	for ( ; first != last; first++) {
	    out << first->second << " "; 
	}
	out << endl;
    }
}

void PubSubManager::DumpEvents(ostream& out)
{
    out << "MyEvents: " << endl;
    for (EventMMapIter mit = m_MyEvents.begin();
	 mit != m_MyEvents.end(); mit++) {
	EventSet *events = mit->second;
	for (EventSetIter sit = events->begin(); 
	     sit != events->end(); sit++) {
	    out << mit->first << " -> " << *sit << endl;
	}
    }
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
