////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#ifndef __OMMESSAGE__H
#define __OMMESSAGE__H

#include <mercury/MercMessage.h>
#include <om/GObjectInfo.h>
#include <om/GUpdate.h>
#include <om/GInterest.h>
#include <om/DeltaInfo.h>
#include <util/ExpLog.h>

//#define UPDATE_TIMESTAMPS

extern MsgType MSG_APP, MSG_APP_UPDATE, MSG_APP_UPDATE_ACK, MSG_APP_REMOTE_UPDATE, 
    MSG_APP_REGISTER,
    MSG_APP_FETCHREQ, MSG_APP_FETCHRESP,
    MSG_APP_MIGRATEREQ,
    MSG_SIDLIST;

void OM_RegisterMessageTypes ();
void OM_DumpMessageTypes (FILE *fp);

struct MsgApp : public MercMessage {
    DECLARE_TYPE(MercMessage, MsgApp);

    ///// FLAGS
    bool        m_IsForwarded;
    bool        m_HasNodeLoadInfo;
    bool        m_ResetConnection;

    ///// ORIGINAL SENDER (if msg forwarded)
    IPEndPoint  m_AppSender;

    ///// LOAD INFO
    CostMetric  m_ToLowWaterMark;
    CostMetric  m_ToHighWaterMark;
    CostMetric  m_Load;

    MsgApp(IPEndPoint creator=g_LocalSID);
    virtual ~MsgApp();
    MsgApp(Packet *pkt);

    virtual void Serialize(Packet *pkt);
    virtual uint32  GetLength();

    inline IPEndPoint GetSender() { 
	if (m_AppSender == SID_NONE)
	    return sender;
	else
	    return m_AppSender; 
    }

    inline bool IsForwarded() { return m_IsForwarded; }
    inline void SetIsForwarded(bool val) { m_IsForwarded = true; }

    inline bool HasLoadInfo() { return m_HasNodeLoadInfo; }
    inline void ClearLoadInfo() { m_HasNodeLoadInfo = false; }
    inline void SetLoadInfo(CostMetric toLowWaterMark, 
			    CostMetric toHighWaterMark, 
			    CostMetric load)
	{
	    m_HasNodeLoadInfo = true;
	    m_ToLowWaterMark = toLowWaterMark;
	    m_ToHighWaterMark = toHighWaterMark;
	    m_Load = load;
	}

    inline CostMetric GetToLowWaterMark()  { return m_ToLowWaterMark; }
    inline CostMetric GetToHighWaterMark() { return m_ToHighWaterMark; }
    inline CostMetric GetLoad() { return m_Load; }

    inline bool IsResetConnection() { return m_ResetConnection; }
    inline void SetResetConnection(bool val) { m_ResetConnection = val; }

    const char *TypeString() { return "MSG_APP"; }

    void Print(FILE *file);
    void Print(ostream& os);
};

struct MsgAppUpdate : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppUpdate);

    bool   onlyNew;     // asynchronous informing of new updates (measurement)
    byte   epoch;       // seqno epoch
    uint32 updateSeqno; // the sequence number for this packet
    byte   cumulative;  // packet is cumulative for past X packets
    GUpdateMap updates; // new updates to remote node

#ifdef UPDATE_TIMESTAMPS
    // XXX debug
    TimeVal timestamp;
#endif

    // HACK: not serialized
    TimeVal timeout;

    MsgAppUpdate(bool onlyNew = false, IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator), onlyNew(onlyNew), timeout(TIME_NONE),
    epoch((byte)-1), updateSeqno((uint32)-1), cumulative((byte)-1) {}
    virtual ~MsgAppUpdate() {
	// XXX TODO: these should be allocated from a pool...
	for (GUpdateMapIter it = updates.begin(); it != updates.end(); it++)
	    delete it->second;
    }

    MsgAppUpdate(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_UPDATE"; }

    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " epoch=" << (int)epoch 
	   << " uSeqno=" << updateSeqno
	   << " cum=" << (int)cumulative
	   << " updates=[";
	for (GUpdateMapIter it = updates.begin(); it != updates.end(); it++) {
	    if (it != updates.begin()) os << ",";
	    os << it->second;
	}
	os << "]";
    }
};

struct MsgAppUpdateAck : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppUpdateAck);

    byte   epoch;
    uint32 ackSeqno;

#ifdef UPDATE_TIMESTAMPS
    // XXX debug
    TimeVal timestamp;
#endif

    MsgAppUpdateAck(IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator),
    epoch((byte)-1), ackSeqno((uint32)-1) {}
    virtual ~MsgAppUpdateAck() {}

    MsgAppUpdateAck(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_UPDATE_ACK"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " ackSeqno=" << ackSeqno;
    }
};

struct MsgAppRemoteUpdate : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppRemoteUpdate);

    GUpdateMap updates; // new updates to primaries on remote node

    MsgAppRemoteUpdate(IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator) {}
    virtual ~MsgAppRemoteUpdate() {
	// XXX TODO: these should be allocated from a pool...
	for (GUpdateMapIter it = updates.begin(); it != updates.end(); it++)
	    delete it->second;
    }

    MsgAppRemoteUpdate(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_REMOTE_UPDATE"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " updates=[";
	for (GUpdateMapIter it = updates.begin(); it != updates.end(); it++) {
	    if (it != updates.begin()) os << ",";
	    os << it->second;
	}
	os << "]";
    }
};

struct MsgAppRegister : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppRegister);

    vector<GInterest> interests;

    MsgAppRegister(IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator) {}
    virtual ~MsgAppRegister() {}

    MsgAppRegister(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_REGISTER"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " interests=[";
	for (uint32 i=0; i<interests.size(); i++) {
	    if (i != 0) os << ",";
	    interests[i].Print(os);
	}
	os << "]";
    }
};


struct MsgAppFetchReq : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppFetchReq);

    guid_t guid;

    MsgAppFetchReq(guid_t guid, IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator), guid(guid) {}
    virtual ~MsgAppFetchReq() {}

    MsgAppFetchReq(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_FETCHREQ"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " " << guid;
    }
};

struct MsgAppFetchResp : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgAppFetchResp);

    GUpdate *update;

    MsgAppFetchResp() : MsgApp(), 
    update(new GUpdate(false, GUID_NONE)) {}
    MsgAppFetchResp(GUID guid, const DeltaMask& mask, Packet *delta, 
		    IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator),
    update(new GUpdate(false, guid)) {
	if (delta != NULL) {
	    update->isNew = true;
	    update->delta = delta;
	    update->mask  = mask;
	} else {
	    update->isDelete = true;
	}
    }
    virtual ~MsgAppFetchResp() {
	delete update;
    }

    MsgAppFetchResp(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_FETCHRESP"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " " << &update;
    }
};

// XXX Since this is causing us so much pain, just make this
// message reliable. :P Therefore don't need anything else...
// however, this means migration can not be canceled...
struct MsgMigrateReq : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgMigrateReq);

    GUpdate *update;
    uint16 eventSeqno;
    CostEstimate costEstimate;
    SIDDeltaMap registrations;

    MsgMigrateReq(IPEndPoint creator=g_LocalSID) : 
	MsgApp(creator),
    update(new GUpdate(false, GUID_NONE)),
    eventSeqno(0) {}
    virtual ~MsgMigrateReq() {
	delete update;
    }

    MsgMigrateReq(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32  GetLength();
    void Print(FILE *stream);

    const char *TypeString() { return "MSG_APP_MIGRATEREQ"; }
    void Print(ostream& os) {
	MsgApp::Print(os);
	os << " " << update
	   << " eventSeqno=" << eventSeqno
	   << " regs=[";
	for (SIDDeltaMapIter it=registrations.begin();
	     it != registrations.end(); it++) {
	    if (it != registrations.begin()) os << ",";
	    os << it->first;
	}
	os << "]";
    }
};


struct MsgSIDList : public MsgApp {
    DECLARE_TYPE (MercMessage, MsgSIDList);

    SIDSet  sidlist;

    MsgSIDList(IPEndPoint &sender) : MsgApp(sender) {}
    virtual ~MsgSIDList() {}

    MsgSIDList(Packet *pkt) : MsgApp(pkt) {
	uint32 count = pkt->ReadInt();
	for (uint32 i=0; i<count; i++) {
	    SID sid(pkt);
	    sidlist.insert(sid);
	}
    }
    void Serialize(Packet *pkt) { 
	MsgApp::Serialize(pkt);
	pkt->WriteInt(sidlist.size());
	for (SIDSetIter it = sidlist.begin(); it != sidlist.end(); it++) {
	    SID sid = *it;
	    sid.Serialize(pkt);
	}
    }   
    uint32  GetLength() {
	uint32 len = MsgApp::GetLength() + 4;
	for (SIDSetIter it = sidlist.begin(); it != sidlist.end(); it++) {
	    SID sid = *it;
	    len += sid.GetLength();
	}
	return len;
    }
    void Print(FILE *stream) { ASSERT(false); }

    const char *TypeString() { return "MSG_SIDLIST"; }
};

/*
  struct MsgMigrateResp : public MsgApp {
  guid_t guid;

  MsgMigrateResp(guid_t guid, IPEndPoint creator=g_LocalSID) : 
  guid(guid), MsgApp(creator) {}
  virtual ~MsgMigrateResp() {}

  MsgMigrateResp(Packet *pkt);
  void Serialize(Packet *pkt);
  uint32  GetLength();
  void Print(FILE *stream);

  const char *TypeString() { return "MSG_APP_MIGRATERESP"; }
  };

  struct MsgMigrateCancel : public MsgApp {
  guid_t guid;

  MsgMigrateCancel(guid_t guid, IPEndPoint creator=g_LocalSID) : 
  guid(guid), MsgApp(creator) {}
  virtual ~MsgMigrateCancel() {}

  MsgMigrateCancel(Packet *pkt);
  void Serialize(Packet *pkt);
  uint32  GetLength();
  void Print(FILE *stream);

  const char *TypeString() { return "MSG_APP_MIGRATECANCEL"; }
  };

  struct MsgMigrateAck : public MsgApp {
  guid_t guid;

  MsgMigrateAck(guid_t guid, IPEndPoint creator=g_LocalSID) : 
  guid(guid), MsgApp(creator) {}
  virtual ~MsgMigrateAck() {}

  MsgMigrateAck(Packet *pkt);
  void Serialize(Packet *pkt);
  uint32  GetLength();
  void Print(FILE *stream);

  const char *TypeString() { return "MSG_APP_MIGRATEACK"; }
  };
*/

#endif // __OMMESSAGE__H
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
