////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <mercury/MercMessage.h>
#include <om/OMMessage.h>

MsgType MSG_APP, MSG_APP_UPDATE, MSG_APP_UPDATE_ACK, MSG_APP_REMOTE_UPDATE, 
    MSG_APP_REGISTER,
    MSG_APP_FETCHREQ, MSG_APP_FETCHRESP,
    MSG_APP_MIGRATEREQ,
    MSG_SIDLIST;

void OM_RegisterMessageTypes () 
{
    //MSG_APP = REGISTER_TYPE (MercMessage, MsgApp);
    MSG_APP_UPDATE = REGISTER_TYPE (MercMessage, MsgAppUpdate);
    MSG_APP_UPDATE_ACK = REGISTER_TYPE (MercMessage, MsgAppUpdateAck);
    MSG_APP_REMOTE_UPDATE = REGISTER_TYPE (MercMessage, MsgAppRemoteUpdate);
    MSG_APP_REGISTER = REGISTER_TYPE (MercMessage, MsgAppRegister);
    MSG_APP_FETCHREQ = REGISTER_TYPE (MercMessage, MsgAppFetchReq);
    MSG_APP_FETCHRESP = REGISTER_TYPE (MercMessage, MsgAppFetchResp);
    MSG_APP_MIGRATEREQ = REGISTER_TYPE (MercMessage, MsgMigrateReq);
    /*
      case MSG_APP_MIGRATERESP:
      return new MsgMigrateResp(pkt);
      case MSG_APP_MIGRATECANCEL:
      return new MsgMigrateCancel(pkt);
      case MSG_APP_MIGRATEACK:
      return new MsgMigrateAck(pkt);
    */
    MSG_SIDLIST = REGISTER_TYPE (MercMessage, MsgSIDList);
}

static void _dump_type (FILE *fp, char *str, MsgType t)
{
    ASSERT (str != NULL);
    fprintf (fp, "%s %d\n", str, (int) t);
}

#define DUMP_TYPE(T)   _dump_type(fp, #T, T)

void OM_DumpMessageTypes (FILE *fp) 
{
    DUMP_TYPE(MSG_APP_UPDATE);
    DUMP_TYPE(MSG_APP_UPDATE_ACK);
    DUMP_TYPE(MSG_APP_REMOTE_UPDATE);
    DUMP_TYPE(MSG_APP_REGISTER);
    DUMP_TYPE(MSG_APP_FETCHREQ);
    DUMP_TYPE(MSG_APP_FETCHRESP);
    DUMP_TYPE(MSG_APP_MIGRATEREQ);
    DUMP_TYPE(MSG_SIDLIST);
}

MsgApp::MsgApp(IPEndPoint creator) : 
    m_HasNodeLoadInfo(false),
    m_IsForwarded(creator != g_LocalSID),
    m_ResetConnection(false),
    m_AppSender(creator),
    MercMessage() 
{
    // sender is always me!
    this->sender = g_LocalSID;
}

MsgApp::~MsgApp() {}

MsgApp::MsgApp(Packet *pkt) : MercMessage(pkt) { 
    byte flags        = pkt->ReadByte();
    m_IsForwarded     = flags & 1;
    m_HasNodeLoadInfo = flags & 2;
    m_ResetConnection = flags & 4;

    if (m_IsForwarded)
	m_AppSender = IPEndPoint(pkt);
    else 
	m_AppSender = SID_NONE;

    if (m_HasNodeLoadInfo) {
	m_ToLowWaterMark  = pkt->ReadFloat();
	m_ToHighWaterMark = pkt->ReadFloat();
	m_Load            = pkt->ReadFloat();
    }
}

void MsgApp::Serialize(Packet *pkt) { 
    MercMessage::Serialize(pkt);
    ASSERT(!IsMercMsg());

    byte flags = 0;
    if (m_IsForwarded)
	flags |= 1;
    if (m_HasNodeLoadInfo)
	flags |= 2;
    if (m_ResetConnection)
	flags |= 4;
    pkt->WriteByte((byte)flags);

    if (m_IsForwarded)
	m_AppSender.Serialize(pkt);

    if (m_HasNodeLoadInfo) {
	pkt->WriteFloat(m_ToLowWaterMark);
	pkt->WriteFloat(m_ToHighWaterMark);
	pkt->WriteFloat(m_Load);
    }
}

uint32  MsgApp::GetLength() {
    /* msg + flags */
    return MercMessage::GetLength() + 1
	+ (m_IsForwarded ? m_AppSender.GetLength() : 0)
	+ (m_HasNodeLoadInfo ? 4*3 : 0);
}

void MsgApp::Print(FILE *file) {
    MercMessage::Print(file);
}

void MsgApp::Print(ostream& os) {
    MercMessage::Print(os);
    os << " fwded=" << m_IsForwarded;
    if (m_IsForwarded)
	os << " orig=" << m_AppSender;
    if (m_HasNodeLoadInfo) {
	os << " loadinfo=[" 
	   << m_ToLowWaterMark << ","
	   << m_ToHighWaterMark << ","
	   << m_Load << "]";
    }
}

//----------------------- MSG_APP_UPDATE ---------------------
MsgAppUpdate::MsgAppUpdate(Packet * pkt):MsgApp(pkt)
{
    onlyNew       = pkt->ReadBool();
    epoch         = pkt->ReadByte();
    updateSeqno   = pkt->ReadInt();
    cumulative    = pkt->ReadByte();

    uint16 len = pkt->ReadShort();
    for (uint32 i=0; i<len; i++) {
	GUpdate *update = new GUpdate(pkt);
	updates[update->guid] = update;
    }

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    ::Deserialize(timestamp, pkt);
#endif
}

void MsgAppUpdate::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);

    pkt->WriteBool(onlyNew);
    pkt->WriteByte(epoch);
    pkt->WriteInt(updateSeqno);
    pkt->WriteByte(cumulative);

    ASSERT(updates.size() <= 0xFFFF);
    pkt->WriteShort((uint16)updates.size());
    for (GUpdateMapIter it = updates.begin();
	 it != updates.end(); it++) {
	it->second->Serialize(pkt);
    }

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    ::Serialize(timestamp, pkt);
#endif
}

uint32 MsgAppUpdate::GetLength()
{
    uint32 length = MsgApp::GetLength() + 1 + 1 + 4 + 1 + 2;

    for (GUpdateMapIter it = updates.begin();
	 it != updates.end(); it++) {
	length += it->second->GetLength();
    }

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    length += ::GetLength(timestamp);
#endif

    return length;
}

void MsgAppUpdate::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

//----------------------- MSG_APP_UPDATE_ACK ---------------------
MsgAppUpdateAck::MsgAppUpdateAck(Packet * pkt):MsgApp(pkt)
{
    epoch         = pkt->ReadByte();
    ackSeqno      = pkt->ReadInt();

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    ::Deserialize(timestamp, pkt);
#endif
}

void MsgAppUpdateAck::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);

    pkt->WriteByte(epoch);
    pkt->WriteInt(ackSeqno);

#ifdef UPDATE_TIMESTAMPS
    // xxx debug
    ::Serialize(timestamp, pkt);
#endif
}

uint32 MsgAppUpdateAck::GetLength()
{
    return MsgApp::GetLength() + 1 + 4 
#ifdef UPDATE_TIMESTAMPS
	+ ::GetLength(timestamp)
#endif
	;
}

void MsgAppUpdateAck::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

//----------------------- MSG_APP_REMOTE_UPDATE ---------------------
MsgAppRemoteUpdate::MsgAppRemoteUpdate(Packet * pkt):MsgApp(pkt)
{
    uint16 len = pkt->ReadShort();
    for (uint32 i=0; i<len; i++) {
	GUpdate *update = new GUpdate(pkt);
	updates[update->guid] = update;
    }
}

void MsgAppRemoteUpdate::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);

    ASSERT(updates.size() < 0xFFFF);
    pkt->WriteShort((uint16)updates.size());
    for (GUpdateMapIter it = updates.begin();
	 it != updates.end(); it++) {
	it->second->Serialize(pkt);
    }
}

uint32 MsgAppRemoteUpdate::GetLength()
{
    uint32 length = MsgApp::GetLength() + 2;

    for (GUpdateMapIter it = updates.begin();
	 it != updates.end(); it++) {
	length += it->second->GetLength();
    }

    return length;
}

void MsgAppRemoteUpdate::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

//----------------------- MSG_APP_REGISTER ---------------------
MsgAppRegister::MsgAppRegister(Packet * pkt):MsgApp(pkt)
{
    uint16 len = pkt->ReadShort();
    for (int i=0; i<len; i++) {
	GInterest interest = GInterest(pkt);
	interests.push_back(interest);
    }
}

void MsgAppRegister::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    ASSERT(interests.size() < 0xFFFF);
    pkt->WriteShort((uint16)interests.size());
    for (uint32 i=0; i<interests.size(); i++) {
	interests[i].Serialize(pkt);
    }
}

uint32 MsgAppRegister::GetLength()
{
    uint32 length = MsgApp::GetLength() + 2;

    for (uint32 i=0; i<interests.size(); i++) {
	length += interests[i].GetLength();
    }

    return length;
}

void MsgAppRegister::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

//----------------------- MSG_APP_FETCHREQ ---------------------
MsgAppFetchReq::MsgAppFetchReq(Packet * pkt):MsgApp(pkt)
{
    guid = GUID(pkt);
}

void MsgAppFetchReq::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    guid.Serialize(pkt);
}

uint32 MsgAppFetchReq::GetLength()
{
    return MsgApp::GetLength() + guid.GetLength();
}

void MsgAppFetchReq::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
    guid.Print(stream);
}

//----------------------- MSG_APP_FETCHRESP ---------------------
MsgAppFetchResp::MsgAppFetchResp(Packet * pkt): MsgApp(pkt) 
{	
    update = new GUpdate(pkt);
}

void MsgAppFetchResp::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    update->Serialize(pkt);
}

uint32 MsgAppFetchResp::GetLength()
{
    return MsgApp::GetLength() + update->GetLength();
}

void MsgAppFetchResp::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

//----------------------- MSG_APP_MIGRATEREQ ---------------------
MsgMigrateReq::MsgMigrateReq(Packet * pkt):MsgApp(pkt)
{
    update = new GUpdate(pkt);
    eventSeqno = pkt->ReadShort();
    costEstimate = CostEstimate(pkt);
    uint16 len = pkt->ReadShort();
    for (int i=0; i<len; i++) {
	SID sid = SID(pkt);
	DeltaInfo dinfo = DeltaInfo(pkt);
	registrations[sid] = dinfo;
    }
}

void MsgMigrateReq::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    update->Serialize(pkt);
    pkt->WriteShort(eventSeqno);
    costEstimate.Serialize(pkt);
    ASSERT((uint16)registrations.size() < 0xFFFF);
    pkt->WriteShort((uint16)registrations.size());
    for (SIDDeltaMapIter it = registrations.begin(); 
	 it != registrations.end(); it++) {
	SID sid = it->first;
	sid.Serialize(pkt);
	it->second.Serialize(pkt);
    }
}

uint32 MsgMigrateReq::GetLength()
{
    int length = MsgApp::GetLength() + update->GetLength();
    length += costEstimate.GetLength();
    length += 2;
    length += 2;
    for (SIDDeltaMapIter it = registrations.begin(); 
	 it != registrations.end(); it++) {
	SID sid = it->first;
	length += sid.GetLength();
	length += it->second.GetLength();
    }
    return length;
}

void MsgMigrateReq::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
}

/*
//----------------------- MSG_APP_MIGRATERESP ---------------------
MsgMigrateResp::MsgMigrateResp(Packet * pkt):MsgApp(pkt)
{
guid = GUID(pkt);
}

void MsgMigrateResp::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    guid.Serialize(pkt);
}

uint32 MsgMigrateResp::GetLength()
{
    return MsgApp::GetLength() + guid.GetLength();
}

void MsgMigrateResp::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
    guid.Print(stream);
}

//----------------------- MSG_APP_MIGRATECANCEL ---------------------
MsgMigrateCancel::MsgMigrateCancel(Packet * pkt):MsgApp(pkt)
{
    guid = GUID(pkt);
}

void MsgMigrateCancel::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    guid.Serialize(pkt);
}

uint32 MsgMigrateCancel::GetLength()
{
    return MsgApp::GetLength() + guid.GetLength();
}

void MsgMigrateCancel::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
    guid.Print(stream);
}

//----------------------- MSG_APP_MIGRATEACK ---------------------
MsgMigrateAck::MsgMigrateAck(Packet * pkt):MsgApp(pkt)
{
    guid = GUID(pkt);
}

void MsgMigrateAck::Serialize(Packet * pkt)
{
    MsgApp::Serialize(pkt);
    guid.Serialize(pkt);
}

uint32 MsgMigrateAck::GetLength()
{
    return MsgApp::GetLength() + guid.GetLength();
}

void MsgMigrateAck::Print(FILE * stream)
{
    MsgApp::Print(stream);
    fprintf(stream, "\n");
    guid.Print(stream);
}
*/

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
