////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef __OM_LOGS_H__
#define __OM_LOGS_H__

#include <util/IPEndPoint.h>
#include <util/ID.h>
#include <util/types.h>
#include <util/ExpLog.h>


struct MigrationLogEntry : public ExpLogEntry
{
    GUID guid;
    SID  to;

    MigrationLogEntry() {}
    MigrationLogEntry(const GUID& guid, const SID& to) :
	guid(guid), to(to) {}
    virtual ~MigrationLogEntry() {};

    uint32 Dump(FILE *fp) {
	uint32 ret = 0;
	ret += _PrintTime (fp, time);
	ret += _PrintGUID (fp, guid);
	ret += _PrintSID (fp, to);
	ret += fprintf (fp, "\n");
	return ret;
    }
};

struct RepConnSendDroppedEntry : public ExpLogEntry
{
    SID dest;
    uint16 sent, async, lost;
    float avg_window;

    RepConnSendDroppedEntry() {}
    RepConnSendDroppedEntry(const SID& dest, uint16 sent, uint16 async, uint16 lost, float avg_window) :
	dest(dest), sent(sent), async(async), lost(lost), avg_window(avg_window) {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += _PrintSID (fp, dest);
	ret += fprintf(fp, "\t%d\t%d\t%d\t%.2f\n",
		sent, async, lost, avg_window);
	return ret;
    }
};

struct RepConnRecvDroppedEntry : public ExpLogEntry
{
    SID dest;
    uint16 recv, missing, old;
    uint16 async, amissing, aold;

    RepConnRecvDroppedEntry() {}
    RepConnRecvDroppedEntry(const SID& dest, uint16 recv, uint16 missing, uint16 old, uint16 async, uint16 amissing, uint16 aold) :
	dest(dest), recv(recv), missing(missing), old(old), async(async), amissing(amissing), aold(aold) {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += _PrintSID (fp, dest);
	ret += fprintf(fp, "\t%d\t%d\t%d\t%d\t%d\t%d\n",
		recv, missing, old,
		async, amissing, aold);
	return ret;
    }
};

struct RepConnSendEntry : public ExpLogEntry
{
    SID dest;
    sint32 interarrival;
    uint16 size;
    byte n, d, u;

    RepConnSendEntry() {}
    RepConnSendEntry(const SID& dest, sint32 interarrival, 
	    uint16 size, uint32 n, uint32 d, uint32 u) :
	dest(dest), interarrival(interarrival), size(size),
	n((byte)n), d((byte)d), u((byte)u) {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += _PrintSID (fp, dest);

	ret += fprintf(fp, "\t%d\t%d\t%d\t%d\t%d\n",
		interarrival, size,
		(int)n, (int)d, (int)u);
	return ret;
    }
};

struct RepConnRecvEntry : public ExpLogEntry
{
    SID src;
    sint32 interarrival;
    uint16 size;
    byte n, d, u;

    RepConnRecvEntry() {}
    RepConnRecvEntry(const SID& src, sint32 interarrival, 
	    uint16 size, uint32 n, uint32 d, uint32 u) :
	src(src), interarrival(interarrival), size(size),
	n((byte)n), d((byte)d), u((byte)u) {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += _PrintSID (fp, src);

	ret += fprintf(fp, "\t%d\t%d\t%d\t%d\t%d\n",
		interarrival, size,
		(int)n, (int)d, (int)u);
	return ret;
    }
};

#define UPDATE_MSG_LOG

#ifdef UPDATE_MSG_LOG
struct UpdateMsgEntry : public ExpLogEntry 
{
    uint16 n_te, n_e;         // #temporary, #total entities
    uint   total_te, total_e; // total size;
    uint16 hdr_len;           // update packet header length

    UpdateMsgEntry () {}
    UpdateMsgEntry (uint16 n_te, uint16 n_e, uint total_te, uint total_e, uint16 hdr_len) : 
	n_te (n_te), n_e (n_e), total_te (total_te), total_e (total_e), hdr_len (hdr_len) {}

    uint32 Dump (FILE *fp) {
	return fprintf (fp, "%13.3f\t%d\t%d\t%d\t%d\t%d\n",
		(double)time.tv_sec +
		(double)time.tv_usec/USEC_IN_SEC,
		(int) n_te, (int) total_te,
		(int) n_e,  (int) total_e,
		(int) hdr_len);
    }
};

DECLARE_EXPLOG(UpdateMsgLog, UpdateMsgEntry);
#endif

///////////////////////////////////////////////////////////////////////////////

DECLARE_EXPLOG(MigrationLog, MigrationLogEntry);
DECLARE_EXPLOG(RepConnSendDroppedLog, RepConnSendDroppedEntry);
DECLARE_EXPLOG(RepConnRecvDroppedLog, RepConnRecvDroppedEntry);
DECLARE_EXPLOG(RepConnSendLog, RepConnSendEntry);
DECLARE_EXPLOG(RepConnRecvLog, RepConnRecvEntry);

extern void InitOMLogs(IPEndPoint sid);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
