////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef OM_LOAD_MANAGER__H
#define OM_LOAD_MANAGER__H

#include <hash_map.h>
#include <mercury/common.h>
#include <mercury/Sampling.h>
#include <mercury/Event.h>
#include <mercury/Interest.h>
#include <om/common.h>
#include <om/ManagerParams.h>
#include <wan-env/RealNet.h>
#include <util/TimedStruct.h>
#include <util/GPL/DataChunker.h>
#include <util/GPL/refcnt.h>

class Manager;
class EventDemux;

///////////////////////////////////////////////////////////////////////////////
//
// Data Items
//

class NodeLoadInfo : public TimedStruct {
 public:
    SID        sid;
    CostMetric toLowWaterMark;
    CostMetric toHighWaterMark;
    CostMetric load;

    NodeLoadInfo() : TimedStruct(ManagerParams::NODE_LOAD_INFO_TTL) {}
    NodeLoadInfo(const SID& id, 
		 CostMetric tolow, 
		 CostMetric tohigh, 
		 CostMetric load) : 
	TimedStruct(ManagerParams::NODE_LOAD_INFO_TTL),
	sid(id), toLowWaterMark(tolow), 
	toHighWaterMark(tohigh), load(load) {}
    virtual ~NodeLoadInfo() {}

    NodeLoadInfo(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *) { }
};

extern EventType PUB_OM_LOAD;
class OMLoadEvent : public PointEvent
{
    NodeLoadInfo m_LoadInfo;
 protected:
    DECLARE_TYPE (Event, OMLoadEvent);
 public:
    /**
     * @param hubid the hub which this gets published to (load info hub)
     * @param info  current info about our node.
     */
    OMLoadEvent(uint32 hubid, const NodeLoadInfo& info);
    virtual bool LessThan(const Event *oe) const;
    virtual void OnInterestMatch(Interest *in);
    virtual bool OverwriteEvent(Event *old) const;
    virtual ostream& Print (ostream& out) const;

    virtual const char *TypeString() { return "OMLOAD_EVENT"; }
    const NodeLoadInfo& GetLoadInfo() const { return m_LoadInfo; }

    OMLoadEvent(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *);
};

extern EventType INTEREST_OM_LOAD;
class OMLoadInterest : public Interest
{
 protected:
    DECLARE_TYPE (Interest, OMLoadInterest);
 public:
    /**
     * @param minRequired the min spare capacity we need to be useful
     */
    OMLoadInterest(uint32 hubid, CostMetric minRequired);
    virtual ~OMLoadInterest() {}

    OMLoadInterest(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
};

// typedef'd so we can change it later if we want
typedef LoadMetric OMLoadMetric;

///////////////////////////////////////////////////////////////////////////////
//
// Sampling Management Components
//

class OMLoadManager;

class OMLoadSampler : public Sampler
{
    OMLoadManager *m_Manager;
 public:
    OMLoadSampler (OMLoadManager *m) : m_Manager (m) {}
    virtual const char *GetName () const;
    virtual int GetLocalRadius () const;
    virtual u_long GetSampleLifeTime () const;
    virtual int GetNumReportSamples () const;
    virtual int GetRandomWalkInterval () const;
    virtual Metric* GetPointEstimate ();
    virtual Metric* MakeLocalEstimate (vector<Metric *>& samples);
};

// Facade Interface ontop of Mercury, so we can substitute a
// broadcast mechanism as a replacement for testing (if needed)
class OMLoadAggregator
{
 public:
    virtual ~OMLoadAggregator () {}

    virtual bool EstimateSystemLoad(float *mean, float *stddev) = 0;
    virtual void SendEvent(OMLoadEvent *ev) = 0;
    virtual void RegisterInterest(OMLoadInterest *in) = 0;
    virtual OMLoadEvent *ReadEvent() = 0;
    virtual void DoWork() = 0;
};

class MercuryLoadAggregator : public OMLoadAggregator
{
    uint32 m_HubID;
    MercuryNode *m_Node;
    EventDemux  *m_Demux;
    OMLoadSampler *m_Sampler;
 public:
    MercuryLoadAggregator(uint32 hubid, MercuryNode *n, 
			  EventDemux *d, OMLoadSampler *s);
    virtual ~MercuryLoadAggregator() {}
    virtual bool EstimateSystemLoad(float *mean, float *stddev);
    virtual void SendEvent(OMLoadEvent *ev);
    virtual void RegisterInterest(OMLoadInterest *in);
    virtual OMLoadEvent *ReadEvent();
    virtual void DoWork();
};

typedef hash_map<SID, float, hash_SID, equal_SID> LoadEstimateMap;
typedef LoadEstimateMap::iterator LoadEstimateMapIter;

class Scheduler;
class RealNet;
class MercMessage;
class MsgBlob;
class MsgSubscription;
class MsgPublication;

/*
 * this just implements TimeNow ()
 */
class SchedulerWithoutEventQueue : public Scheduler {
    TimeVal m_Now;
 public:
    SchedulerWithoutEventQueue () {}

    virtual void RaiseEvent (ref<SchedulerEvent> event, IPEndPoint&, 
			     u_long millis) {}
    virtual void CancelEvent (ref<SchedulerEvent> event) {}
    virtual void ProcessTill(TimeVal&) {}
    virtual void ProcessFor (u_long millis) {}
    virtual void Reset () {}
    virtual TimeVal& TimeNow () {
	OS::GetCurrentTime(&m_Now);
	return m_Now;
    }

    void FireEvents () { ASSERT (false); }
};

class BroadcastLoadAggregator : public OMLoadAggregator
{
    OMLoadManager *m_Manager;
    Scheduler *m_Scheduler;
    RealNet *m_Network;

    vector<SID> m_Others;
    LoadEstimateMap m_Map;
    // local events we created
    list<OMLoadEvent *> m_LocalEvents;
    list<OMLoadEvent *> m_RemoteEvents;

    void Broadcast(MercMessage *msg);
    void Send(SID target, MercMessage *msg);
    SID  NextInRing();
    void ProcessMessages(sint32 maxMsgs);
    void ProcessMessage(MercMessage *m);
    void ProcessLoadEstimate(MsgBlob *msg);
    void BroadcastLoadEstimate();
    void HandleSubscription(MsgSubscription *msg);
    void HandlePublication(MsgPublication *msg);
    void GarbageCollect(const TimeVal& now);
 public:
    BroadcastLoadAggregator(OMLoadManager *m);
    virtual ~BroadcastLoadAggregator();
    virtual bool EstimateSystemLoad(float *mean, float *stddev);
    virtual void SendEvent(OMLoadEvent *ev);
    virtual void RegisterInterest(OMLoadInterest *in);
    virtual OMLoadEvent *ReadEvent();
    virtual void DoWork();
};

class OMLoadSearcher
{
    OMLoadManager *m_Manager;
    OMLoadSampler *m_Sampler;
    OMLoadAggregator *m_Aggregator;

    float m_MinRequired;
    bool  m_IsQuerying;
    uint32 m_WaitFor;

    TimeVal m_LastSearch;
    TimeVal m_LastQuery;

    vector< ref<NodeLoadInfo> > m_Results;

    void PublishLoad();
    void RegisterSearch();
 public:
    // XXX -- need some maximum to bound the ends of the ring
    // even though a node could potentially have infinite spare capacity...
    // Hopefully mercury will do load balancing appropriately for this ring
    // since most of the high end is likely to be empty...
    // ~1.25MBps ~ 10MBps
    static const float MAX_SPARE_CAPACITY = 1250000.0;

    OMLoadSearcher(OMLoadManager *m, OMLoadSampler *s, OMLoadAggregator *a);
    virtual ~OMLoadSearcher();

    /** Begin querying for some guy with at least required spare capacity */
    void   BeginQuery(float required, uint32 waitfor = 1);
    /** End euerying */
    void   EndQuery();
    uint32 NumResults();
    void   GetResults(vector< ref<NodeLoadInfo> > *ret);

    void   DoWork();
};

///////////////////////////////////////////////////////////////////////////////
//
// Main Load Sampling Interface
//

typedef enum { OMLOADAGG_MERCURY, OMLOADAGG_BROADCAST } OMLoadAggType;
typedef enum { OMLOAD_LOW, OMLOAD_MID_LOW, OMLOAD_MID_HIGH, OMLOAD_HIGH } OMLoadStatus;

typedef map<SID, ref<NodeLoadInfo>, less_SID> NodeLoadMap;
typedef NodeLoadMap::iterator NodeLoadMapIter;

class OMLoadManager
{
    friend class OMLoadSampler;

    // hub used for load info
    byte m_HubID;

    // node -> predicted load
    NodeLoadMap m_LoadMap;
    CostMetric  m_LoadTarget;
    CostMetric  m_WindowTarget;

    CostMetric  m_Load;
    CostMetric  m_HighWaterMark;
    CostMetric  m_LowWaterMark;

    // for debugging (the actual EMA load read in from RealNet)
    CostMetric  m_LoadActual;

    // these are normalized cost metrics (load/capacity)
    float       m_SystemLoadMean;
    float       m_SystemLoadStdDev;

    OMLoadAggregator *m_Aggregator;
    OMLoadSearcher   *m_Searcher;
    OMLoadSampler    *m_Sampler;

    Manager *m_Manager;
    RealNet *m_Network;

    void GarbageCollect();
    bool RecalculateSystemLoad();
    void EstimateLoadParameters();
    void ProcessSearchResults();

 public:
    OMLoadManager(const string& hubid, Manager *m, RealNet *n, 
		  OMLoadAggType t = OMLOADAGG_MERCURY);
    virtual ~OMLoadManager();

    byte GetHubID() { return m_HubID; }
    NodeLoadMap *GetLoadMap() { return &m_LoadMap; }
    void       SetLoad(CostMetric load) { m_Load = load; }
    CostMetric GetLoad() const { return m_Load; }
    CostMetric GetLoadTarget() const { return m_LoadTarget; }
    CostMetric GetHighWaterMark() const { return m_HighWaterMark; }
    CostMetric GetLowWaterMark() const { return m_LowWaterMark; }
    CostMetric GetCapacity() const { return ManagerParams::CAPACITY; }
    const NodeLoadInfo& GetLoadInfo() const;
    OMLoadStatus GetLoadStatus() const;

    void Record(const NodeLoadInfo& info);
    void DoWork();
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
