////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/**
 *   copyright       : (C) 2003-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
 *
 * $Id: OMEvent.h 2382 2005-11-03 22:54:59Z ashu $
 **/
#ifndef __OMEVENT__H
#define __OMEVENT__H

#include <mercury/Event.h>
#include <util/TimedStruct.h>
#include <om/common.h>
#include <om/GUpdate.h>

class OMInterest;

// for load prediction
extern const string ATTR_LOAD; // HACK: should not need this
extern const string ATTR_TO_LOW_WATER_MARK;
extern const string ATTR_TO_HIGH_WATER_MARK;

/**
 * This represents an Event as needed by the OM component.
 * Contains some extra information needed for tracking object
 * interactions.
 *
 **/

extern EventType PUB_OM;
void OM_RegisterEventTypes ();

class OMEvent : public MercuryEvent, public TimedStruct
{
    guid_t     m_GUID;           // target of event
    sid_t      m_SID;            // location of the primary
    Tags       m_Tags;           // tags inserted by the matching RP in Mercury

    uint16     m_SequenceNumber; // ordering object updates

    GUpdate   *m_Update; // possible full data update
 protected:
    DECLARE_TYPE(Event, OMEvent);
 public:
    OMEvent();
    OMEvent(guid_t target, sid_t loc);
    OMEvent(Packet *pkt);
    OMEvent(const OMEvent& other);
    virtual ~OMEvent() {
	if (m_Update) { delete m_Update; }
    }

    virtual const char *TypeString() { return "OMEVENT"; }

    bool IsPointEvent();

    void SetUpdate(GUpdate *update) {
	m_Update = update;
    }

    GUpdate *GetUpdate() {
	return m_Update;
    }

    // inherits range event interface

    // point event interface -- mascarade as a point event
    void AddTuple(const Tuple& t);
    const Value *GetValue(int attr);

    void SetLifeTime(uint32 t) {
	TimedStruct::SetLifeTime(t);
	MercuryEvent::SetLifeTime(t);
    }

    inline guid_t  GetGUID() const { return m_GUID; }
    inline sid_t   GetSID() const  { return m_SID; }
    inline void    SetSID(SID sid) { m_SID = sid; }

    inline uint16 GetSeqno() const { return m_SequenceNumber; }
    inline void SetSeqno(uint16 seqno) {
	m_SequenceNumber = seqno; 
    }
    static inline uint16 GetNextSeqno(uint16 seqno) {
	// to enforce 3 epochs, when the epoch number reaches 11
	// force it back to 0
	if ( (((++seqno) >> 14) & 3) == 3 )
	    return seqno & 0x3FFF;
	else
	    return seqno;
    }

    virtual ostream& Print (ostream& out) const;

    inline bool IsNewer(OMEvent *earlier) const {
	//  __________________________  
	// |                          | 
	// | ... < 00 < 01 < 10 < ... |  
	uint16 this_epoch  = (GetSeqno()>>14)&3;
	uint16 other_epoch = (earlier->GetSeqno()>>14)&3;

	// I am newer if my epoch is not just behind the other one
	// -- since there are 3 epochs, this enforces a circular order
	return ((this_epoch+1)%3) != other_epoch ||
	    (this_epoch == other_epoch && 
	     GetSeqno() > earlier->GetSeqno());
    }

    virtual bool LessThan(const Event *oe) const;

    virtual bool OverwriteEvent(Event *old) const;

    void    AddTag(GUID tag);
    void    AddTags(OMInterest *in);
    Tags&   GetTags();

    //// Serialization
    virtual void   Serialize(Packet *pkt);
    virtual uint32 GetLength();
    virtual void   Print(FILE *stream);    
};

#endif // __OMEVENT__H
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
