////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef __MIGRATION_POLICY_H__
#define __MIGRATION_POLICY_H__

#include <queue>
#include <om/common.h>
#include <om/OMLoadManager.h>

    class ObjectStore;
class Manager;
class GObject;
class GObjectInfo;

typedef map<GUID,SID,less_GUID> MigrateMap;
typedef MigrateMap::iterator MigrateMapIter;

/**
 * A policy implementation defines how the system should create and
 * migrate objects during execution.
 */
class MigrationPolicy {
 protected:

    Manager *m_Manager;

 public:
    virtual ~MigrationPolicy () {}

    void SetManager(Manager *manager) {
	m_Manager = manager;
    }

    /**
     * Consider whether to migrate objects off this <tt>node</tt>.
     * Place the objects to migrate and the nodes to migrate them to
     * into <tt>toMigrate</tt>.
     *
     * @param toMigrate   an GUID to SID map that the implementor
     *                    should place the result in. It should contain
     *                    a mapping of objects and where they should
     *                    migrate to. (do not include objects that do
     *                    not migrate)
     * @param locChange   XXX UNUSED!
     */
    virtual void Migrate(MigrateMap& toMigrate,
			 GObject *locChange = NULL) = 0;

    /**
     * Same, except this is called once per frame
     */
    virtual void MigrateFreq(MigrateMap& toMigrate) {}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Never migrate objects
 */
class StaticPlacementPolicy : public MigrationPolicy {
 public:

    virtual void Migrate(MigrateMap& toMigrate,
			 GObject *locChange = NULL);

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Randomly migrate objects
 */
class RandomPlacementPolicy : public MigrationPolicy {
 protected:

    SIDSet m_Servers;
    SIDVec m_ServersList;

 public:

    virtual ~RandomPlacementPolicy() {}

    void AddServer(SID sid) {
	if (m_Servers.find(sid) == m_Servers.end()) {
	    m_Servers.insert(sid);
	    m_ServersList.push_back(sid);
	}
    }

    void Migrate(MigrateMap& toMigrate,
		 GObject *locChange = NULL);

};

///////////////////////////////////////////////////////////////////////////////

typedef hash_map<uint32, SID> RegionMap;
typedef RegionMap::iterator RegionMapIter;

/**
 * Region-based placement policy (assume objects have an integer attribute
 * "node" which is their region (BSP tree node).
 */
class RegionPlacementPolicy : public MigrationPolicy {
 private:
 protected:

    SID getResponsibleServer(uint32 region);
 public:

    RegionPlacementPolicy();

    virtual ~RegionPlacementPolicy() { }

    virtual void MigrateFreq(MigrateMap& toMigrate);

    virtual void Migrate(MigrateMap& toMigrate,
			 GObject *locChange = NULL) {}

};

///////////////////////////////////////////////////////////////////////////////

// define so we don't do the clustering heurisitic
//#define NO_CLUSTERING   1
// define so we don't to the load shedding heuristic
//#define NO_LOADSHEDDING 1

// the random fraction of the candidates to consider for load shedding
#define LOAD_SHEDDING_CANDIDATE_FRAC 0.75
#define LOAD_SHEDDING_CANDIDATES_MAX 20

typedef hash_set<GObjectInfo *, hash_ptr, equal_ptr> Component;
typedef Component::iterator ComponentIter;
typedef vector<Component> Components;
typedef Components::iterator ComponentsIter;

typedef struct {
    uint32   orig_index;
    CostMetric savings;
} CCPair;

// sorts in descending order (most savings first)
struct descending_All {
    bool operator() ( const CCPair& a, const CCPair& b) const {
	return !(a.savings < b.savings);
    }
};

// sorts in  ascending order (smallest comp first)
struct ascending_All {
    bool operator() ( const CCPair& a, const CCPair& b) const {
	return a.savings < b.savings;
    }
};

typedef vector<CCPair> Savings;
typedef Savings::iterator SavingsIter;

struct NodeLoadComparator : 
    public binary_function<NodeLoadInfo *, NodeLoadInfo *,bool> {

    /**
     * Returns true a > b. (sort in descending order -- consider
     * guys with most spare capacity first)
     */
    bool operator()(NodeLoadInfo *a, NodeLoadInfo *b) {
	return a->toHighWaterMark > b->toHighWaterMark;
    }
};

typedef priority_queue<NodeLoadInfo *, vector<NodeLoadInfo *>, 
		       NodeLoadComparator > LoadQueue;

///////////////////////////////////////////////////////////////////////////////

ostream& operator <<(ostream& out, MigrateMap& mm);

/**
 * Interest graph based heuristics.
 */
class LoadBalancedPolicy : public MigrationPolicy {
 protected:

    void cluster(MigrateMap& toMigrate, CostMetric load);
    void offload(MigrateMap& toMigrate, CostMetric& load);
    void markLocal(Component& visited, Component& component, 
		   GObjectInfo *obj, ObjectStore *store);
    void calcSavings(Components& cc, Savings& save, ObjectStore *store);
    void calcSavings(CCPair *savings, Component& c, ObjectStore *store);
    void calcCost(CCPair *cost, SID& target, Component& c, ObjectStore *store);
    void calcLoadVec(SIDSet& candidates, NodeLoadMap *lmap, LoadQueue& load);

    bool localInterest(GObjectInfo *obj, MigrateMap& toMigrate);
    CostMetric calcSavings(GObjectInfo *obj, ObjectStore *store,
			   MigrateMap& toMigrate);
    CostMetric calcCost(GObjectInfo *obj, SID& target, ObjectStore *store,
			MigrateMap& toMigrate);

 public:

    virtual void Migrate(MigrateMap& toMigrate,
			 GObject *locChange = NULL);

};


#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
