////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __MANAGER_PARAMS_H__
#define __MANAGER_PARAMS_H__

#include <hash_map.h>
#include <om/common.h>
#include <mercury/Constraint.h>

/**
 * Parameters for Manager.
 */
namespace ManagerParams {

    ///////////////////////////////////////////////////////////////////////////
    ///// SOFT-STATE TTLS

    // how long we keep sending updates to remote replicas after we stop
    // receiving registration heart-beats. Should be longer than 
    // REPLICA_INTEREST_TTL (otherwise the remote node may keep a stale
    // replica). This is just a "back-up" mechanism to garbage collect
    // maintenance state because the remote node will tell us explicitly
    // when REPLICA_INTEREST_TTL runs out.
    extern uint32 REPLICA_MAINTAIN_TTL;

    // If the source has not been interested in the target for this long, then
    // the interest is stale and should be garbage collected. Whenever an
    // interest link is refreshed, it is refreshed for this long +
    // how long the object says it is interested.
    extern uint32 INTEREST_LINK_TTL;

    // If no primary is interested in a replica for this long, garbage collect
    // it. Whenever interest is refreshed, it is refreshed for this long +
    // how long the object says it is interested.
    extern uint32 REPLICA_INTEREST_TTL;

    // How long to keep GUIDs for deleted objects
    // ~ how long another node may keep a replica without receiving an update
    //   should be longer than the longest pub/sub TTL
    extern uint32 DELETED_GUID_TTL;

    // How long to keep forwarding pointers
    // ~ how long another node may keep a replica without receiving a new pub
    extern uint32 FORWARDING_PTR_TTL;

    // How long to keep load information about remote nodes
    // ~ how long "load" values can be trusted before going stale
    extern uint32 NODE_LOAD_INFO_TTL;

    // the fudge factor for deciding whether we should unregister
    // publications or subscriptions that have not yet expired
    extern uint32 PUBSUB_TTL_FUDGE;

    // the extra time we give to our own publications and subscriptions
    // in the system (e.g., something like the storage latency) 
    //
    // XXX UNUSED!!!
    extern uint32 PUBSUB_TTL_EXTRA;

    ///////////////////////////////////////////////////////////////////////////
    ///// TIMERS

    // How often to send pubs (of non-new objects), only used if triggered
    // pubs aren't enabled
    extern uint32 PUBLICATION_INTERVAL;

    // How often to send heartbeats for replicas that we wish to receive deltas
    // ~ faster than REPLICA_MAINTAIN_TTL (like 1/2)
    extern uint32 REPLICA_MAINTAIN_HB;

    // How long to wait before retransmitting MigrationResps that aren't Ack'd
    // XXX HACK -- This should be adaptive
    // ~ RTT + processing
    extern uint32 MIGRATION_RESP_RETRANSMIT;

    // How often to broadcast load info to neighboring nodes that are receiving
    // replica updates from us (so they can make decisions about migration)
    // ~ on the order of now long load is relatively stable. Should
    // definitely be more frequent than NODE_LOAD_INFO_TTL + transmit time
    extern uint32 BROADCAST_LOAD_INFO_TIMER;

    // How often to publish load info and resubscibe to load info in a range
    // ~ on the order of how long load is relatively stable. Should
    // definitely be more frequent than NODE_LOAD_INFO_TTL + transmit time
    extern uint32 PUBLISH_LOAD_INFO_TIMER;

    // How often to initiate a new search for lightly loaded guys
    extern uint32 SEARCH_LOAD_INFO_TIMER;

    // How often we reestimate local load
    extern uint32 LOAD_ESTIMATION_TIMER;

    // The length of time until interests are considered "stable"
    // XXX HACK -- This should be app configurable
    extern uint32 INTEREST_STABILIZATION_TIME;

    // How often to run optimization heuristics to consider migration
    // ~ on the order of how long interests take to stabilize and
    //   shorter than 1/2 the avg. length of a stable interest
    extern uint32 MIGRATION_CONSIDER;

    // At what grainularity should softstate be checked for expiration
    // e.g., if the TTL on softstate is X, then we check expiration every
    // X/SOFTSTATE_EXPIRE_GRAINULARITY
    extern uint32 SOFTSTATE_EXPIRE_GRAINULARITY;

    ///////////////////////////////////////////////////////////////////////////
    ///// LOAD PREDICTION INFO

    // Initial low water mark (right now == bandwidth in Bps)
    extern CostMetric LOW_WATER_MARK;
    // Initial high water mark (right now == bandwidth in Bps)
    extern CostMetric HIGH_WATER_MARK;
    // Capacity (right now == bandwidth in Bps)
    extern CostMetric CAPACITY;
    // The multiplicative factor around the mean that we use as low
    // and high water marks (low = mean - alpha*mean, high = mean + alpha*mean)
    extern float LOAD_WINDOW_ALPHA;
    // Minimum difference between high and low water marks (Bps)
    extern CostMetric LOAD_MIN_WINDOW;
    // Maximum high water mark (XXX: what was this for? Unused)
    //extern CostMetric LOAD_MAX_HIGH_WATER_MARK;

    // Weight for exponentially weighted average to interpolate toward
    // a target load value (weight*old + (1-weight)*new).
    extern float LOAD_EMA_WEIGHT;
    // Overestimate the cost + savings of clusters we offload to account
    // for variance and to avoid oscillation by this much
    extern float LOAD_OFFLOAD_FACTOR;
    // Minimum number of load samples from the system in order to estimate
    // mean load + stddev (normalized) of system
    extern uint32 LOAD_MIN_AGGREGATION_SAMPLES;

    // Mercury hub used for load info searches
    extern char LOAD_INFO_HUB[];

    // Parameters for when we we broadcasting load info (debugging)
    extern uint32 BROADCAST_LOAD_INFO_PORT;
    extern char   BROADCAST_LOAD_INFO_OTHERS[];

    ///////////////////////////////////////////////////////////////////////////
    ///// FLAGS

    // Wait for all nodes to join the mercury network before starting
    extern bool WAIT_ALL_JOIN;

    // Publish deltas with the first publication from new objects
    extern bool PUBLISH_NEW_DELTAS;

    // Proactively replicate "attached" objects
    extern bool PROACTIVE_REPLICATION;

    // Enable exchange of load information (requires a hub called "Load")
    extern bool ENABLE_LOAD_INFO;

    // Broadcast load info instead of using Mercury -- for debugging
    extern bool BROADCAST_LOAD_INFO;

    // Run periodic maintenance events every iteration
    extern bool FREQUENT_MAINTENANCE;

    // Check Invariants
    extern bool CHECK_INVARIANTS;

    ///////////////////////////////////////////////////////////////////////////
    ///// OTHER

    // maximum size of the data in update messages
    extern int MAX_UPDATE_MSG_SIZE;

    extern char MERC_ATTR_BOUNDS_STR[];
    extern hash_map<string, Constraint, hash_string> MERC_ATTR_BOUNDS;

    // migration placement policy {static,rand,region,load}
    extern char PLACEMENT_POLICY[];
    extern char REGION_MAP_STR[];
    extern hash_map<uint32, SID> REGION_MAP;

    extern char TERMINAL_PASSWORD[];

    extern char CONFIG_OPTS[];

    extern char CONFIG_FILE[];

};

void ConfigManagerParams();
void ScaleManagerParameters (float factor);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
