////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  Manager.h

  Object Management Interface.

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

***************************************************************************/

#ifndef __MANAGER_H__
#define __MANAGER_H__

#include <list>
#include <pthread.h>
#include <gmpxx.h>
#include <wan-env/WANMercuryNode.h>
#include <mercury/MercMessage.h>
#include <util/GPL/DataChunker.h>
#include <util/TimedStruct.h>
#include <util/Mutex.h>
#include <om/common.h>
#include <om/OMEvent.h>
#include <om/OMInterest.h>
#include <om/GObject.h>
#include <om/GObjectInfo.h>
#include <om/GameAdaptor.h>
#include <om/ManagerParams.h>
#include <om/ReplicaConnection.h>
#include <om/DirectRouter.h>
#include <om/Terminal.h>
#include <om/PubSubManager.h>
#include <om/OMLoadManager.h>
#include <om/MigrationPolicy.h>

// enable the remote execution terminal
#define ENABLE_TERMINAL 1

// the -v verbosity level needed for CheckInvariants() to run
#define CHECK_INVARIANTS_DBG_LEVEL 10

// enable the load balance object placement test
//#define ENABLE_LB_TEST 1

struct MsgAppRemoteUpdate;
struct MsgAppRegister;
struct MsgAppFetchReq;
struct MsgAppFetchResp;
struct MsgAppUpdate;
struct MsgAppUpdateAck;
struct MsgMigrateReq;

class EventDemux;

using namespace ManagerParams;

#define PERIODIC(time_msec, now, blurb)   \
do {  \
    static TimeVal last = TIME_NONE; \
	if (ManagerParams::FREQUENT_MAINTENANCE || \
		(last + (time_msec)) <= (now)) { \
	    blurb ; last = (now); \
	} \
} while (0)

inline uint32 _time_left_msec(uint64 stoptime_usec) {
    uint64 now = CurrentTimeUsec();
    return now > stoptime_usec ? 0 : (uint32) ((stoptime_usec - now) / USEC_IN_MSEC);
}

#define TIME_STOP_USEC(timeout_msec) (CurrentTimeUsec() + (timeout_msec)*USEC_IN_MSEC)
#define TIME_LEFT_MSEC(stoptime_usec) _time_left_msec(stoptime_usec)
#define TIMED_OUT_USEC(stoptime_usec) (CurrentTimeUsec() > (stoptime_usec))

///////////////////////////////////////////////////////////////////////////////
///// Utility Containers

typedef map<SID, MsgAppRemoteUpdate *, less_SID> RemoteUpdateMap;
typedef RemoteUpdateMap::iterator RemoteUpdateMapIter;

typedef map<SID, MsgAppRegister *, less_SID> RegisterMap;
typedef RegisterMap::iterator RegisterMapIter;

///////////////////////////////////////////////////////////////////////////////
///// Migration Data Structures

/**
 * Migration status information used while migration is in progress.
 * The same structure is used by both the sender and receiver of the
 * migration.
 */
class MigrationInfo : public TimedStruct {
 public:
    sid_t   from, to;
    GUID guid;
    SIDDeltaMap *registrations;

    MigrationInfo() : TimedStruct((uint32)0), registrations(NULL) {}
    MigrationInfo(uint32 ttl_msec, TimeVal now = TIME_NONE) :
	TimedStruct(ttl_msec, now), registrations(NULL) {}
    virtual ~MigrationInfo() { delete registrations; }
};

typedef map<guid_t, MigrationInfo, less_GUID> MigrationInfoMap;
typedef MigrationInfoMap::iterator MigrationInfoMapIter;

typedef map<GUID, TimedGUID *, less_GUID> DeletedSet;
typedef DeletedSet::iterator DeletedSetIter;

typedef map<GUID, TimedSID *, less_GUID> FwdPtrMap;
typedef FwdPtrMap::iterator FwdPtrMapIter;

struct MigrationPending {
    GUID guid;
    SID target;

    MigrationPending(GUID guid, SID target) :
	guid(guid), target(target) {}
};

typedef list<MigrationPending> MigrateQueue;
typedef MigrateQueue::iterator MigrateQueueIter;

///////////////////////////////////////////////////////////////////////////////

struct OMAttrInfo {
    Constraint cons;
    mpf_class  range;
    pair<real64,real64> asreals;

    OMAttrInfo() : cons(0, Value(0), Value(0)) {}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * The Object management interface. Used as follows:
 *
 * Initialized along with MercuryNode with:
 *
 * <code>InitializeMercury(...);</code>
 *
 * Then obtain an instance using:
 *
 * <code>Manager *manager = Manager::GetInstance(myAdaptor);</code>
 *
 * Then, you should periodically call the function:
 *
 * <code>manager->Recv();</code>
 *
 * at a rate at least as fast as the game frame rate (better if faster).
 *
 * Each time the game state changes (i.e., once per frame), you should
 * call:
 *
 * <code>manager->Send();</code>
 *
 * When constructing new objects to add to the object store, you should
 * use <code>manager->CreateGUID()</code> as a unique GUID to assign to
 * the object. After adding the object to the store, inform the Manager
 * using <code>manager->AddGUID(guid)</code>. When you remove a primary
 * object from the store, inform the Manager using 
 * <code>manager->DeleteGUID()</code>.
 */
class Manager {
 private:

    friend class DirectRouter;
    friend class MigrationPolicy;
    friend class RandomPlacementPolicy;
    friend class RegionPlacementPolicy;
    friend class LoadBalancedPolicy;
    friend class OMLoadManager;
    friend class Terminal;

    friend class TerminalWorker;
    friend class ReplicaConnectionSender;
    friend class ReplicaConnectionReceiver;
    friend class GameAdaptor;
    friend class GObject;
    friend class GObjectInfo;
    friend class ObjectStore;
    friend class OMLoadEvent;
    friend class OMLoadInterest;

    // WARN on some really pathological cases
    static const uint32 MAX_PUBCOUNT_WARN   = 256;
    static const uint32 MAX_SUBCOUNT_WARN   = 128;
    static const uint32 MAX_MATCHCOUNT_WARN = 512;

    void _CheckSpecialInvariant ();
 protected:

    // singleton pattern
    static Manager *m_Instance;

    ///////////////////////////////////////////////////////////////////////////
    ///// Object Pools

    ClassChunker<InterestLink> *m_InterestLinkPool;
    ClassChunker<GObjectInfo>  *m_GObjectInfoPool;
    ClassChunker<TimedGUID>    *m_TimedGUIDPool;
    ClassChunker<TimedSID>     *m_TimedSIDPool;

    ///////////////////////////////////////////////////////////////////////////
    ///// Network

    WANMercuryNode *m_PubSubRouter;
    EventDemux     *m_PubSubEvents;
    DirectRouter   *m_DirectRouter;

    OMLoadManager *m_LoadManager;
    PubSubManager *m_PubSubManager;
    hash_map<string, int, hash_string> m_AttrNameMap;
    hash_map<int, OMAttrInfo> m_AttrInfo;
    /*
      hash_map<int, Constraint> m_HubConstraints;
      hash_map<int, mpf_class> m_HubRanges;
      hash_map<int, pair<real64,real64> > m_HubConstraintsAsReals;
    */

    ///////////////////////////////////////////////////////////////////////////
    ///// Replica Maintenance Connections

    OutgoingReplicaConnectionMap m_ReplicaConnSenderMap;
    IncomingReplicaConnectionMap m_ReplicaConnReceiverMap;

    ///////////////////////////////////////////////////////////////////////////
    ///// Object Management

    // Hard State
    GUIDSet         m_PendingFetchReqs; // pending requests we have sent
    GObjectMap      m_ObjectArrivals;   // new objects that have arrived
    RegisterMap     m_ToRegister;       // new objects to register interest in

    // Soft State (mostly)
    // The object interest graph -- maintains the things our primary objects
    // are interested in (make sure we keep this stuff up to date), and
    // keeps the book keeping around for the replicas that are interested
    // in us.
    //
    // NOTE: objects in this map do not necessarily have to be in our
    // object store or pending store (it means we want to get them). 
    // Objects that aren't in this map may be in our object or pending store.
    // We do keep the "info" and the "object" loosely coupled so if we
    // garbage collect one, it doesn't "break" the other.
    GObjectInfoMap  m_ObjectInfo;

    // Soft State (managed by HandleReplicaManagementPeriodic())
    DeletedSet      m_DeletedGUIDs;     // primaries which were deleted here
    FwdPtrMap       m_ForwardingPtrs;   // ptrs to migrated objects (cache)

    // TODO: all the above soft-state containers should have an auxiliary
    // prio_queue ordered by timeout, so that expiration is handled more
    // efficiently (e.g., not O(N) time).

    ///////////////////////////////////////////////////////////////////////////
    ///// Migration Protocol Management

    // policy for when to do migration
    MigrationPolicy *m_MigrationPolicy;

    // objects to migrate during the next call to Send()
    MigrateQueue m_MigrateQueue;

    // objects that just migrated to us; we have not yet added them
    // as replicas to our object store (may have unresolved pointers
    MigrationInfoMap m_MigrateIncoming;

    ///////////////////////////////////////////////////////////////////////////
    ///// My Information

    GameAdaptor *m_Game;
    guid_t m_UniqueGUID;

    ///////////////////////////////////////////////////////////////////////////
    ///// Remote Terminal

    Terminal *m_Terminal;
    // big phat lock for Send() Recv(), only used for the
    // Terminal since everything else is single threaded
    Mutex m_ManagerLock;

    ///////////////////////////////////////////////////////////////////////////
    ///// Private Member Functions

    ///// Mercury Interface
    void InitializeHubConstraint(OMAttrInfo& info);
    void InitializeHubConstraints();

    ///// Object Pointer Resolution
    void Resolve(GObject *obj);
    int  MarkResolvable(GObjectMap *new_map,
			GUIDSet *resolved, 
			GUIDSet *unresolved,
			GUIDSet *temp_resolved,
			GObject *obj);

    ///// Fetching Full Copies of Objects
    void Fetch(SIDMap *tofetch, GUID ignore);
    void HandleFetchReq(MsgAppFetchReq *req);
    void HandleFetchResp(MsgAppFetchResp *resp);

    ///// Incoming Publications, Deltas, and Updates
    void HandleEvent(OMEvent *ev, bool cached);
    bool HandlePub(OMEvent *ev, bool cached);

    void HandleUpdate(MsgAppUpdate *);
    void HandleUpdateAck(MsgAppUpdateAck *);
    void HandleRemoteUpdate(MsgAppRemoteUpdate *);

    ///// Management of Remote Replicas
    void HandleReplicaRegister(MsgAppRegister *msg);
    void HandleReplicaPeriodic(TimeVal& now);
    void HandleReplicaConnPeriodic(TimeVal& now);
    MsgAppRegister *GetCurrRegisterMsg(SID sid);

    ///// Migration Protocol
    void PerformMigrations();
    void _PerformMigration(GObject *obj, SID target);
    void HandleMigrateReq(MsgMigrateReq *req);
    void HandleMigrateOk(guid_t guid);

    ///// Migration Heuristics
    void ProcessMigrateMap(MigrateMap& todo);
    void HandleMigrateConsider(TimeVal& now);
    void HandleMigrateConsiderFreq(TimeVal& now);

    ///// Soft State Management
    void HandleObjectInfoPeriodic(TimeVal& now);
    void HandleDeletedGUIDsPeriodic(TimeVal& now);
    void HandleForwardingPtrsPeriodic(TimeVal& now);

    ///// Interest Management
    //void PubSubProcessLatestInterests();
    //void PubSubInterestsPeriodic(TimeVal& now);

    ///// Load Prediction Management
    void RecordLoadInfo(MsgApp *msg);
    //void RecordLoadInfo(OMEvent *evt);
    //void HandleLoadPeriodic(TimeVal& now);

    ///// Object status
    virtual bool IsIncomingMigration(guid_t guid) {
	return m_MigrateIncoming.find(guid) != 
	    m_MigrateIncoming.end();
    }
    virtual bool IsDeleted(guid_t guid) {
	return m_DeletedGUIDs.find(guid) != m_DeletedGUIDs.end();
    }

    // Resolve references for objects in todo
    void RegisterObjects(GObjectMap *todo);

    // Allocates nodes in the interest graph and deals with the interest links
    GObjectInfo *AllocGObjectInfo(GUID guid, SID sid, bool isReplica);
    void FreeGObjectInfo(GObjectInfo *info, bool deallocate = true);

    TimedGUID *AllocTimedGUID(GUID guid, uint32 ttl);
    void FreeTimedGUID(TimedGUID *tguid);
    TimedSID *AllocTimedSID(SID sid, uint32 ttl);
    void FreeTimedSID(TimedSID *tsid);
    //NodeLoadInfo *AllocNodeLoadInfo(SID sid);
    //void FreeNodeLoadInfo(NodeLoadInfo *info);

    // Look for an object in the object store, then in the pending store
    GObject *FindObject(GUID& guid);

    // Called to add a new Replica to the Pending Store
    void AddReplica(GObject *obj);

    // For the migration policy
    virtual GameAdaptor *GetGame() { return m_Game; }
 public:
    virtual GObjectInfoMap *GetObjectInfo() { return &m_ObjectInfo; }
 protected:
    virtual ObjectStore    *GetObjectStore() { return m_Game->GetObjectStore(); }
    virtual ObjectStore    *GetPendingStore() { return m_Game->GetPendingStore(); }
    virtual NodeLoadMap    *GetNodeLoadMap() { return m_LoadManager->GetLoadMap(); }
    virtual GObjectMap     *GetObjectArrivals() { return &m_ObjectArrivals; }

    /** Dummy constructor */
    Manager();

    /**
     * This should be called when a new primary object is added to the object
     * store by the application. This should be called *AFTER* the object
     * is actually added to the object store.
     *
     * This notification is necessary to setup the interest management stuff.
     *
     * This is now an automatic call-back from the GameAdaptor.
     */
    virtual void AddObject(GObject *obj);

    /**
     * This should be called when an object is permanently deleted.
     * We require the application to call this to ensure that we maintain
     * enough state about the GUID to inform other nodes that the guid
     * is gone.
     *
     * This is now an automatic call-back from the GameAdaptor. It is
     * called just before the object is actually removed.
     */
    virtual void DeleteObject(GObject *obj);

 public:

    // Terminal Thread Locking -- needed if you the object store
    inline void Lock() { 
#ifdef ENABLE_TERMINAL
	m_ManagerLock.Acquire();
#endif
    }
    inline void Unlock() { 
#ifdef ENABLE_TERMINAL
	m_ManagerLock.Release(); 
#endif
    }

    MercuryNode *GetMerc() {
	return m_PubSubRouter;
    }

    DirectRouter *GetDirectRouter () {
	return m_DirectRouter;
    }

    EventDemux *GetDemux() {
	return m_PubSubEvents;
    }

    /**
     * Constructs an instance of the manager that interfaces with the game.
     * Will initialize the required network connections.
     */
    Manager(GameAdaptor *game, bool startNetwork=true);

    static Manager *GetInstance(GameAdaptor *game);
    static Manager *GetInstance();
    static void DestroyInstance() {
	if (m_Instance)
	    delete m_Instance;
	m_Instance = NULL;
    }

    virtual ~Manager();

    /**
     * Send new events and process pending ones. The manager guarantees that
     * it maintains no game state inbetween calls to this function. Hence,
     * the GameAdaptor is only required to maintain consistent views of its
     * object store and GObjects to the Manager during this function's
     * execution.
     * <p>
     * This function just calls Send() and then Recv(). You can ise the
     * individual functions when you are certain that you don't have stuff
     * to send (e.g., nothing was updated by the app).
     */
    virtual void Exec(uint32 timeout = 60000);

    /**
     * Send new events that are triggered by modifications to the game state,
     * and send keep-alives for replicas that we are maintaining.
     */
    virtual void Send(uint32 timeout = 60000);

    /**
     * Receive and process pending events from the network and modify the
     * game state accordingly. This function is also used to perform
     * the periodic "management" of manager state such as retransmissions
     * and garbage collection. Hence, this function should be called
     * frequently.
     *
     * XXX: Recv is not the best name for this function...
     */
    virtual void Recv(uint32 timeout = 60000, bool doMaintenance = true);

    /**
     * Perform periodic maintenance. Should be called ~once per frame.
     */
    virtual void Maintenance(uint32 timeout = 60000);

    /**
     * Just make Mercury tick once, without doing any periodic OM stuff
     */
    virtual void Merc(uint32 timeout = 60000);

    /**
     * Return a new globally unique ID that can be assigned to an object.
     */
    guid_t CreateGUID();

    /**
     * Get the unique SID of this node.
     */
    sid_t GetSID();

    //private:
    /**
     * Convert from an application data value to a Mercury range value.
     */
    Value ConvertToValue(int attr, real64 from, real64 min, real64 max);
    Value ConvertToValue(int attr, real64 from);

    /**
     * Convert from an application data value to a Mercury range value.
     *
     * XXX: can we get numerical error when doing casts here???
     */
    real64 ConvertFromValue(int attr, const Value& from, real64 min, real64 max);
    real64 ConvertFromValue(int attr, const Value& from);
    //public:
    int GetAttrIndex(const string& attr);
    Value ConvertToValue(const string& attr, real64 from, real64 min, real64 max);
    Value ConvertToValue(const string& attr, real64 from);
    real64 ConvertFromValue(const string& attr, const Value& from, real64 min, real64 max);
    real64 ConvertFromValue(const string& attr, const Value& from);

    OMInterest *CreateInterest();
    void FreeInterest(OMInterest *in);
    OMEvent *CreateEvent(const GUID& guid);
    void FreeEvent(OMEvent *ev);

    /**
     * Migrate an object to another node. Assumes the object IsMigratable().
     * This method doesn't guarantee that the migration succeeds, only that
     * it is attempted.
     */
    virtual void Migrate(GObject *obj, sid_t target);

    /**
     * Return the current load value on this node. This is something like
     * outbound + inbound bandwidth, plus maybe some cpu load.
     */
    CostMetric GetLoad(TimeVal& now);
    CostMetric GetLoadTarget();
    CostMetric GetHighWaterMark();
    CostMetric GetLowWaterMark();
    CostMetric GetCapacity();
    void       SetLoad(CostMetric load) {
	m_LoadManager->SetLoad(load);
    }

    ///////////////////////////////////////////////////////////////////////////
    ///// DEBUGGING

    //bool m_UseDummyLoad;
    //CostMetric m_DummyLoad;

#ifdef DEBUG
    char m_OutputBuffer[1024*1024];
#else
    char m_OutputBuffer[1]; // unsuable!
#endif

    const char *DumpInterestGraphToGraphViz();
    const char *DumpInterestGraph();
    const char *DumpObjectStore();
    const char *DumpObjectInfo();
    const char *DumpNodeLoadMap();
    const char *DumpForwardingPtrs();
    const char *DumpDeletedGUIDs();
    const char *DumpReplicaConnections();

    void CheckAlwaysInvariants();
    void CheckEntryInvariants();
    void CheckExitInvariants();

#ifdef ENABLE_LB_TEST
    // doesn't work anymore...
    static void DoLoadBalancedPolicyTest(int argc, char **argv);
    void MakeStableLink(GObjectInfo *from, GObjectInfo *to);
    bool LoadBalancedPolicyTest(int argc, char **argv);

#endif

};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
