////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <om/GUpdate.h>
#include <mercury/ObjectLogs.h>

GUpdate::GUpdate() : isUpdate(false), isNew(false), isDelete(false),
			 isProactive(false), guid(GUID_NONE), delta(NULL) {
    OBJECT_LOG_DO( nonce = 0 );
  }

GUpdate::GUpdate(bool isUpdate, GUID guid) : 
    isUpdate(isUpdate), guid(guid), isNew(false), isDelete(false), 
    isProactive(false), mask(DELTA_MASK_NONE), delta(NULL) {
    OBJECT_LOG_DO( nonce = 0 );
}

GUpdate::GUpdate(const GUpdate& other) :
    isUpdate(other.isUpdate), guid(other.guid), isNew(other.isNew),
    isDelete(other.isDelete), isProactive(other.isProactive),
    mask(other.mask), delta(other.delta ? new Packet(*other.delta) : NULL)
{
}

GUpdate::~GUpdate() {
    if (delta) {
	delete delta;
    }
}

GUpdate::GUpdate(Packet *pkt) {
    byte flags  = pkt->ReadByte();
    isUpdate    = flags & 1;
    isNew       = flags & 2;
    isDelete    = flags & 4;
    isProactive = flags & 8;
    guid = GUID(pkt);

    if (!isDelete) {
	mask = DeltaMask(pkt);
	delta = new Packet(pkt);
	ASSERT(delta->GetUsed () > 0);
    } else {
	delta = NULL;
    }

    OBJECT_LOG_DO( nonce = pkt->ReadInt() );
}

void GUpdate::Serialize(Packet *pkt) {
    byte flags = 0;
    if (isUpdate)
	flags |= 1;
    if (isNew)
	flags |= 2;
    if (isDelete)
	flags |= 4;
    if (isProactive)
	flags |= 8;
    pkt->WriteByte(flags);

    ASSERT(guid != GUID_NONE);
    guid.Serialize(pkt);

    if (!isDelete) {
	mask.Serialize(pkt);
	ASSERT(delta);
	ASSERT(delta->GetUsed () > 0);
	delta->Serialize(pkt);
    }

    OBJECT_LOG_DO( pkt->WriteInt(nonce) );
}

uint32 GUpdate::GetLength() {
    uint32 length = 0;

    length += 1;
    length += guid.GetLength();

    if (!isDelete) {
	length += mask.GetLength();

	ASSERT(delta);
	length += delta->GetLength();
    }

    OBJECT_LOG_DO( length += 4 );

    return length;
}

void GUpdate::Print(FILE *stream) {
    ASSERT(false);
}

ostream& operator<<(ostream& out, GUpdate *update)
{
    out << "(GUpdate " << update->guid
	<< " isUpdate=" << update->isUpdate
	<< " isNew=" << update->isNew
	<< " isDelete=" << update->isDelete
	<< " isProactive=" << update->isProactive;
    if (!update->isDelete) {
	out << " mask=" << update->mask;
	if (update->delta)
	    out << " data=[" << (update->delta->GetLength()-4) << "bytes]";
    }
    out << ")";
    return out;
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
