////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __GOBJECT_H__
#define __GOBJECT_H__

#include <map>
#include <om/common.h>
#include <om/OMEvent.h>
#include <om/OMInterest.h>
#include <om/GUpdate.h>
#include <om/GObjectInfo.h>

///////////////////////////////////////////////////////////////////////////////

/**
 * This interface is used to refer to "pointers" to GObjects. When a GObject
 * is distributed, it might have dangling references, in which case this is
 * used by the application to indicate what reference is dangling. The
 * Manager doesn't ever need to inspect the guts of this reference; when
 * it wants to resolve the reference, it just passes the resolved
 * GObject pointer and this to the object with the dangling pointer.
 */
class GObjectRef {
public:
    virtual ~GObjectRef () {}
    virtual GUID GetTarget() = 0;
};

typedef list<GObjectRef *> GObjectRefList;
typedef GObjectRefList::iterator GObjectRefListIter;

///////////////////////////////////////////////////////////////////////////////

/**
 * The interface that objects managed by Colyseus must implement. There are
 * basically 4 things that must be implemented:
 *
 * <ol>
 * <li>PackUpdate - marshalling delta encoded state.
 * <li>UnpackUpdate - unmarshalling delta encoded state.
 * <li>FillPub - get the object's current "location" in game space.
 * <li>FillInterest - get the object's current "area of interest" in
 *     game space.
 * </ol>
 */
class GObject {
 private:

    friend class Manager;
    friend class OMLoadManager;
    friend class ReplicaConnectionSender;
    friend class ReplicaConnectionReceiver;
    friend class GameVisualizer; // XXX HACK

    DeltaMask m_DeltaMask;
    DeltaMask m_RemoteDeltaMask;
    GObjectInfoIface *m_Info;
    
    bool m_Deleted;

 protected:

    inline GObjectInfoIface *GetInfo() { return m_Info; }

    inline void SetSID(sid_t sid) {
	m_Info->SetSID(sid);
    }
    inline void SetIsReplica(bool isReplica) { 
	m_Info->SetIsReplica(isReplica); 
    }

    inline bool IsRemoteDirty() {
	// state changes, or events to pass to clients...
	return m_RemoteDeltaMask != DELTA_MASK_NONE;
    }
    inline void SetRemoteDeltaMask(const DeltaMask& mask) {
	m_RemoteDeltaMask = mask;
    }
    inline void SetRemoteDeltaMaskField(uint32 index) {
	m_RemoteDeltaMask.Set(index);
    }

 public:

    /** 
     * Construct a new local object 
     */
    GObject(guid_t guid);

    /**
     * Construct a replica object from the network
     */
    GObject(GObjectInfoIface *info);

    virtual ~GObject();

    /**
     * If the game wants to keep an object around even after it is
     * "deleted" from Colyseus's viewpoint, then it can call this
     * function instead of the descructor. When calling the destructor
     * later, it will do the right thing. A deleted object must
     * already be removed from the object store (obviously, otherwise
     * the Manager can still reference it).
     *
     * You should NOT use any methods of this API on deleted
     * objects -- doing so will cause Bad Things.
     *
     * The destructor will automatically call this if not called already.
     */
    void Delete();

    /**
     * True if Delete() was called and this object is no longer registered
     * with Colyseus.
     */
    bool IsDeleted();

    ///////////////////////////////////////////////////////////////////////////
    // COLYSEUS API

    /**
     * The GUID uniquely identifying the object.
     */
    inline const guid_t GetGUID() { return m_Info->GetGUID(); }

    /**
     * The SID that identifies the current location of the primary
     */
    inline const sid_t  GetSID() { return m_Info->GetSID(); }

    /**
     * True iff this is a replica and not a primary
     */
    inline bool IsReplica() { return m_Info->IsReplica(); }

    ///////////////////////////////////////////////////////////////////////////
    // MERCURY API

    /**
     * True if the object was just created. Should only be called once
     * per call to Send(). If this is true when Send() is called, then a
     * full replica of the object will be sent along with the publication.
     * Must enabled ManagerParams::PUBLISH_NEW_DELTA for it to take effect.
     *
     * XXX: rename me!
     */
    virtual bool IsNew() = 0;

    /**
     * True if this object is "attached" to some other object(s) and if someone
     * is interested in the other objects, then they are surely interested
     * in this object. Therefore we will proactively replicate this object
     * to the nodes which have replicas of the other objects. May be
     * dynamic.
     *
     * This is an optimization to ensure quick replication of newly created
     * objects. E.G., projectiles that originate from a known existing object.
     *
     * @param others the objects this one is attached to.
     */
    virtual void AttachedTo(GUIDList *others) { }

    /**
     * True if this object has some objects attached to it and if someone
     * is interested in this object, it is surely interested in the others.
     * This is just an analog to AttachedTo() and applications can implement
     * whichever is more convenient (or both).
     *
     * This is an optimization to ensure quick replication of newly created
     * objects. E.G., projectiles that originate from a known existing object.
     *
     * @param others the objects that are attached to this.
     */
    virtual void AttachTo(GUIDList *others) { }

    /**
     * Get the publications for this object we wish to register and unregister.
     *
     * @param reg    the new publications to register
     *               the application should call Manager::CreateEvent()
     *               to allocate an event.
     * @param unreg  the publications in curr to unregister.
     * @param curr   the set of current publications that are still in the
     *               system.
     */
    virtual void FillEvents(EventList *reg,
			    EventList *unreg,
			    EventSet  *curr) = 0;

    /**
     * Given a publication that matched one of this object's subscriptions,
     * if this object is interested in the event, then return a non-zero
     * time for how long this interest should last before it has to be
     * refreshed. This will be called once per frame with that event
     * until the event expires (or we obtain the object).
     *
     * @param ev the event to determine if we are interested
     * @param curr the set of current subsciptions our app has in the system
     *
     * @return the length of time the interest should last from now 
     *         until it needs to be refreshed (either by receiving a
     *         new event in which case this function will be called
     *         again, or by a callback with InterestTime(GObject *)
     *         just before the interest expires. In milliseconds.
     *         A value of 0 means we are not interested.
     */
    virtual uint32 InterestTime(OMEvent *ev, InterestMap *curr) = 0;

    /**
     * Given another object, determine how long we should remain interested
     * in that object before it has to be refreshed. This will be called
     * when we receive an Event for this object or when the object is
     * about to expire.
     *
     * @return the length of time the interest should last from now
     *         until it needs to be refreshed. In milliseconds.
     *         A value of 0 means we are not interested.
     */
    virtual uint32 InterestTime(GObject *obj, InterestMap *curr) = 0;

    ///////////////////////////////////////////////////////////////////////////
    // DELTA ENCODING AND UPDATES

    /**
     * Clear all the bits in the delta mask.
     */
    inline void ClearDeltaMask() {
	m_DeltaMask.Clear();
    }

    /**
     * Get a reference to the delta mask.
     */
    inline DeltaMask& GetDeltaMask() {
	return m_DeltaMask;
    }

    /**
     * See if the object is dirty (relative to local updates).
     */
    inline bool IsDirty() {
	// state changes, or events to pass to clients...
	return m_DeltaMask != DELTA_MASK_NONE;
    }

    /**
     * Set the entire delta mask.
     */
    inline void SetDeltaMask(const DeltaMask& mask) {
	m_DeltaMask = mask;
    }

    /**
     * Set a bit in the delta mask.
     */
    inline void SetDeltaMaskField(uint32 index) {
	m_DeltaMask.Set(index);
    }

    /**
     * Return a DeltaMask for (de)serializing the entirety of the object
     * (e.g., when we need to construct a completely new copy).
     */
    virtual const DeltaMask& GetInitDeltaMask() = 0;

    /**
     * Clear all the bits in the delta mask of remote updates. This should be
     * called by the application once it has finished processing all updates
     * made by the manager (via remote hosts). Only relevant for replicas.
     */
    inline void ClearRemoteDeltaMask() {
	m_RemoteDeltaMask.Clear();
    }

    /**
     * Get a reference to the delta mask of remote updates since last clear.
     * This mask tells the application what bits have been changed during calls
     * to Manager::Recv(). Only relevant for replicas.
     */
    inline DeltaMask& GetRemoteDeltaMask() {
	return m_RemoteDeltaMask;
    }

    /**
     * Fill a buffer with the object delta. The delta mask enumerates the 
     * fields that are out of date for the remote replica. (i.e., this is
     * the cumulative OR of all GetDeltaMask() of lost updates.
     */
    virtual void PackUpdate(Packet *buffer, const DeltaMask& mask) = 0;

    /**
     * Unpack an object delta from a buffer. The delta mask enumerates
     * the parts of the objects that were changed. If this update would
     * result in unresolved references, they should be placed in
     * unresolved so they can be fetched. Note: the update should
     * still be applied (or "saved") by the object, because it will not
     * get the update again (future updates won't count it as a delta)
     */
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
			      SIDMap *unresolved) = 0;

    /**
     * Get the set of all references this object has to other objects.
     * Only required for primaries. The Manager ensures that if an object
     * maintains a remote reference in this list, the object will not be
     * deleted if it goes out of the area of interest scope. However, it
     * *will* be deleted if the object is actually deleted by the primary.
     */
    virtual void GetAllRefs(GObjectRefList *tofill) {}

    /**
     * Get the set of all unresolved references to other objects. The
     * default implementation assumes there are no pointers. The
     * Manager guarantees that it will only hold the GObjectRefs in
     * the list until the end of a Send() or Recv() (i.e., it will
     * relinquish them when the application resumes, so the
     * application can safely delete them). If you implement this
     * function, you must also implement ResolveRef().
     *
     * @param tofill a list of GObjectRefs (dangling pointers)
     */
    virtual void GetRefs(GObjectRefList *tofill) {};

    /**
     * Resolve a dangling pointer. I.E., if GetRefs() returns a non-empty
     * list of dangling pointers, when we resolve those pointers, the
     * Manager will call this function with the ref pointer and the resolved
     * object pointer. When this is called, the Manager will have control
     * within a Send() or Recv(). It will only be called on GObjectRefs that
     * are returned by GetRefs within the same Send() or Recv().
     *
     * @param ref the reference to the dangling pointer
     * @param obj the resolved reference. If we were told the object was
     * *deleted* (and hence, no longer exists) then this will be NULL.
     */
    virtual void ResolveRef(GObjectRef *ref, GObject *obj) {
	// default implementation assumes no obj-obj pointers
	ASSERT(0);
    };

    ///////////////////////////////////////////////////////////////////////////
    // MIGRATION API (OPTIONAL)

    /**
     * This returns the region ID of the region the object is in. Used
     * for region based migration.
     */
    virtual uint32 GetRegion() { return 0; }

    /**
     * True if the object management layer is allowed to migrate this
     * object to another node in the system. This value may be
     * dynamic, but may not change during a migration attempt.
     */
    virtual bool IsMigratable() { return true; }

    /**
     * Obtain the expected lifetime of an object. Some object life times
     * (like computer generated temporaries) are predictable, so this can
     * help prediction. In particular we use this for object migration
     * hystericies (there is no point in migrating an object that will die
     * in the near future). Return 0 if the expected lifetime is unknown.
     */
    virtual uint32 ExpectedLifetime() { return 0; }

    /**
     * This callback is invoked when the Manager wants to begin the
     * migration process for this object. This will only be called
     * if IsMigratable() returns true. This object remains a primary
     * on this node until CommitMigration() is called.
     *
     * @param target the target node that the manager will migrate this
     * object to. This is just a hint and probably doesn't need to be
     * used by the application until we want to deal with client migration.
     * In that case, the application needs to coordinate the migration
     * of the client connection.
     * @return the callback should return an amount of time that serves
     * as a hint as to how long the Manager should attempt the migration
     * before giving up and canceling. (milliseconds)
     */
    virtual uint32 BeginMigration(sid_t target) { return 1000; }

    /**
     * This callback is invoked when this object is in the process of
     * migration (BeginMigration() was called) and the Manager cancels
     * the migration attempt. This will never be called after
     * CommitMigration() is invoked.
     *
     * @param target the node we were in the process of migrating to.
     */
    virtual void CancelMigration(sid_t target) { return; }

    /**
     * This callback is invoked when the migration to the remote node
     * is completed (this is not exactly true, but it is when the migration
     * process has moved to a point where it can not be canceled).
     * This is called just *before* isReplica and SID variables are toggled.
     *
     * @param target the node we migrated to. This object's SID should
     * now be this.
     */
    virtual void CommitMigration(sid_t target) { return; }

    ///////////////////////////////////////////////////////////////////////////
    // LOAD PREDICTION (XXX TODO: auto-calc this)

    /**
     * This method should return the fixed "cost" of this object,
     * which is used in load prediction. For example, the outbound
     * bandwidth consumed (since currently our only metric of interest
     * is outbound bandwidth) by the object if it is a Client.
     *
     * Default implementation calls GetDeltaCost().
     * XXX FIXME? Should be ~delta cost + all our interests.
     */
    virtual CostMetric GetFixedCost() { return GetDeltaCost(); }

    /**
     * This method should return the delta "cost" of this object,
     * which is used in load prediction. I.E., this is
     * avg_delta_size/delta_rate in Bps.  The default implementation
     * has the Manager estimate this.
     */
    virtual CostMetric GetDeltaCost()  { 
	return static_cast<GObjectInfo *>(m_Info)->GetCostEstimate()->GetEstimate();
    }

    ///////////////////////////////////////////////////////////////////////////
    // DEBUGGING

    /**
     * Optional interface child classes can implement to output the game
     * related fields of the object to the output stream. This will be
     * appended to the output when the GObject object is used as the rhs
     * of operator<<.
     *
     * @param out the output stream to write to.
     */
    virtual void PrintFields(ostream& out) {}
};

ostream& operator<<(ostream& out, GObject *objp);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
