////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __EVENT_DEMUX_H__
#define __EVENT_DEMUX_H__

#include <hash_map.h>
#include <list>
#include <mercury/MercuryNode.h>
#include <mercury/Event.h>

///////////////////////////////////////////////////////////////////////////////

typedef hash_map< EventType, list<Event *>* > EventQueueMap;
typedef EventQueueMap::iterator EventQueueMapIter;

/**
 * Lazy event demultiplexor for reading Events of different types from
 * MercuryNode. (Lazy because it doesn't do any work until you call
 * ReadEvent).
 */
class EventDemux {
 private:
    EventQueueMap m_Map;
    MercuryNode  *m_Node;

    /**
     * Reads events until we get an event of type 'until' or no more events.
     */
    void ReadEvents(EventType until) {
	Event *ev;
	while ((ev = m_Node->ReadEvent()) != NULL) {
	    EventType t = ev->GetType();
	    EventQueueMapIter q = m_Map.find(t);
	    list<Event *> *l;
	    if (q == m_Map.end()) {
		l = new list<Event *>;
		m_Map.insert(pair<EventType, list<Event *>* >(t, l));
	    } else {
		l = q->second;
	    }
	    l->push_back(ev);
	    if (t == until)
		return;
	}
    }
 public:
    EventDemux(MercuryNode *node) : m_Node(node) {}

    /**
     * Read one event of type 't' and cast it to type T. NULL if none.
     */
    template<class T>
	T *ReadEvent(EventType t) {
	EventQueueMapIter q = m_Map.find(t);

	if (q == m_Map.end() || q->second->size() == 0) {
	    ReadEvents(t);
	}

	q = m_Map.find(t);
	if (q == m_Map.end() || q->second->size() == 0) {
	    return NULL;
	} else {
	    T *next = static_cast<T *>( q->second->front() );
	    q->second->pop_front();
	    return next;
	}
    }
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
