////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __DELTA_MASK_H__
#define __DELTA_MASK_H__

#include <om/common.h>

// must be a multiple of 31
// 
#define MAX_DELTA_FIELDS 372   
/* Had to bump it because #entity_fields + #client_fields in quake3 = ~266...
 * Although we use bits for the clusters only, I am using the DeltaMask for 
 * some other purposes also */

    /**
     * Auto-resizing bit-mask with MAX_DELTA_FIELDS bits. The 32nd bit in each
     * word is a flag which is used to indicate if the next word is used (i.e.,
     * has bits set in it). We assume that the most "popular" fields are encoded
     * by bits near the front of the bitmask, so the mask rarely extends beyond
     * a single word. Thus, we assume that if the 32nd bit in a word is NOT set
     * then it marks the end of the mask (i.e., if the 5th word is included, then
     * the 32nd bit in the 1st, 2nd, 3rd, and 4th words must be set).
     */
class DeltaMask : public Serializable {
private:
    friend ostream& operator<<(ostream& out, const DeltaMask& mask);

    // the 32nd bit of each word is unused locally; only used when serialized
    uint32 bits[(MAX_DELTA_FIELDS + (MAX_DELTA_FIELDS/31))/32];
    int max_word; // largest word with bits set

public:
    DeltaMask() { Clear(); }
    DeltaMask(uint8 i00,     uint8 i01 = 0, 
	    uint8 i02 = 0, uint8 i03 = 0, 
	    uint8 i04 = 0, uint8 i05 = 0, 
	    uint8 i06 = 0, uint8 i07 = 0, 
	    uint8 i08 = 0, uint8 i09 = 0,
	    uint8 i10 = 0, uint8 i11 = 0, 
	    uint8 i12 = 0, uint8 i13 = 0, 
	    uint8 i14 = 0, uint8 i15 = 0, 
	    uint8 i16 = 0, uint8 i17 = 0, 
	    uint8 i18 = 0, uint8 i19 = 0,
	    uint8 i20 = 0, uint8 i21 = 0, 
	    uint8 i22 = 0, uint8 i23 = 0, 
	    uint8 i24 = 0, uint8 i25 = 0, 
	    uint8 i26 = 0, uint8 i27 = 0, 
	    uint8 i28 = 0, uint8 i29 = 0,
	    uint8 i30 = 0, uint8 i31 = 0) {
	Clear();
	Set(0, i00);
	Set(1, i01);
	Set(2, i02);
	Set(3, i03);
	Set(4, i04);
	Set(5, i05);
	Set(6, i06);
	Set(7, i07);
	Set(8, i08);
	Set(9, i09);
	Set(10, i10);
	Set(11, i11);
	Set(12, i12);
	Set(13, i13);
	Set(14, i14);
	Set(15, i15);
	Set(16, i16);
	Set(17, i17);
	Set(18, i18);
	Set(19, i19);
	Set(20, i20);
	Set(21, i21);
	Set(22, i22);
	Set(23, i23);
	Set(24, i24);
	Set(25, i25);
	Set(26, i26);
	Set(27, i27);
	Set(28, i28);
	Set(29, i29);
	Set(29, i29);
	Set(30, i30);
	Set(31, i31);
    }
    virtual ~DeltaMask() {}

    DeltaMask(const DeltaMask &other) {
	Clear();
	max_word = other.max_word;
	memcpy((void *)bits, (const void *)other.bits, sizeof(bits));
    }

    inline bool operator==(const DeltaMask& other) const {
	if (max_word != other.max_word)
	    return false;

	return !memcmp((const void *)bits, 
		(const void *)other.bits, 
		4*(max_word+1));
    }

    inline bool operator!=(const DeltaMask& other) const {
	return ! (*this == other);
    }

    inline void Merge(const DeltaMask& other) {
	// basically means: *this |= other;		
	max_word = MAX(max_word, other.max_word);
	for (int i=0; i<=max_word; i++) {
	    bits[i] |= other.bits[i];
	}
    }

    inline int MaxIndex() const {
	return (max_word+1)*31;
    }

    inline bool IsSet(int index) const {
	ASSERT(index < MAX_DELTA_FIELDS);
	index = index + index/31;

	return bits[index/32] & (uint32)(1 << (index%32));
    }

    inline void Set(int index) {
	ASSERT(index < MAX_DELTA_FIELDS);
	index = index + index/31;

	bits[index/32] |= (uint32)(1 << (index%32));

	max_word = MAX(max_word, index/32);
    }

    inline void Set(int index, ubyte val) {
	ASSERT(index < MAX_DELTA_FIELDS);
	index = index + index/31;

	if (val)
	    bits[index/32] |= (uint32)(1 << (index%32));
	else
	    bits[index/32] &= ~(uint32)(1 << (index%32));

	max_word = MAX(max_word, index/32);
    }

    inline void Clear() {
	memset((void *)bits, 0, sizeof(bits));
	max_word = 0;
    }

    DeltaMask(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *);
};

ostream& operator<<(ostream& out, const DeltaMask& mask);

extern const DeltaMask DELTA_MASK_NONE;

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
