////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  common.h

  Common game data structures and functions.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_COMMON_H__
#define __GAME_COMMON_H__

#include <mercury/common.h>
#include <om/OMInterest.h>
#include <om/OMEvent.h>
#include <util/types.h>
#include <math.h>

///////////////////////////////////////////////////////////////////////////////

typedef enum { GAME_OK, GAME_ERROR, GAME_EXIT } gRetCode;

///////////////////////////////////////////////////////////////////////////////

// We will identify static objects as objects with IP == GUID_NONE.
// This allows for 2^(16+32) total static objects in the system total
inline bool IsStaticGUID(const GUID& guid) {
    return guid.m_IP == 0;
}

// We will identify dynamic objects as objects with IP != GUID_NONE
// This allows for 2^(16+32) objects to be created per IP-address.
inline bool IsDynamicGUID(const GUID& guid) {
    return guid.m_IP != 0;
}


///////////////////////////////////////////////////////////////////////////////

// size of vector real numbers
typedef real32 VEC_TYPE;
// standard vec3_t array
typedef VEC_TYPE vec3_t[3];

/**
 * vec3_t as a serializable class + common operations
 */
class Vec3 : public Serializable {
 private:
    vec3_t elems;
 public:

    inline Vec3() {
	elems[0] = 0;
	elems[1] = 0;
	elems[2] = 0;
    }
    inline Vec3(VEC_TYPE x, VEC_TYPE y, VEC_TYPE z) {
	elems[0] = x;
	elems[1] = y;
	elems[2] = z;
    }
    inline Vec3(const Vec3& other) {
	elems[0] = other[0];
	elems[1] = other[1];
	elems[2] = other[2];
    }
    inline Vec3(const vec3_t& other) {
	elems[0] = other[0];
	elems[1] = other[1];
	elems[2] = other[2];
    }

    inline VEC_TYPE& operator[](std::size_t i) {
	return elems[i];
    }
    inline const VEC_TYPE& operator[](std::size_t i) const { 
	return elems[i]; 
    }

    //
    // Basic vector operations
    //
    // Modified from Quake2 game/q_shared.c
    //

    inline const bool operator == ( const Vec3& v ) const
	{
	    return (v[0]==elems[0] && v[1]==elems[1] && v[2]==elems[2]);
	}

    inline const bool operator != ( const Vec3& v ) const
	{
	    return !(v == *this);
	}

    Vec3 Perpendicular() const;
    Vec3 ProjectOnPlane(const Vec3& normal) const;
    Vec3 RotateAroundVector(const Vec3& dir, real32 rad ) const;

    //increment
    inline const Vec3& operator += ( const Vec3& v )
	{
	    elems[0]+=v.elems[0];
	    elems[1]+=v.elems[1];
	    elems[2]+=v.elems[2];
	    return *this;
	}

    //decrement
    inline const Vec3& operator -= ( const Vec3& v )
	{
	    elems[0]-=v.elems[0];
	    elems[1]-=v.elems[1];
	    elems[2]-=v.elems[2];
	    return *this;
	}

    //self-multiply
    inline const Vec3& operator *= ( const VEC_TYPE& s )
	{
	    elems[0]*=s;
	    elems[1]*=s;
	    elems[2]*=s;

	    return *this;
	}

    //self-divide
    inline const Vec3& operator /= ( const VEC_TYPE& s )
	{
	    const VEC_TYPE r = 1 / s;
	    elems[0] *= r;
	    elems[1] *= r;
	    elems[2] *= r;
	    return *this;
	}

    inline const VEC_TYPE Length() const 
	{
	    int	i;
	    VEC_TYPE length;

	    length = 0;
	    for (i=0 ; i< 3 ; i++)
		length += elems[i]*elems[i];
	    length = sqrt (length);

	    return length;
	}
    inline const VEC_TYPE Distance(const Vec3& to) const
	{
	    VEC_TYPE dx = elems[0]-to[0];
	    VEC_TYPE dy = elems[1]-to[1];
	    VEC_TYPE dz = elems[2]-to[2];

	    return sqrt(dx*dx+dy*dy+dz*dz);
	}
    inline const VEC_TYPE Dot(const Vec3& v2) const
	{
	    return elems[0]*v2[0] + elems[1]*v2[1] + elems[2]*v2[2];
	}
    inline const Vec3 Cross(const Vec3& v2) const 
	{
	    return Vec3(elems[1]*v2[2] - elems[2]*v2[1],
			elems[2]*v2[0] - elems[0]*v2[2],
			elems[0]*v2[1] - elems[1]*v2[0]);
	}
    inline const real32 Angle(const Vec3& v2) const
	{
	    return acos( this->Dot(v2)/( this->Length() * v2.Length() ) );
	}
    inline const Vec3 Normalize() const
	{
	    VEC_TYPE length, ilength;

	    length = elems[0]*elems[0] + elems[1]*elems[1] + elems[2]*elems[2];
	    length = sqrt (length);

	    if (length > 0) {
		ilength = 1/length;
		return Vec3(elems[0]*ilength,
			    elems[1]*ilength,
			    elems[2]*ilength);
	    } else {
		return Vec3();
	    }
	}
    inline const Vec3 Unit() const
	{
	    return Normalize();
	}
    inline const Vec3 Invert() const
	{
	    return Vec3(-elems[0],
			-elems[1],
			-elems[2]);
	}
    inline const Vec3 operator-() const 
	{
	    return Invert();
	}
    inline const Vec3 Scale(VEC_TYPE scale) const 
	{
	    return Vec3(elems[0]*scale,
			elems[1]*scale,
			elems[2]*scale);
	}
    inline const Vec3 operator*(VEC_TYPE scale) const
	{
	    return Scale(scale);
	}
    friend inline const Vec3 operator*(const VEC_TYPE scale, const Vec3& v)
	{
	    return v.Scale(scale);
	}
    inline const Vec3 Subtract(const Vec3& vecb) const 
	{
	    return Vec3(elems[0]-vecb[0],
			elems[1]-vecb[1],
			elems[2]-vecb[2]);
	}
    inline const Vec3 operator-(const Vec3& other) const 
	{
	    return Subtract(other);
	}
    inline const Vec3 operator-(VEC_TYPE v) const
	{
	    return Vec3(elems[0]-v,
			elems[1]-v,
			elems[2]-v);
	}
    inline const Vec3 Add(const Vec3& vecb) const
	{
	    return Vec3(elems[0]+vecb[0],
			elems[1]+vecb[1],
			elems[2]+vecb[2]);
	}
    inline const Vec3 operator+(const Vec3& other) const 
	{
	    return Add(other);
	}
    inline const Vec3 operator+(VEC_TYPE v) const
	{
	    return Vec3(elems[0]+v,
			elems[1]+v,
			elems[2]+v);
	}
    inline const Vec3 operator/(VEC_TYPE v) const
	{
	    return Vec3(elems[0]/v,
			elems[1]/v,
			elems[2]/v);
	}
    inline void Set(const VEC_TYPE x, const VEC_TYPE y, const VEC_TYPE z) 
	{
	    elems[0] = x;
	    elems[1] = y;
	    elems[2] = z;
	}

    Vec3(OMEvent *);
    void FillEvent(OMEvent *) const;

    Vec3(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *fp) { ASSERT(0); }

    static const Vec3 Random() {
	return Vec3( drand48() - 0.5, drand48() - 0.5, drand48() - 0.5 );
    }

    static const Vec3 ORIGIN;
    static const Vec3 ZERO;
};

ostream& operator<<(ostream& out, const Vec3& v);

///////////////////////////////////////////////////////////////////////////////

/**
 * bounding box + common operations
 */
class BBox : public Serializable {
 public:
    Vec3 min, max;

    inline BBox() {}
    inline BBox(const Vec3& pt) : min(pt), max(pt) {}
    inline BBox(const Vec3& min, const Vec3& max) : min(min), max(max) {}
    inline BBox(const BBox& other) : min(other.min), max(other.max) {}
#ifdef DEBUG
    virtual ~BBox() {
	for (uint32 i=0; i<3; i++) {
	    ASSERT(min[i] <= max[i]);
	}
    }
#endif

    BBox(OMInterest *);
    BBox(OMEvent *);
    void FillInterest(OMInterest *) const;
    void FillEvent(OMEvent *) const;

    real32 Volume() const;
    bool   Contains(const BBox& other) const;
    bool   Contains(const Vec3& pt) const;
    bool   Overlaps(const BBox& other) const;
    BBox   GetOverlap(const BBox& other) const;
    BBox   GetMerged(const BBox& other) const;

    BBox(Packet *pkt);
    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *fp) { ASSERT(0); }
};

template <class T>
void BBoxToType(T *elt, const BBox *bbox);
template <class T>
void TypeToBBox(BBox *bbox, T *elt);

ostream& operator<<(ostream& out, const BBox& v);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
