////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  common.cpp

  Common game data structures and functions.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include "common.h"
#include "GameManager.h"
#include "GameWorld.h"
#include <om/Manager.h>

static const char *g_AttrNames[] = { "x", "y", "z" };

const Vec3 Vec3::ORIGIN;
const Vec3 Vec3::ZERO;

Vec3::Vec3(Packet *pkt) {
    elems[0] = pkt->ReadFloat();
    elems[1] = pkt->ReadFloat();
    elems[2] = pkt->ReadFloat();
}

void Vec3::Serialize(Packet *pkt) {
    pkt->WriteFloat(elems[0]);	
    pkt->WriteFloat(elems[1]);
    pkt->WriteFloat(elems[2]);
}

uint32 Vec3::GetLength() {
    return 3*sizeof(VEC_TYPE);
}

Vec3::Vec3(OMEvent *ev) {
    Manager *m = GameManager::GetInstance()->GetManager();
    GameWorld *w = GameManager::GetInstance()->GetWorld();
    BBox ext = w->GetExtent();

    const Value *val;

    ASSERT(ev->IsPointEvent());

    // XXX HACK: if using DHT region keys, the first index is for the DHT key
    uint32 offset = 0;
    if (GameManager::publishAsDHTRegions) {
	offset = 1;
    }

    for (int i=0; i<3; i++) {
	elems[i] = 0;      // ASHWIN
	if (!w->PublishDim(i)) continue;

	const string name(g_AttrNames[i + offset]);
	int attr = m->GetAttrIndex(name);

	val = ev->GetValue(attr);
	ASSERT(val);
	elems[i] = m->ConvertFromValue(name, 
				       *val, 
				       ext.min[i], 
				       ext.max[i]); 
    }
}

void Vec3::FillEvent(OMEvent *ev) const {
    Manager *m = GameManager::GetInstance()->GetManager();
    GameWorld *w = GameManager::GetInstance()->GetWorld();
    BBox ext = w->GetExtent();

    uint32 offset = 0;
    if (GameManager::publishAsDHTRegions) {
	offset = 1;
    }

    for (int i=0; i<3; i++) {
	if (!w->PublishDim(i)) continue;

	const string name(g_AttrNames[i + offset]);
	int attr = m->GetAttrIndex(name);

	Value v = m->ConvertToValue(name, 
				    elems[i], 
				    ext.min[i], 
				    ext.max[i]);
	ev->AddTuple( Tuple(attr, v ) );
    }
}

ostream& operator<<(ostream& out, const Vec3& v)
{
    out << "(" << v[0] << "," << v[1] << "," << v[2] << ")";
    return out;
}


static void R_ConcatRotations (float in1[3][3], float in2[3][3], float out[3][3]);

Vec3 Vec3::Perpendicular() const
{
    Vec3 src = Normalize();
    Vec3 dst;
    int	pos;
    int i;
    float minelem = 1.0F;
    Vec3 tempvec;

    /*
    ** find the smallest magnitude axially aligned vector
    */
    for ( pos = 0, i = 0; i < 3; i++ ) {
	if ( fabs( src[i] ) < minelem ) {
	    pos = i;
	    minelem = fabs( src[i] );
	}
    }
    tempvec[0] = tempvec[1] = tempvec[2] = 0.0F;
    tempvec[pos] = 1.0F;

    /*
    ** project the point onto the plane defined by src
    */
    dst = tempvec.ProjectOnPlane( src );

    /*
    ** normalize the result
    */
    return dst.Normalize();
}

Vec3 Vec3::ProjectOnPlane(const Vec3& normal) const
{
    const Vec3& p = *this;
    float d;
    Vec3 n;
    float inv_denom;

    inv_denom = 1.0F / normal.Dot( normal );

    d = normal.Dot( p ) * inv_denom;

    n[0] = normal[0] * inv_denom;
    n[1] = normal[1] * inv_denom;
    n[2] = normal[2] * inv_denom;

    return Vec3( p[0] - d * n[0],
		 p[1] - d * n[1],
		 p[2] - d * n[2] );
}

Vec3 Vec3::RotateAroundVector(const Vec3& dir, real32 rad ) const
{
    const Vec3& point = *this;
    Vec3 dst;

    float	m[3][3];
    float	im[3][3];
    float	zrot[3][3];
    float	tmpmat[3][3];
    float	rot[3][3];
    int	i;
    Vec3 vr, vup, vf;

    vf = dir;
    vr = dir.Perpendicular();
    vup = vr.Cross( vf );

    m[0][0] = vr[0];
    m[1][0] = vr[1];
    m[2][0] = vr[2];

    m[0][1] = vup[0];
    m[1][1] = vup[1];
    m[2][1] = vup[2];

    m[0][2] = vf[0];
    m[1][2] = vf[1];
    m[2][2] = vf[2];

    memcpy( im, m, sizeof( im ) );

    im[0][1] = m[1][0];
    im[0][2] = m[2][0];
    im[1][0] = m[0][1];
    im[1][2] = m[2][1];
    im[2][0] = m[0][2];
    im[2][1] = m[1][2];

    memset( zrot, 0, sizeof( zrot ) );
    zrot[0][0] = zrot[1][1] = zrot[2][2] = 1.0F;

    zrot[0][0] = cos( rad );
    zrot[0][1] = sin( rad );
    zrot[1][0] = -sin( rad );
    zrot[1][1] = cos( rad );

    R_ConcatRotations( m, zrot, tmpmat );
    R_ConcatRotations( tmpmat, im, rot );

    for ( i = 0; i < 3; i++ ) {
	dst[i] = 
	    rot[i][0] * point[0] + 
	    rot[i][1] * point[1] + 
	    rot[i][2] * point[2];
    }
    return dst;
}

void R_ConcatRotations (float in1[3][3], float in2[3][3], float out[3][3])
{
    out[0][0] = in1[0][0] * in2[0][0] + in1[0][1] * in2[1][0] +
	in1[0][2] * in2[2][0];
    out[0][1] = in1[0][0] * in2[0][1] + in1[0][1] * in2[1][1] +
	in1[0][2] * in2[2][1];
    out[0][2] = in1[0][0] * in2[0][2] + in1[0][1] * in2[1][2] +
	in1[0][2] * in2[2][2];
    out[1][0] = in1[1][0] * in2[0][0] + in1[1][1] * in2[1][0] +
	in1[1][2] * in2[2][0];
    out[1][1] = in1[1][0] * in2[0][1] + in1[1][1] * in2[1][1] +
	in1[1][2] * in2[2][1];
    out[1][2] = in1[1][0] * in2[0][2] + in1[1][1] * in2[1][2] +
	in1[1][2] * in2[2][2];
    out[2][0] = in1[2][0] * in2[0][0] + in1[2][1] * in2[1][0] +
	in1[2][2] * in2[2][0];
    out[2][1] = in1[2][0] * in2[0][1] + in1[2][1] * in2[1][1] +
	in1[2][2] * in2[2][1];
    out[2][2] = in1[2][0] * in2[0][2] + in1[2][1] * in2[1][2] +
	in1[2][2] * in2[2][2];
}


///////////////////////////////////////////////////////////////////////////////

BBox::BBox(Packet *pkt) : min(Vec3(pkt)), max(Vec3(pkt)) {}

void BBox::Serialize(Packet *pkt) {
    min.Serialize(pkt);
    max.Serialize(pkt);
}

uint32 BBox::GetLength() {
    return min.GetLength() + max.GetLength();
}

BBox::BBox(OMInterest *elt)
{
    TypeToBBox<OMInterest>(this, elt);
}

BBox::BBox(OMEvent *elt)
{
    ASSERT(!elt->IsPointEvent());
    TypeToBBox<OMEvent>(this, elt);
}

void BBox::FillInterest(OMInterest *in) const
{
    BBoxToType<OMInterest>(in, this);
}

void BBox::FillEvent(OMEvent *ev) const
{
    BBoxToType<OMEvent>(ev, this);
}

real32 BBox::Volume() const
{
    return (max[0] - min[0])*(max[1] - min[1])*(max[2] - min[2]);
}

bool BBox::Contains(const BBox& other) const
{
    for (int i=0; i<3; i++) {
	if (min[i] > other.min[i] || max[i] < other.max[i]) {
	    return false;
	}
    }

    return true;
}

bool BBox::Contains(const Vec3& pt) const
{
    for (int i=0; i<3; i++) {
	if (min[i] > pt[i] || max[i] < pt[i]) {
	    return false;
	}
    }

    return true;
}

static bool _OverlapsHelper(const Vec3& min1, const Vec3& max1,
			    const Vec3& min2, const Vec3& max2)
{
    for (int i=0; i<3; i++) {
	if (!( min1[i] <= min2[i] && min2[i] <= max1[i] || 
	       min1[i] <= max2[i] && max2[i] <= max1[i] )) {
	    return false;
	}
    }

    return true;
}

bool BBox::Overlaps(const BBox& other) const
{
    return 
	_OverlapsHelper(other.min, other.max, min, max) ||
	_OverlapsHelper(min, max, other.min, other.max);
}

BBox BBox::GetOverlap(const BBox& other) const
{
    BBox ret;

    if (!Overlaps(other)) {
	return ret; // empty bbox
    }

    for (int i=0; i<3; i++) {
	// The overlapping min point is larger of the two minimums
	if (min[i] > other.min[i]) {
	    ret.min[i] = min[i];
	} else {
	    ret.min[i] = other.min[i];
	}

	// The overlapping max point is the smaller of the two maximums
	if (max[i] < other.max[i]) {
	    ret.max[i] = max[i];
	} else {
	    ret.max[i] = other.max[i];
	}
    }

    return ret;
}

BBox BBox::GetMerged(const BBox& other) const
{
    BBox ret;

    for (int i=0; i<3; i++) {
	ret.min[i] = MIN(min[i], other.min[i]);
	ret.max[i] = MAX(max[i], other.max[i]);
    }

    return ret;
}

ostream& operator<<(ostream& out, const BBox& v)
{
    out << "(BBox min=" << v.min << " max=" << v.max << ")";
    return out;
}

///////////////////////////////////////////////////////////////////////////////

template <class T>
void BBoxToType(T *elt, const BBox *bbox)
{
    Manager *m = GameManager::GetInstance()->GetManager();
    GameWorld *w = GameManager::GetInstance()->GetWorld();
    BBox ext = w->GetExtent();

    uint32 offset = 0;

    // XXX HACK This is such a large hack!
    // XXX THIS IS INCORRECT FOR DHT RANGE PUBS!!!! 
    // (though I guess such things do not exist)
    if (GameManager::publishAsDHTRegions) {

	const string name(g_AttrNames[0]);
	int attr = m->GetAttrIndex(name);

	// add the key constraint
	Constraint k(attr,
		     m->ConvertToValue(name, 
				       bbox->min[0], 
				       ext.min[0], 
				       ext.max[0]), 
		     m->ConvertToValue(name, 
				       bbox->max[0], 
				       ext.min[0], 
				       ext.max[0]));
	elt->AddConstraint( k );

	// the other constraints come after the key constraint
	// for interests, just set them to the entire range because
	// we only match on the key anyway, so we match anything in
	// the remainder
	offset = 1;

	//return;
    }

    for (int i=0; i<3; i++) {
	if (!w->PublishDim(i)) continue;

	const string name(g_AttrNames[i+offset]);
	int attr = m->GetAttrIndex(name);

	Constraint c(attr, 
		     m->ConvertToValue(name, 
				       bbox->min[i+offset], 
				       ext.min[i], 
				       ext.max[i]), 
		     m->ConvertToValue(name, 
				       bbox->max[i+offset], 
				       ext.min[i], 
				       ext.max[i]));
	ASSERTDO(c.GetMin() < c.GetMax(), WARN << "bbox=" << *bbox << " cons=" << c << endl);

	elt->AddConstraint( c );
    }
}


template <class T>
void TypeToBBox(BBox *bbox, T *elt)
{
    Manager *m = GameManager::GetInstance()->GetManager();
    GameWorld *w = GameManager::GetInstance()->GetWorld();
    BBox ext = w->GetExtent();

    uint32 offset = 0;
    if (GameManager::publishAsDHTRegions) {
	offset = 1;
    }

    for (int i=0; i<3; i++) {
	if (!w->PublishDim(i)) continue;

	const string name(g_AttrNames[i+offset]);
	int attr = m->GetAttrIndex(name);

	const Constraint *c = elt->GetConstraintByAttr(attr);
	ASSERT(c);
	bbox->min[i] = m->ConvertFromValue(name, 
					   c->GetMin(), 
					   ext.min[i], 
					   ext.max[i]);
	bbox->max[i] = m->ConvertFromValue(name, 
					   c->GetMax(), 
					   ext.min[i], 
					   ext.max[i]);
    }
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
