////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameWorld.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_WORLD_H__
#define __GAME_WORLD_H__

#include <gameapi/common.h>
#include <list>

class GameObject;
class DynamicGameObject;
class WayPointManager;

/**
 * Interface for the game geometry, tiles, or whatever. When an object enters
 * or leaves the GameStore, it is linked/unlinked from the game world.
 */
class GameWorld {
private:
    friend class GameVisualizer;
    friend class GameStore;
    friend class DynamicObjectStore;
public:

    virtual ~GameWorld () {}

    /**
     * Called when an object is added to the GameStore.
     */
    virtual void Link(GameObject *obj) = 0;

    /**
     * Called when an object is removed from the GameStore.
     */
    virtual void Unlink(GameObject *obj) = 0;

    /**
     * Called when object moved.
     */
    virtual void Update(GameObject *obj) = 0;

    /**
     * Get the area of interest for an object in this world. If
     * coordination with the object is required, you should override
     * DynamicGameObject::GetAreaOfInterest() since that is what will
     * be called by the GameManager. By default that method only delegates
     * to this. Return the TTL of the interests.
     */
    virtual uint32 GetAreaOfInterest(list<BBox> *toFill, 
	    DynamicGameObject *obj) = 0;

    /**
     * Box around the entire world.
     */
    virtual const BBox& GetExtent() = 0;

    /**
     * Publish this dimension of the world (x,y,z) = (0,1,2)
     * These are the dims that are matched against (not necesarily routed)
     * 
     * @param dim the dimension (x,y,z) = (0,1,2)
     * @return true if Colyseus should publish this dimension
     */
    virtual const bool PublishDim(uint32 dim) = 0;

    /**
     * Clip the bounding box to world boundaries.
     */
    virtual BBox Clip(const BBox& clip) = 0;

    /**
     * Clip each dimension of the point to be within world boundaries.
     */
    virtual Vec3 Clip(const Vec3& clip) = 0;
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
