////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <om/Manager.h>
#include <wan-env/RealNet.h>
#include <gameapi/GameObject.h>
#include <gameapi/GameManager.h>
#include <gameapi/GameStore.h>
#include <gameapi/GameWorld.h>
#include <gameapi/GameVisualizer.h>
#include <mercury/options.h>   // FIXME: i should not be storing these options in mercury's g_Preferneces

///////////////////////////////////////////////////////////////////////////////
// Visualization

static byte sg_Buffer[UDP_MAXLEN];
static uint32 sg_BufLen = 0;

static void initialize_socket(int &udp_sock)
{
    if (OS::CreateSocket(&udp_sock, PROTO_UDP) < 0) 
	Debug::die("Could not create udp socket for the visualizer!");

    struct sockaddr_in	server_address;
    memset((char *)&server_address, 0, sizeof(server_address));

    server_address.sin_family = AF_INET;
    server_address.sin_addr.s_addr = g_LocalSID.GetIP();
    server_address.sin_port = htons(g_Preferences.port + 10000);  // From the Terminal code

    // Bind so that the visualizer can figure out who we are!
    if (bind(udp_sock, (struct sockaddr *) &server_address, sizeof(server_address)) < 0) {
	perror ("bind");
	Debug::die ("could not bind the udp socket for sending to the visualizer!");
    }
}

// debug debug
int XXX_framenum = 0;

static void send_data(int udp_sock, struct sockaddr_in &vis_addr)
{
    if (vis_addr.sin_addr.s_addr == 0)
	return;

#if 0
    FILE *fp = fopen ("/tmp/write-log", "a");
    fprintf (fp, "framenum=%d len=%d:::\n", XXX_framenum, sg_BufLen);
    for (unsigned int i = 0; i < sg_BufLen; i++) {
	fprintf (fp, "%02x ", sg_Buffer[i]);
	if (i != 0 && i % 20 == 0)
	    fprintf (fp, "\n");
    }
    fprintf (fp, "\n");
    fclose (fp);
#endif

    if (sendto(udp_sock, sg_Buffer, sg_BufLen, 0, 
	       (struct sockaddr *) &vis_addr, sizeof(vis_addr)) < 0) 
	{
	    DBG_DO {
		perror("sendto");
		Debug::warn("sendto call failed");
	    }
	}
}

static void write_float(float f)
{
    uint32 l = htonl( *(uint32 *)&f );
    memcpy(sg_Buffer + sg_BufLen, &l, 4);
    sg_BufLen += 4;
}

static inline void write_byte(byte b)
{
    sg_Buffer[sg_BufLen++] = b;
}

static inline void write_int(int i)
{
    uint32 l = htonl(i);
    memcpy(sg_Buffer + sg_BufLen, &l, 4);
    sg_BufLen += 4;
}

static void write_vec(const Vec3& pt) {
    for (int i = 0; i < 3; i++) 
	write_float(pt[i]);
}

static void write_bbox(const BBox& box) {
    write_vec(box.min);
    write_vec(box.max);
}

static void dump_dg_edict(DynamicGameObject *dobj)
{
    byte mask = 0;
    if (dobj->IsReplica()) 
	mask |= STATUS_REPLICA;

    // delegate to the subclass
    mask |= dobj->GetVisualizerType ();

    write_byte(mask);

    Vec3 origin = dobj->GetOrigin ();
    // cerr << "classname=" << dobj->GetClassName() << "origin=" << origin << endl;
    // write_vec(origin);
    write_float(origin[0]);
    write_float(origin[1]);
    write_float(origin[2]);

    BBox box, pbox;
    if (!dobj->IsReplica ()) {
	list<BBox> aoi;
	dobj->GetAreaOfInterest( &aoi, GameManager::GetInstance()->GetWorld() );

	if ( aoi.size() > 0 ) {
	    box = aoi.front();
	}
#ifdef SAVE_PREDICTED_AOI
	// if debug enabled, the predicted aoi is also saved
	list<BBox> *paoi = dobj->GetPredictedAreaOfInterest();
	if (paoi->size() > 0) {
	    pbox = paoi->front();
	}
#else
	// else, don't know the predicted bbox
	pbox = box;
#endif
    }

    write_bbox(box);
    write_bbox(pbox);
}

///////////////////////////////////////////////////////////////////////////////

bool GameVisualizer::SetUpdateRate (int rate) 
{
    if (rate < 0 || rate > 100)
	return false;
    m_UpdateRate = rate;
    return true;
}

bool GameVisualizer::SetTargetIP(const char *ip)
{
    if (!strcmp(ip, "")) {
	bzero(&m_VisAddr, sizeof(m_VisAddr));
	return true;
    }

    IPEndPoint visIP((char *)ip, (int)VISUALIZER_PORT);
    if (visIP == SID_NONE) {
	return false;
    }

    memset((char *) &m_VisAddr, 0, sizeof(m_VisAddr));
    m_VisAddr.sin_family = AF_INET;
    m_VisAddr.sin_addr.s_addr = visIP.GetIP();
    m_VisAddr.sin_port = htons(visIP.GetPort());

    return true;
}

void GameVisualizer::SendEdicts (callback<bool>::ref send_chunk_cb)
{
    DynamicGameObject *obj;
    GameStore *s = GameManager::GetInstance()->GetStore();

    sg_BufLen = 0;

    write_byte(MSGTYPE_ENTITIES);
    write_int(GameManager::GetInstance()->GetFrameNumber());

    XXX_framenum = GameManager::GetInstance()->GetFrameNumber();

    s->BeginDynamic();

    while ((obj = s->NextDynamic()) != NULL) {
	if (sg_BufLen + EDICT_DUMP_SIZE > UDP_MAXLEN) {
	    ASSERT((sg_BufLen - 1 - 4) % EDICT_DUMP_SIZE == 0);

	    if (!send_chunk_cb ())
		return;

	    // send_data(udp_sock, m_VisAddr);			
	    sg_BufLen = 0;

	    write_byte(MSGTYPE_ENTITIES);
	    write_int(GameManager::GetInstance()->GetFrameNumber());
	}

	dump_dg_edict(obj);
    }

    if (sg_BufLen > 0) {
	ASSERT((sg_BufLen - 1 - 4) % EDICT_DUMP_SIZE == 0);

	if (!send_chunk_cb ())
	    return;

	// send_data(udp_sock, m_VisAddr);
    }
}

void GameVisualizer::SendInterests (callback<bool>::ref send_chunk_cb)
{
    DynamicGameObject *obj, *other;
    GameStore *s = GameManager::GetInstance()->GetStore();

    vector<VizInterestLink> links;

    s->BeginDynamic();
    while ((obj = s->NextDynamic()) != NULL) {
	GObjectInfo *info = static_cast<GObjectInfo *>(obj->GetInfo());
	InterestLinkMap *ilinks = info->GetInterests();

	for (InterestLinkMapIter it = ilinks->begin(); it != ilinks->end();
		it++) {
	    InterestLink *link = it->second;
	    GObjectInfo  *targ = link->GetTarget();

	    other = dynamic_cast<DynamicGameObject *>(s->Find(targ->GetGUID()));
	    if (other) {
		const Vec3& from = obj->GetOrigin();
		const Vec3& to = other->GetOrigin(); 

		VizInterestLink l;
		l.from[0] = from[0];
		l.from[1] = from[1];
		l.to[0]   = to[0];
		l.to[1]   = to[1];
		l.stable  = (byte)link->IsStable();

		links.push_back(l);
	    }
	}
    }

    sg_BufLen = 0;

    write_byte(MSGTYPE_INTGRAPH);
    write_int(GameManager::GetInstance()->GetFrameNumber());

    for (uint32 i=0; i<links.size(); i++) {
	if (sg_BufLen + EDICT_DUMP_SIZE > UDP_MAXLEN) {
	    ASSERT((sg_BufLen - 1 - 4) % LINK_DUMP_SIZE == 0);

	    if (!send_chunk_cb ())
		return;

	    // send_data(udp_sock, m_VisAddr);

	    sg_BufLen = 0;

	    write_byte(MSGTYPE_INTGRAPH);
	    write_int(GameManager::GetInstance()->GetFrameNumber());
	}

	const VizInterestLink& l = links[i];

	write_float(l.from[0]);
	write_float(l.from[1]);

	write_float(l.to[0]);
	write_float(l.to[1]);

	write_byte(l.stable);
    }

    if (sg_BufLen > 0) {
	ASSERT((sg_BufLen - 1 - 4) % LINK_DUMP_SIZE == 0);

	if (!send_chunk_cb ())
	    return;

	// send_data(udp_sock, m_VisAddr);
    }
}

bool udp_callback (int udp_sock, struct sockaddr_in vis_sock) {
    send_data (udp_sock, vis_sock);
    return true;
}

void GameVisualizer::SendUpdates()
{
    static bool inited = false;
    static int  udp_sock = -1;
    TimeVal now = GameManager::GetInstance ()->ThisFrameTime ();
    sint64 elapsed_msecs = now - m_LastSent;

    if (m_VisAddr.sin_addr.s_addr == 0) {
	return;
    }
    if (elapsed_msecs < (1000 / m_UpdateRate))
	return;

    if (!inited) {
	initialize_socket(udp_sock);
	inited = true;
    }

    m_LastSent = now;

    // send over all objects in store
    SendEdicts (wrap (udp_callback, udp_sock, m_VisAddr));

    // send over interest graph (from me => others)
    SendInterests (wrap (udp_callback, udp_sock, m_VisAddr));
}

// Just because `wrap' does not like passing references 
// as currying arguments
//
class Wrapper {
public:
    ostream &out;
    Wrapper (ostream &o) : out (o) {}
};

static bool hacky_tcp_callback (Wrapper *o) {
    ostream& out = o->out;

    out << "length=" << sg_BufLen << endl;
    out << "=========================" << endl;
    for (uint32 i = 0; i < sg_BufLen; i++) { 
	out << sg_Buffer [i];
    }
    out << endl << "=========================" << endl;
    return true;
}

void GameVisualizer::SendTCPUpdate (ostream& out) 
{    
    Wrapper *w = new Wrapper (out);
    SendEdicts (wrap (hacky_tcp_callback, w));
    SendInterests (wrap (hacky_tcp_callback, w));

    delete w;
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
