////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameStore.cpp

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include "GameObject.h"
#include "GameWorld.h"
#include "GameStore.h"
#include <util/Benchmark.h>

void DynamicObjectStore::Begin() 
{
    // XXX -- one day, make this into a mutable-safe iterator instead
    m_Iter.clear();
    for (DynamicObjMapIter it = m_Store.begin(); it != m_Store.end(); it++)
	m_Iter.push_back(it->second);
    m_Curr = m_Iter.begin();
}

GObject *DynamicObjectStore::Next() 
{
    if (m_Curr == m_Iter.end()) {
	return NULL;
    } else {
	GObject *next = *m_Curr;
	m_Curr++;
	return next;
    }
}

uint32 DynamicObjectStore::Size()
{
    return m_Store.size();
}

GObject *DynamicObjectStore::Find(guid_t guid) 
{
    DynamicObjMapIter p = m_Store.find(guid);
    if (p == m_Store.end()) {
	return NULL;
    } else {
	return p->second;
    }
}

void DynamicObjectStore::_ManagerAdd(GObject *obj) {
    ASSERT(dynamic_cast<GameObject *>(obj)->IsDynamic());
    ASSERT(dynamic_cast<DynamicGameObject *>(obj));

    DynamicObjMapIter p = m_Store.find(obj->GetGUID());
    if (p != m_Store.end()) {
	// throw away the old copy
	if (m_LinkOnManagerAdd) 
	    m_World->Unlink(static_cast<GameObject *>(p->second));
	delete p->second;
	m_Store.erase(p);
    }

    m_Store[obj->GetGUID()] = static_cast<DynamicGameObject *>(obj);
    if (m_LinkOnManagerAdd)
	m_World->Link(static_cast<DynamicGameObject *>(obj));
}

void DynamicObjectStore::_ManagerRemove(guid_t guid) 
{
    DynamicObjMapIter p = m_Store.find(guid);
    if (p != m_Store.end()) {
	if (m_LinkOnManagerAdd) 
	    m_World->Unlink(static_cast<GameObject *>(p->second));
	m_Store.erase(p);
    }
    // DO NOT DELETE -- manager may resuse it elsewhere
}

void DynamicObjectStore::_ApplicationAdd(GObject *obj)
{
    ASSERT(dynamic_cast<DynamicGameObject *>(obj));

    START(temp::addfind);
    DynamicObjMapIter p = m_Store.find(obj->GetGUID());
    STOP(temp::addfind);

    ASSERT(p == m_Store.end());

    m_Store[obj->GetGUID()] = static_cast<DynamicGameObject *>(obj);
    if (m_LinkOnManagerAdd) {
	START(temp::addlink);
	m_World->Link(static_cast<DynamicGameObject *>(obj));
	STOP(temp::addlink);
    }
}

void DynamicObjectStore::_ApplicationRemove(guid_t guid)
{
    DynamicObjMapIter p = m_Store.find(guid);
    if (p != m_Store.end()) {
	if (m_LinkOnManagerAdd) 
	    m_World->Unlink(static_cast<GameObject *>(p->second));
	m_Store.erase(p);
    }
}

///////////////////////////////////////////////////////////////////////////////

GameStore *GameStore::m_Instance = NULL;

GameStore::GameStore(GameWorld *world) : m_World(world) {
    m_DynamicStore   = new DynamicObjectStore(m_World, true);
    m_DynamicPending = new DynamicObjectStore(m_World, false);

}

GameStore *GameStore::GetInstance(GameWorld *world)
{
    if (m_Instance == NULL) {
	m_Instance = new GameStore(world);
    }
    return m_Instance;
}

GameStore *GameStore::GetInstance()
{
    ASSERT(m_Instance != NULL);
    return m_Instance;
}

GameStore::~GameStore() {
    // XXX - this isn't really a very clean cleanup -- we still need to
    // delete the actual objects!
    delete m_DynamicStore;
    delete m_DynamicPending;
    m_StaticStore.clear();
}

GameObject *GameStore::Find(GUID guid)
{
    // first look in the dynamic store
    DynamicGameObject *obj = 
	static_cast<DynamicGameObject *>(m_DynamicStore->Find(guid));
    if (obj)
	return obj;
    // then look in the static store
    StaticObjMapIter p = m_StaticStore.find(guid);
    if (p != m_StaticStore.end()) {
	return p->second;
    }
    // didn't find it
    return NULL;
}

void GameStore::Add(GameObject *obj)
{
    if (obj->IsDynamic()) {
	START(temp::dynadd);
	ASSERT(dynamic_cast<DynamicGameObject *>(obj));

	m_DynamicStore->ApplicationAdd(dynamic_cast<DynamicGameObject *>(obj));
	STOP(temp::dynadd);

	// DynamicStore already links with world
    } else {
	START(temp::statadd);
	ASSERT(dynamic_cast<StaticGameObject *>(obj));
	ASSERT(m_StaticStore.find(obj->GetGUID()) == m_StaticStore.end());
	m_StaticStore[obj->GetGUID()] = dynamic_cast<StaticGameObject *>(obj);
	m_World->Link(obj);
	STOP(temp::statadd);
    }
}

GameObject *GameStore::Remove(GUID guid)
{
    DynamicGameObject *obj = 
	static_cast<DynamicGameObject *>(m_DynamicStore->Find(guid));

    if (obj) {
	ASSERT(!obj->IsReplica ());
	m_DynamicStore->ApplicationRemove(guid);
	// DynamicStore already unlinks with world
	return obj;
    } else {
	ASSERTDO(false, WARN << "I should never come here! guid=" << guid << endl);

	obj = 
	    static_cast<DynamicGameObject *>(m_DynamicPending->Find(guid));
	if (obj) {
	    m_DynamicPending->ApplicationRemove(guid);
	    return obj;
	}
    }
    StaticObjMapIter p = m_StaticStore.find(guid);
    if (p != m_StaticStore.end()) {
	StaticGameObject *obj = p->second;
	m_StaticStore.erase(p);
	m_World->Unlink(obj);
	return obj;
    }
    return NULL;
}

void GameStore::Begin()
{
    m_DynamicStore->Begin();

    // XXX: one day make this into a safely mutable iterator
    m_StaticIter.clear();
    for (StaticObjMapIter it = m_StaticStore.begin(); 
	 it != m_StaticStore.end(); it++)
	m_StaticIter.push_back(it->second);
    m_StaticCurr = m_StaticIter.begin();
}

GameObject *GameStore::Next()
{
    GameObject *obj = static_cast<DynamicGameObject *>(m_DynamicStore->Next());
    if (obj) {
	return obj;
    }
    if (m_StaticCurr == m_StaticIter.end()) {
	return NULL;
    } else {
	obj = *m_StaticCurr;
	m_StaticCurr++;
	return obj;
    }
}

uint32 GameStore::Size()
{
    return m_DynamicStore->Size() + m_StaticStore.size();
}

void GameStore::BeginDynamic()
{
    m_DynamicStore->Begin();
}

DynamicGameObject *GameStore::NextDynamic()
{
    return static_cast<DynamicGameObject *>(m_DynamicStore->Next());
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
