////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameManager.h

  Top-level interface to the game application.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#ifndef __GAME_MODULE_H__
#define __GAME_MODULE_H__

#include <gameapi/common.h>
#include <gameapi/GameObject.h>
#include <om/Terminal.h>   // FIXME: move this to util/

class GameManager;

/**
 * This is the primary interface to the game code. It contains a series of
 * game-specific callbacks that are called when running the game.
 */
class GameModule {
public:
    virtual ~GameModule () {}

    /**
     * @return the interval between each frame in msec. (e.g., 1/fps x 1000)
     */
    virtual const uint32 FrameInterval() = 0;

    /**
     * Called just before the game begins.
     *
     * @param manager the global instance of the GameManager
     * (this is passed to the function for convenience only)
     * @return GAME_OK unless there was an unrecoverable error
     */
    virtual gRetCode Init(GameManager *manager) = 0;

    /**
     * Called (approximately) once every FrameInterval()
     *
     * @param manager the global instance of the GameManager
     * (this is passed to the function for convenience only)
     * @return GAME_OK unless there was an unrecoverable error
     */
    virtual gRetCode RunFrame(GameManager *manager) = 0;

    /**
     * Called just after the game exits.
     *
     * @param manager the global instance of the GameManager
     * (this is passed to the function for convenience only)
     * @return GAME_OK unless there was an unrecoverable error
     */
    virtual gRetCode Shutdown(GameManager *manager) = 0;

    /**
     * Given an object type code, construct a "baseline" copy of that object
     * type. After construction, the newly construced object will usually
     * have DynamicGameObject::UnpackUpdate() called on it to deserialize
     * the actual object. Do not link the object to the game world.
     *
     * @param type the typecode for the object we wish to construct.
     * @param info the Colyseus information about the object (GUID, SID, etc.)
     * @return a newly constructed game object.
     */
    virtual DynamicGameObject *ConstructObject(gTypeCode type,
	    GObjectInfoIface *info) = 0;

    /**
     * CURRENTLY UNUSED. WILL NOT BE CALLED.
     */
    virtual GameObject *ConstructObject(Packet *pkt) = 0;

    /**
     * Delete a game object created with ConstructObject(...). Do not
     * unlink the object from the game world. Just free resources associated
     * with it.
     *
     * @param obj the object to be deleted.
     */
    virtual void DeleteObject(DynamicGameObject *obj) = 0;

    /**
     * Install handlers for the terminal interface, so game-specific
     * commands can be answered. [optional method]
     *
     * @param tofill a list of handlers, already filled in by manager and gamemanager
     */
    virtual void InstallTerminalHandlers (InputHandlers *tofill) 
    {
    }
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
