////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameManager.cpp

  Top-level interface to the game application.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef BENCHMARK_REQUIRED
#define  BENCHMARK_REQUIRED
#endif

#include "GameManager.h"
#include "GameObject.h"
#include "GameStore.h"
#include "GameModule.h"
#include "GameWorld.h"
#include "GameDatabase.h"
#include "GameClientHandler.h"
#include "GameLogs.h"
#include <gameapi/NullClientHandler.h>
#include <gameapi/NullInterestFactory.h>
#include "DummyManager.h"
#include <om/Manager.h>
#include <om/OMInterest.h>
#include <util/Options.h>

static uint32 g_FrameLimit;
static uint32 g_TimeLimit;
static char g_VisIP[255];
static char g_BenchmarkPatternStr[255];
static char *g_BenchmarkPatterns[255];
static uint32 g_BenchmarkPrintInterval;

bool GameManager::publishAsDHTRegions = false;

static OptionType g_GameOptions[] =
{
    { '#', "GameAPI Parameters", OPT_SEP, "", NULL, "", NULL },
    { '/', "framelimit", OPT_INT, 
	"exit after this many frame (0 means unlimited)", 
	&g_FrameLimit, "0", NULL },
    { '/', "timelimit", OPT_INT, 
	"exit after this many msecs (0 means unlimited)", 
	&g_TimeLimit, "0", NULL },
    { '/', "dht", OPT_BOOL|OPT_NOARG, 
	"publish/subscribe using DHT region keys instead of range queries (requires --regions-per-side)", 
	&GameManager::publishAsDHTRegions, "0", (void *)"1" },
    { '/', "visualizer-ip", OPT_STR, 
	"IP of the visualizer server (port:60000)", 
	g_VisIP, "", NULL },
    { 'B', "benchmark-patterns", OPT_STR,
	"show benchmark data for items matching these patterns", 
	g_BenchmarkPatternStr, "", NULL},
    { '/', "benchmark-print-interval", OPT_INT,
	"how often to print and restart benchmarks (ms)", 
	&g_BenchmarkPrintInterval, "10000", NULL},
    { 0, 0, 0, 0, 0, 0, 0 }
};

///////////////////////////////////////////////////////////////////////////////

GameManager *GameManager::m_Instance = NULL;

GameManager::GameManager(int *pArgc, char *argv[], 
	OptionType appOptions[],
	bool printOptions) :
    m_Module(NULL), m_Store(NULL), m_World(NULL), m_Database(NULL), 
    m_ClientHandler(NULL), m_Manager(NULL), m_GlobalFilter(NULL)
{
    OptionType *opts;
    if (appOptions == NULL) {
	opts = g_GameOptions;
    } else {
	int mlen = 0, alen = 0;
	for ( ; g_GameOptions[mlen].longword ; mlen++)
	    ;
	for ( ; appOptions[alen].longword ; alen++)
	    ;
	opts = new OptionType[mlen + alen + 1];
	memcpy(opts, g_GameOptions, mlen*sizeof(OptionType));
	memcpy(opts+mlen, appOptions, alen*sizeof(OptionType));
	bzero(opts+mlen+alen, sizeof(OptionType));
    }

    // init colyseus options
    InitializeColyseus(pArgc, argv, opts, printOptions);

    if (appOptions != NULL) delete[] opts;

    // init benchmark strings
    int index = 0;
    char *str = strtok(g_BenchmarkPatternStr, ",;");
    while (str != NULL) {
	g_BenchmarkPatterns[index++] = str;
	if (index == 254) break;
	str = strtok(NULL, ",;:");
    }
    g_BenchmarkPatterns[index] = NULL;

    m_Visualizer.SetTargetIP(g_VisIP);
}

GameManager *GameManager::GetInstance(int *pArgc, char *argv[], 
	OptionType appOptions[],
	bool printOptions)
{
    if (m_Instance == NULL) {
	m_Instance = new GameManager(pArgc, argv, appOptions, printOptions);
    }
    return m_Instance;
}

GUID GameManager::CreateGUID() {
    ASSERT(m_Manager);
    return m_Manager->CreateGUID();
}

GameManager::~GameManager()
{
    delete m_Store;
    delete m_Database;
    Manager::DestroyInstance();
}

void GameManager::Init(GameModule *module,
	GameWorld  *world,
	GameClientHandler *chandler,
	GameInterestFactory *ifactory,
	list<GameDataSource *> *sources,
	bool dummy)
{
    ASSERT(module);
    ASSERT(world);
    m_Module = module;
    m_World = world;

    if (chandler) {
	m_ClientHandler = chandler;
    } else {
	m_ClientHandler = new NullClientHandler();
	// TODO: tiny mem leak!
    }

    if (ifactory) {
	m_InterestFactory = ifactory;
    } else {
	m_InterestFactory = new NullInterestFactory();
	// TODO: tiny mem leak!
    }

    m_Database = GameDatabase::GetInstance();
    if (sources) {
	for (list<GameDataSource *>::iterator it = sources->begin();
		it != sources->end(); it++) {
	    m_Database->AddDataSource(*it);
	}
    }

    m_Store = GameStore::GetInstance(m_World);

    DB (-1) << "initializing manager " << endl;
    if (dummy)
	m_Manager = new DummyManager();
    else
	m_Manager = Manager::GetInstance(this);

    TDB (-1) << "initializing game logging " << endl;
    // initialize logging
    if (g_MeasurementParams.enabled)
	InitGameLogs(g_LocalSID);

    TDB (-1) << "initializing game module" << endl;
    gRetCode ret = m_Module->Init(this);
    if (ret != GAME_OK) {
	WARN << "GameModule::Init failed to initialize with code: " 
	    << ret << endl;
	exit(ret);
    }
    TDB(-1) << "finished initialization of game module" << endl;
}

#define CHECK_RETURN(blurb) { \
    ret = blurb; \
    if (ret == GAME_ERROR) { \
	WARN << "Got game error on call to: \"" << #blurb << "\"" << endl; \
	break; \
    } else if (ret == GAME_EXIT) { break; } \
}

void GameManager::Run()
{
    ASSERT(m_Module);
    ASSERT(m_ClientHandler);
    ASSERT(m_Manager);

    TDB (-1) << "Starting the game!" << endl;

    gRetCode ret = GAME_OK;
    uint32 frameInterval = m_Module->FrameInterval();
    uint32 pollInterval = m_ClientHandler->PollInterval();
    uint32 frameSlack = 10; // XXX TODO: periodically re-estimate

    m_FrameNumber = 0;
    m_LastFrame = CurrentTime() + -frameInterval;
    m_ThisFrame = CurrentTime();
    m_NextFrame = m_ThisFrame + frameInterval;
    uint64 stopFrame = TIME_STOP_USEC( NextFrameTime() - CurrentTime() );
    uint64 nextClientPoll;

    TimeVal stopTime = m_ThisFrame + g_TimeLimit;

    sint32 remaining;

    // logging entry
    static QuakeStatsLogEntry stats;
    static double time;

    while (1) {
	START(GM_FRAME_TOTAL);
	stats = QuakeStatsLogEntry();

	// measurement
	stats.frameSlack = 
	    MAX(0, NextFrameTime() - CurrentTime());

	// run this at least once per frame
	START(GM_FRAME_IDLE);
	m_Manager->Maintenance( TIME_LEFT_MSEC(stopFrame) );
	nextClientPoll = TIME_STOP_USEC( 0 );
	do {
	    // Process packets
	    START(GM_FRAME_RECV);
	    m_Manager->Recv( TIME_LEFT_MSEC(stopFrame), false );
	    STOP_ASSIGN(time, GM_FRAME_RECV);
	    stats.dgIdleTime += time;
	    if ( TIMED_OUT_USEC(nextClientPoll) ) {
		ret = m_ClientHandler->RunInterFrame(this);
		if (ret != GAME_OK) break;
		nextClientPoll = TIME_STOP_USEC( pollInterval );
	    }
	} while ( !TIMED_OUT_USEC(stopFrame) );
	CHECK_RETURN( ret );

	/*
	   remaining = TIME_LEFT_MSEC(stopFrame);
	   if (remaining > 0)
	   usleep(remaining*1000);
	 */

	STOP(GM_FRAME_IDLE);

	START(GM_FRAME_EXEC);

	// don't let the next frame get off by more than the frameInterval
	if (CurrentTime() > m_NextFrame + frameInterval) {
	    m_NextFrame = CurrentTime();
	}

	m_FrameNumber++;
	m_LastFrame = m_ThisFrame;
	m_ThisFrame = m_NextFrame;
	m_NextFrame = m_NextFrame + frameInterval;
	stopFrame = TIME_STOP_USEC( NextFrameTime() - CurrentTime() );

	if (g_FrameLimit && m_FrameNumber > g_FrameLimit) {
	    INFO << " * hit timelimit ! going home " << endl;
	    break;
	}
	if (g_TimeLimit && m_ThisFrame > stopTime) {
	    INFO << " * hit timelimit ! going home " << endl;
	    break;
	}

	START(GM_FRAME_MODULE);
	CHECK_RETURN( m_ClientHandler->RunPreFrame(this) );

	GUIDSet guids, later_guids;
	ObjectStore *store = GetPendingStore ();
	ObjectStore *objStore = GetObjectStore ();

	store->Begin ();
	GObjectInfoMap *oinfo = m_Manager->GetObjectInfo ();

	GObject *o;
	while ((o = store->Next())) {
	    ASSERT (oinfo->find (o->GetGUID ()) != oinfo->end ());
	    // guids.insert (o->GetGUID ());
	}

	CHECK_RETURN( m_Module->RunFrame(this) );
	
	oinfo = m_Manager->GetObjectInfo ();
	store = GetPendingStore ();
	store->Begin ();

	while ((o = store->Next())) {
	    // ASSERT(guids.find (o->GetGUID ()) != guids.end ());
	    ASSERT(store->Find(o->GetGUID ()) != NULL); // i dont trust Find!
	    ASSERT(objStore->Find (o->GetGUID ()) == NULL);
	    ASSERTDO (oinfo->find (o->GetGUID ()) != oinfo->end (), 
		    WARN << "guid=" << o->GetGUID() << " isreplica=" << o->IsReplica() << endl);
	    // later_guids.insert (o->GetGUID ());
	}

//	ASSERT(guids.size () == later_guids.size());
//	ASSERT(guids == later_guids);


	CHECK_RETURN( m_ClientHandler->RunPostFrame(this) );
	STOP(GM_FRAME_MODULE);

	LogObjectDeltaStats();

	START(GM_FRAME_SEND);
	m_Manager->Send( TIME_LEFT_MSEC(stopFrame) );
	STOP_ASSIGN(time, GM_FRAME_SEND);
	stats.dgTime += time;

	STOP_ASSIGN(time, GM_FRAME_EXEC);
	stats.execTime = time;

#if 1
	m_Manager->Lock (); // XXX hack
	m_Visualizer.SendUpdates();
	m_Manager->Unlock ();
#endif

	if (g_MeasurementParams.enabled) {
	    LogStoreStats();
	    LogInterestStats();
	}

	STOP_ASSIGN(time, GM_FRAME_TOTAL);
	stats.frameTime = time;

	LOG(QuakeStatsLog, stats);

#if 1
	PERIODIC(g_BenchmarkPrintInterval, m_ThisFrame, {
		cerr << "free memory: " << Debug::FreeMem() << " MB " << endl;
		Debug::PrintUDPStats (cerr);

		char **pat = g_BenchmarkPatterns;
		if (*pat == NULL) {
		Benchmark::print();
		} else {
		while (*pat != NULL) {
		Benchmark::printpat(*pat);
		pat++;
		}
		}
		Benchmark::restart();
		});
#endif
    }

    DBG << "Shutting down the game!" << endl;

    // tear-down the game; game is responsible for making sure the
    // data sources are flushed if need be
    m_Module->Shutdown(this);

    /// do this as part of another call perhaps..
    // tear-down the manager and Mercury	
    // XXX bug here...
    //delete m_Manager;
    //	m_Manager = NULL;

    // flush logs
    FLUSH_ALL();

    return;
}

void GameManager::Idle(uint32 timeoutMillis)
{
    unsigned long long stoptime = CurrentTimeUsec();
    stoptime += timeoutMillis * USEC_IN_MSEC;

    while (CurrentTimeUsec() < stoptime) {
	m_Manager->Merc(200);
    }
}

///////////////////////////////////////////////////////////////////////////////

uint32 GameManager::GetSendInterval()
{
    ASSERT(m_Module);
    return m_Module->FrameInterval();
}

ObjectStore *GameManager::GetObjectStore()
{
    ASSERT(m_Store);
    return m_Store->m_DynamicStore;
}

ObjectStore *GameManager::GetPendingStore()
{
    ASSERT(m_Store);
    return m_Store->m_DynamicPending;
}

bool GameManager::ProcessInterestFilters(FilterChain *filters,
	AreaOfInterest *toFill,
	AreaOfInterest *orig)
{
    // pass the aoi through the object's individual filter pipeline
    AreaOfInterest *in  = orig;
    AreaOfInterest *out = NULL;
    InterestFilter *prev = NULL, *filter = NULL;

    for (FilterChainIter it = filters->begin();
	    it != filters->end(); it++) {
	InterestFilter *filter = *it;
	out = filter->Filter(in);
	if (prev) {
	    // allocated by previous filter
	    prev->Delete(in);
	}

	prev = filter;
	in = out;
	out = NULL;
    }

    bool nonEmpty = in->size() != 0;

    // save it in the global list
    for (AreaOfInterestIter it = in->begin(); it != in->end(); it++) {
	toFill->push_back( new AnnotatedBBox(*(*it)) );
    }

    // delete the last aoi allocated
    if (prev) {
	// allocated by previous filter
	prev->Delete(in);
    }

    return nonEmpty;
}

void GameManager::FillInterests(InterestList *reg,
	InterestList *unreg,
	GUIDMap *reg_assoc,
	InterestMap *curr)
{
    GameStore *store = GetStore();

    GUIDSet seen;
    AreaOfInterest globalAoi;

    DynamicGameObject *obj;
    store->BeginDynamic();
    while ( (obj = store->NextDynamic()) != NULL ) {
	if ( ! obj->IsReplica() ) {
	    GUID guid = obj->GetGUID();

	    seen.insert(guid);

	    if (m_ObjectFilters.find(guid) == m_ObjectFilters.end()) {
		m_ObjectFilters[guid] = 
		    m_InterestFactory->CreateIndividual(obj);
	    }
	    FilterChain *filters = m_ObjectFilters[guid];
	    ASSERT(filters);

	    // get the initial area of interest from the object
	    list<BBox> aoi;
	    uint32 ttl = obj->GetAreaOfInterest(&aoi, m_World);

	    // convert it to annotated bboxes
	    AreaOfInterest indivAoi;
	    for (list<BBox>::iterator it = aoi.begin(); 
		    it != aoi.end(); it++) {
		AnnotatedBBox *box = new AnnotatedBBox(*it, guid, ttl);
		indivAoi.push_back(box);
	    }

	    bool nonEmpty = 
		ProcessInterestFilters(filters, &globalAoi, &indivAoi);

	    // allocated by above code
	    for (AreaOfInterestIter it = indivAoi.begin();
		    it != indivAoi.end(); it++) {
		delete *it;
	    }
#ifdef SAVE_PREDICTED_AOI
	    // ****************************************************************
	    // for debugging and visualization
	    if (nonEmpty) {
		obj->ClearPredictedAreaOfInterest();
	    }
	    // ****************************************************************
#endif
	}
    }

    // remove the filter chains for the objects no longer present
    list<GUID> todelete;
    for (FilterMapIter it = m_ObjectFilters.begin(); 
	    it != m_ObjectFilters.end(); it++) {
	if (seen.find(it->first) == seen.end()) {
	    m_InterestFactory->Delete(it->second);
	    todelete.push_back(it->first);
	}
    }
    for (list<GUID>::iterator it = todelete.begin(); 
	    it != todelete.end(); it++) {
	m_ObjectFilters.erase(*it);
    }

    // construct a global filter if necessary
    if (m_GlobalFilter == NULL) {
	m_GlobalFilter = m_InterestFactory->CreateGlobal();
    }

    // pass the global aoi through the global filters
    AreaOfInterest finalAoi;
    ProcessInterestFilters(m_GlobalFilter, &finalAoi, &globalAoi);

    // delete input
    for (AreaOfInterestIter it = globalAoi.begin();
	    it != globalAoi.end(); it++) {
	delete *it;
    }

    // convert final aoi into interests
    for (AreaOfInterestIter it = finalAoi.begin();
	    it != finalAoi.end(); it++) {
	AnnotatedBBox *box = *it;
	BBox final = m_World->Clip(*box->GetBBox());
	// don't subscribe to points!
	if (final.min == final.max) continue;

	OMInterest *in = m_Manager->CreateInterest();
	final.FillInterest(in);
	//box->GetBBox()->FillInterest(in);
	in->SetLifeTime(box->GetTTL());

	for (GUIDSetIter it2 = box->GetGUIDs()->begin();
		it2 != box->GetGUIDs()->end(); it2++) {
	    reg_assoc->insert(pair<GUID,GUID>(*it2, in->GetGUID()));
	}

	reg->push_back(in);

#ifdef SAVE_PREDICTED_AOI
	// ********************************************************************
	// for debugging and visualization
	for (GUIDSetIter it = box->GetGUIDs()->begin();
		it != box->GetGUIDs()->end(); it++) {
	    GUID g = *it;
	    DynamicGameObject *obj =
		dynamic_cast<DynamicGameObject *>( store->Find(g) );
	    if (obj) {
		obj->AddPredictedAreaOfInterest(*box->GetBBox());
	    }
	}
	// ********************************************************************
#endif

    }

    // XXX TODO: do some unregistrations?

    // finally, delete the final output
    for (AreaOfInterestIter it = finalAoi.begin();
	    it != finalAoi.end(); it++) {
	delete *it;
    }
}

GObject *GameManager::Construct(GObjectInfoIface *info,
	Packet *pkt, 
	const DeltaMask& mask,
	SIDMap *unresolved)
{
    ASSERT(m_Module);

    // XXX FIXME: don't crash -- is is from the network so it might be
    // garbage...
    ASSERT(mask.IsSet(0));
    gTypeCode type = pkt->ReadInt();
    pkt->ResetBufPosition();

    // create a baseline object
    DynamicGameObject *obj = m_Module->ConstructObject(type, info);
    ASSERT(obj);

    // unpack it from the delta
    obj->UnpackUpdate(pkt, mask, unresolved);

    return obj;
}

void GameManager::Destroy(GUID guid)
{
    ASSERT(m_Store);
    ASSERT(m_Module);

    // XXX - clean this up
    GameObject *obj = m_Store->Find(guid);
    if (!obj)
	obj = static_cast<DynamicGameObject *>(m_Store->m_DynamicPending->Find(guid));
    m_Store->m_DynamicStore->_ApplicationRemove(guid);
    m_Store->m_DynamicPending->_ApplicationRemove(guid);
    if (obj) {
	ASSERT(obj->IsDynamic());
	m_Module->DeleteObject(dynamic_cast<DynamicGameObject *>(obj));
	ASSERT(m_Store->Find(guid) == NULL);
    }
}

///////////////////////////////////////////////////////////////////////////////

void GameManager::LogStoreStats()
{
    DGStoreLogEntry e;

    DynamicGameObject *obj;

    m_Manager->Lock(); // XXX HACK
    m_Store->BeginDynamic();
    while ((obj = m_Store->NextDynamic())) {
	e.no++;

	if (!obj->IsReplica()) {
	    e.np++;
	    switch (obj->GetVisualizerType ()) { 
	    case VisType_Bot:
		e.p_pl++;
		break;
	    case VisType_Missile:
		e.p_mi++;
		break;
	    case VisType_Item:
		e.p_it++;
		break;
	    default:
		e.p_ot++;
	    }
	}
	else {
	    e.nr++;
	    switch (obj->GetVisualizerType ()) { 
	    case VisType_Bot:
		e.r_pl++;
		break;
	    case VisType_Missile:
		e.r_mi++;
		break;
	    case VisType_Item:
		e.r_it++;
		break;
	    default:
		e.r_ot++;
	    }
	}
    }
    m_Manager->Unlock();

    e.frameno = GetFrameNumber();

    LOG(DGStoreLog, e);
}

void GameManager::LogInterestStats()
{
    ObjectInterestEntry e;

    e.frameno = GetFrameNumber();

    DynamicGameObject *obj;
    m_Manager->Lock(); // XXX HACK
    m_Store->BeginDynamic();
    while ((obj = m_Store->NextDynamic()) != NULL) {
	if (obj->IsReplica())
	    continue;

	// At least better than obj->GetType ()
	switch (obj->GetVisualizerType ()) {
	case VisType_Bot:
	    e.type = ObjectInterestEntry::PLAYER;
	    break;
	case VisType_Missile:
	    e.type = ObjectInterestEntry::MISSILE;
	    break;
	default:
	    continue;
	}

	e.guid = obj->GetGUID();
	e.orig = obj->GetOrigin();

	// this is a special log
	if (e.type == ObjectInterestEntry::PLAYER) { 
	    PositionEntry p;
	    p.frameno = GetFrameNumber ();
	    p.player  = (uint32) e.guid.GetLocalOID ();
	    p.origin  = obj->GetOrigin ();

	    LOG(PositionLog, p);

	}

	// this is the non-predictive-sub
	list<BBox> sub;
	obj->GetAreaOfInterest(&sub, m_World);
	ASSERT(sub.begin() != sub.end());
	e.sub = *sub.begin();

	LOG(ObjectInterestLog, e);
    }
    m_Manager->Unlock(); // XXX HACK
}

struct _obj_info {
    uint32 total_size;
    char *classname;
};

// Wow, this is sort of inefficient. Seems to do obj->PackUpdate
// again here just to record stats. oh well.
//
void GameManager::LogObjectDeltaStats()
{
    // XXX HACK (should not be static global)
    // XXX This assumes total sizes remain the same
    static map<GUID, _obj_info, less_GUID> info;
    static Packet pkt(0xFFFF);

    ObjectDeltaEntry de;
    GUIDLogEntry ge;

    de.frameno = GetFrameNumber();

    DynamicGameObject *obj;
    m_Manager->Lock(); // XXX HACK
    m_Store->BeginDynamic();
    while ((obj = m_Store->NextDynamic()) != NULL) {
	if (obj->IsReplica())
	    continue;

	de.guid = obj->GetGUID();
	ge.guid = obj->GetGUID();

	if (info.find(obj->GetGUID()) == info.end()) {
	    // new object
	    const DeltaMask& mask = obj->GetInitDeltaMask();
	    pkt.ResetBufPosition();
	    obj->PackUpdate(&pkt, mask);
	    uint32 len = pkt.GetBufPosition ();

	    _obj_info ifo;
	    ifo.total_size = len;

	    // Atleast somewhat better than obj->GetType ()
	    switch (obj->GetVisualizerType ()) {
	    case VisType_Bot:
		ifo.classname = "player";
		break;
	    case VisType_Missile:
		ifo.classname = "missile";
		break;
	    default:
		ifo.classname = "unknown";
	    }

	    // this supersedes the above.
	    ifo.classname = (char *) obj->GetClassName ();
	    info[obj->GetGUID()] = ifo;

	    de.type = ObjectDeltaEntry::NEW;
	    de.deltaSize = len;
	    de.fullSize  = len;

	    ge.status = GUIDLogEntry::CREATE;
	    ge.classname = ifo.classname;
	    LOG(GUIDLog, ge);
	} else {
	    // update object
	    const DeltaMask& mask = obj->GetDeltaMask();
	    pkt.ResetBufPosition();
	    obj->PackUpdate(&pkt, mask);
	    uint32 len  = pkt.GetBufPosition ();
	    uint32 full = info[obj->GetGUID()].total_size;

	    de.type = ObjectDeltaEntry::UPDATE;
	    de.deltaSize = len;
	    de.fullSize  = full;
	}

	LOG(ObjectDeltaLog, de);
    }

    // find deleted objects
    for (map<GUID, _obj_info, less_GUID>::iterator it = info.begin();
	    it != info.end(); /* !!! */) {
	map<GUID, _obj_info, less_GUID>::iterator oit = it;
	oit++;
	if (m_Store->Find(it->first) != NULL) {
	    it = oit;
	    continue;
	}

	// deleted!
	de.guid = it->first;
	de.type = ObjectDeltaEntry::DELETE;
	de.deltaSize = 0;
	de.fullSize  = 0;

	LOG(ObjectDeltaLog, de);

	ge.guid = it->first;
	ge.status = GUIDLogEntry::DESTROY;
	ge.classname = it->second.classname;
	LOG(GUIDLog, ge);

	info.erase(it);
	it = oit;
    }

    m_Manager->Unlock(); // XXX HACK
}

///////////////////////////////////////////////////////////////////////////////

static void handleVisIP(ostream& out, char **args);
static void handleGetVisInfo(ostream& out, char **args);
static void handleBandwidthStats(ostream& out, char **args);

static InputHandler g_Handlers[] = {
    { "VISIP", "VI", handleVisIP, "ipaddress [pkts-per-second]",
	"Set the IP the visualizer should send to (use 'none' to disable)" },    
    { "GETVIS", "GV", handleGetVisInfo, "",
	"Receive updates the visualizer would have sent over the udp connection" },
    { "BWSTATS", "BW", handleBandwidthStats, "[OM|MERC|ALL]", 
	"Retrieve bandwidth stats for {OM layer, MERC layer, ALL layers (default)}" },
    { NULL, NULL, NULL, NULL }
};

void GameManager::InstallTerminalHandlers(InputHandlers *tofill)
{
    InputHandler *ih = g_Handlers;
    while (ih->cmd) {
	tofill->push_back(ih);
	++ih;
    }

    m_Module->InstallTerminalHandlers (tofill);
}

// From RealNet::__GetSubTypeForRouting...
// XXX FIXME this is bad bad bad. should be in RealNet.cpp
// but this is only for demo purposes
//
#include <mercury/RoutingLogs.h>
#include <mercury/MercMessage.h>
#include <om/OMMessage.h>
#include <wan-env/RealNet.h>

static bool _isMerc(MsgType t) {
    return t < MSG_MERCURY_SENTINEL;
}
static bool _isMercSub(MsgType t) {
    return t == MSG_SUB || t == MSG_SUB_LINEAR || t == MSG_SUB_NOTROUTING;
}
static bool _isMercPub(MsgType t) {
    return t == MSG_PUB || t == MSG_RANGE_PUB_NOTROUTING || 
	t == MSG_RANGE_PUB_LINEAR || t == MSG_RANGE_PUB;
}
static bool _isMercMatch(MsgType t) {
    return t == MSG_MATCHED_PUB || t == MSG_RANGE_MATCHED_PUB;
}
static bool _isOM(MsgType t) {
    return t >= MSG_APP_UPDATE;
}
static void _addEntry (AggMeasurement *one, AggMeasurement *two) {
    one->nsamples += two->nsamples;
    one->size += two->size;
}

static void _FillBWStats (
	RealNet *rnet, 
	AggMeasurement &om, 
	AggMeasurement &merc_pub, 
	AggMeasurement &merc_sub, 
	AggMeasurement &merc_match,
	bool isInbound
	)
{
    MeasurementMap *mesmap;

    if (isInbound)
	mesmap = &rnet->GetInboundMeasurementMap ();
    else
	mesmap = &rnet->GetOutboundMeasurementMap ();

    for (MeasurementMap::iterator it = mesmap->begin (); it != mesmap->end (); ++it) {
	byte type = it->first;
	if (_isOM (type)) {
	    _addEntry (&om, &it->second);
	}
	else if (_isMercSub (type)) {
	    _addEntry (&merc_sub, &it->second);
	}
	else if (_isMercPub (type))  {
	    _addEntry (&merc_pub, &it->second);
	}
	else if (_isMercMatch (type)) {
	    _addEntry (&merc_match, &it->second);
	}
    }
}

void handleBandwidthStats (ostream& out, char **args) 
{
    Manager *man = GameManager::GetInstance ()->GetManager ();
    WANMercuryNode *merc = dynamic_cast<WANMercuryNode *> (man->GetMerc ());
    RealNet *rnet_one = dynamic_cast<RealNet *> (merc->GetNetwork ());
    RealNet *rnet_two = man->GetDirectRouter ()->GetNetwork ();
    
    bool showOM = true;
    bool showMerc = true;
    if (args [1]) {
	if (!strcasecmp (args [1], "OM")) showMerc = false;
	else if (!strcasecmp (args [1], "MERC")) showOM = false;
    }

    AggMeasurement om, merc_sub, merc_pub, merc_match;

    memset (&om, 0, sizeof (AggMeasurement));
    memset (&merc_sub, 0, sizeof (AggMeasurement));
    memset (&merc_pub, 0, sizeof (AggMeasurement));
    memset (&merc_match, 0, sizeof (AggMeasurement));


#define DUMP(tag,entry)  do { \
    out << tag << " " << entry.nsamples << " pkts, " << entry.size << " bytes " << endl; \
  } while (0)

    _FillBWStats (rnet_one, om, merc_sub, merc_pub, merc_match, true);
    _FillBWStats (rnet_two, om, merc_sub, merc_pub, merc_match, true);

    out << "OK:" << endl;
    if (showOM) {
	DUMP("OM inbound", om);
    }
    if (showMerc) {
	DUMP("MERC_SUB inbound", merc_sub);
	DUMP("MERC_PUB inbound", merc_pub);
	DUMP("MERC_MATCH inbound", merc_match);
    }

    memset (&om, 0, sizeof (AggMeasurement));
    memset (&merc_sub, 0, sizeof (AggMeasurement));
    memset (&merc_pub, 0, sizeof (AggMeasurement));
    memset (&merc_match, 0, sizeof (AggMeasurement));

    _FillBWStats (rnet_one, om, merc_sub, merc_pub, merc_match, false);
    _FillBWStats (rnet_two, om, merc_sub, merc_pub, merc_match, false);

    if (showOM) {
	DUMP("OM outbound", om);
    }
    if (showMerc) {
	DUMP("MERC_SUB outbound", merc_sub);
	DUMP("MERC_PUB outbound", merc_pub);
	DUMP("MERC_MATCH outbound", merc_match);
    }
}

void handleVisIP (ostream& out, char **args)
{
    if (!args[1]) {
	out << "ERROR: expected ipaddress or 'none'" << endl;
	return;
    }

    if (!strcasecmp(args[1], "NONE")) {
	strcpy(g_VisIP, "");
    } else {
	strncpy(g_VisIP, args[1], 255);
    }

    if (! GameManager::GetInstance()->GetVisualizer()
	    ->SetTargetIP(g_VisIP) ) {
	out << "ERROR: invalid address" << endl;
	return;
    }

    if (args[2]) {
	int rate = atoi (args[2]);
	if (!GameManager::GetInstance ()->GetVisualizer ()->SetUpdateRate (rate)) {
	    out << "ERROR: invalid update rate" << endl;
	    return;
	}
    }
    out << "OK" << endl;
}

void handleGetVisInfo(ostream& out, char **args)
{
    GameVisualizer *vis = GameManager::GetInstance ()->GetVisualizer ();
    
    out << "OK" << endl;
    vis->SendTCPUpdate (out);
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
