////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __TEST_LOGS_H__
#define __TEST_LOGS_H__

#include "common.h"
#include <util/IPEndPoint.h>
#include <util/ID.h>
#include <util/types.h>
#include <util/ExpLog.h>

struct QuakeStatsLogEntry : public ExpLogEntry
{
    double frameTime;  // total time between frames
    double frameSlack; // 100 - max(timeToExecFrame, 100)
    double execTime;   // total time to exec frame (excluding idle recv())
    double dgTime;     // total time spent executing dg stuff in frame
    double dgIdleTime; // total time spent in recv() during idle period

    QuakeStatsLogEntry() : 
	frameTime(0), frameSlack(0), execTime(0), dgTime(0), dgIdleTime(0) {}
    virtual ~QuakeStatsLogEntry() {}

    uint32 Dump(FILE *fp) {
	return fprintf(fp, "%13.3f\t%.3f\t%.3f\t%.3f\t%.3f\t%.3f\n", 
		(double)time.tv_sec + 
		(double)time.tv_usec/USEC_IN_SEC, 
		frameTime, frameSlack, 
		execTime, dgTime, dgIdleTime);
    }
};

struct DGStoreLogEntry : public ExpLogEntry
{
    uint16 nr, np, no;
    uint16 p_pl, p_mo, p_it, p_mi, p_ot;
    uint16 r_pl, r_mo, r_it, r_mi, r_ot;
    uint16 frameno;

    DGStoreLogEntry() : nr(0), np(0), no(0),
    p_pl(0), p_mo(0), p_it(0), p_mi(0), p_ot(0),
    r_pl(0), r_mo(0), r_it(0), r_mi(0), r_ot(0),
    frameno(0) {};
    virtual ~DGStoreLogEntry() {};

    uint32 Dump(FILE *fp) {
	return fprintf(fp, "%13.3f f %hu o %hu "
		"p %hu pl %hu mo %hu it %hu mi %hu ot %hu "
		"r %hu pl %hu mo %hu it %hu mi %hu ot %hu\n",
		(double)time.tv_sec + 
		(double)time.tv_usec/USEC_IN_SEC,
		frameno, no,
		np, p_pl, p_mo, p_it, p_mi, p_ot,
		nr, r_pl, r_mo, r_it, r_mi, r_ot);
    }
};

struct ObjectInterestEntry : public ExpLogEntry
{
    enum { MISSILE = 'm', PLAYER = 'p', MONSTER = 'o', ITEM = 'i' };

    int    frameno;
    // guid of the object
    GUID   guid;
    // type of object (above)
    char type;
    // the origin point, min and max of predicted bbox
    Vec3 orig;
    // the current subscription
    BBox sub;

    ObjectInterestEntry() {}
    virtual ~ObjectInterestEntry() {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);

	ret += fprintf(fp, "\t%d", frameno);
	ret += _PrintGUID (fp, guid);
	ret += fprintf (fp, "\t%c\t%.3f,%.3f,%.3f"
		"\t%.3f,%.3f,%.3f"
		"\t%.3f,%.3f,%.3f\n",
		type,
		orig[0], orig[1], orig[2], 
		sub.min[0], sub.min[1], sub.min[2], 
		sub.max[0], sub.max[1], sub.max[2]);
	return ret;
    }
};

// for projecting cost for client/server or broadcast
struct ObjectDeltaEntry : public ExpLogEntry
{
    enum { NEW = 'n', DELETE = 'd', UPDATE = 'u' };

    uint32 frameno;
    GUID   guid;
    char   type;
    uint32 deltaSize;
    uint32 fullSize;

    ObjectDeltaEntry() {}
    virtual ~ObjectDeltaEntry() {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += fprintf (fp, "\t%d", frameno);	
	ret += _PrintGUID (fp, guid);
	ret += fprintf(fp, "\t%c\t%d\t%d\n",
		type, deltaSize, fullSize);
	return ret;
    }
};

// when each primary object is created or destroyed
struct GUIDLogEntry : public ExpLogEntry
{
    enum { CREATE = 'C', DESTROY = 'D', UPDATE = 'U' };

    char  status;
    GUID guid;
    char *classname;

    GUIDLogEntry() : status(UPDATE) {}
    virtual ~GUIDLogEntry() {}

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += _PrintGUID (fp, guid);

	ret += fprintf(fp, "\t%s\t%c\n",
		classname,
		status);
	return ret;
    }
};

struct ClientLogEntry : public ExpLogEntry
{
    uint16 frameno;
    GUID guid;
    float update;
    float estimate;

    uint32 Dump(FILE *fp) {
	uint32 ret = _PrintTime (fp, time);
	ret += fprintf (fp, "\t%hu", frameno);	
	ret += _PrintGUID (fp, guid);

	ret += fprintf(fp, "\t%f\t%f\n",
		update, estimate);
	return ret;
    }
};

struct PositionEntry : public ExpLogEntry 
{
    int     frameno;
    uint32  player;
    Vec3    origin;

    uint32 Dump (FILE *fp) { 
	return fprintf(fp, "%d\t%d\t%.3f,%.3f,%.3f\n", frameno, (int) player,
		origin [0], origin [1], origin [2]);
    }
};

DECLARE_EXPLOG(QuakeStatsLog, QuakeStatsLogEntry);
DECLARE_EXPLOG(DGStoreLog, DGStoreLogEntry);
DECLARE_EXPLOG(ObjectInterestLog, ObjectInterestEntry);
DECLARE_EXPLOG(ObjectDeltaLog, ObjectDeltaEntry);
DECLARE_EXPLOG(GUIDLog, GUIDLogEntry);
DECLARE_EXPLOG(ClientLog, ClientLogEntry);
DECLARE_EXPLOG(PositionLog, PositionEntry);

extern void InitGameLogs(SID& sid);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
