////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameInterestManager.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_INTEREST_MANAGER_H__
#define __GAME_INTEREST_MANAGER_H__

#include <util/ID.h>
#include <gameapi/common.h>

    class DynamicGameObject;

class AnnotatedBBox {
 private:
    GUIDSet m_GUIDs;
    uint32 m_TTL;
    BBox m_Box;
 public:
    AnnotatedBBox(const BBox& box, const GUID& guid, uint32 ttl) : 
	m_Box(box), m_TTL(ttl) {
	if (guid != GUID_NONE) {
	    m_GUIDs.insert(guid);
	}
    }
    AnnotatedBBox(const BBox& box, const GUIDSet *guids, uint32 ttl) : 
	m_Box(box), m_TTL(ttl), m_GUIDs(*guids) {
    }

    void AddGUID(const GUID& guid) {
	m_GUIDs.insert(guid);
    }

    void AddGUIDs(const GUIDSet *guids) {
	m_GUIDs.insert(guids->begin(), guids->end());
	/*
	  for (GUIDSetIter it = guids->begin(); it != guids->end(); it++) {
	  AddGUID(*it);
	  }
	*/
    }

    void SetTTL(uint32 ttl) {
	m_TTL = ttl;
    }

    uint32 GetTTL() {
	return m_TTL;
    }

    BBox *GetBBox() {
	return &m_Box;
    }

    GUIDSet *GetGUIDs() {
	return &m_GUIDs;
    }
};

typedef list<AnnotatedBBox *> AreaOfInterest;
typedef AreaOfInterest::iterator AreaOfInterestIter;

/**
 * An Interest is basically a "read". In our game application, we assume the
 * following:
 *
 * <ol>
 * <li> Each object issues a read which describes its current absolute
 *      area of interest. This is a set of hyper-rectangles. For now we
 *      only support 3D rectangles. It is assumed that each object submits
 *      an interest every frame unless it (1) never submits an interest,
 *      or (2) is not longer present.
 * <li> A set of these reads is issued once per frame. This set forms a
 *      single "global-read".
 * </ol>
 *
 * The responsibility of this module is to mutate that global-read into
 * a "more optimal" set of interests by some optimality metric to submit
 * to the system.
 *
 * There are two levels of filters. First, each object has an individual
 * set of interest filters, each of which may keep history. Second, the
 * output of all these filters is merged into a global area of interest.
 * Then we apply the global set of filters to this global area of interest.
 *
 * The following diagram may help illustrate:
 *
 * <pre>
 * obj1.aoi -> filter11 -> ... -> filter1k = aoi1
 * ...
 * objn.aoi -> filtern1 -> ... -> filternk = aoin 
 *
 * {aoi1,...,aoin} -> filter1 -> ... -> filtern = aoi_final
 * </pre>
 */
class InterestFilter {
 public:
    virtual ~InterestFilter () {}

    virtual AreaOfInterest *Filter(AreaOfInterest *in) = 0;
    virtual void Delete(AreaOfInterest *aoi) = 0;
};

typedef list<InterestFilter *> FilterChain;
typedef FilterChain::iterator FilterChainIter;

class GameInterestFactory {
 public:
    virtual ~GameInterestFactory () {}

    /**
     * Constructs a filter chain for each individual object.
     */
    virtual FilterChain *CreateIndividual(DynamicGameObject *o) = 0;

    /**
     * Constructs the filter chain for all primary objects on this node.
     */
    virtual FilterChain *CreateGlobal() = 0;

    /**
     * Delete a filter chain.
     */
    virtual void Delete(FilterChain *chain) = 0;
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
