////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/*
 * Visualizer (v3)
 *
 * (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
 * (C) 2003      Ashwin Bharambe  (  ashu@cs.cmu.edu    )
 *
 * Reimplemented the visualizer (again :P). This time using OpenGL and SDL
 * so hopefully performance will no longer be a problem. You no longer have
 * to pass in the names of servers as inputs, the visualizer will learn them
 * as it receives packets. To toggle between servers, use the left and right
 * mouse buttons.
 *
 * Usage: viz [options] [map_image_file]
 *
 * If the visualizer says "warning: no time to read all the packets!"
 * consistently, that means that it can't keep up with the number of
 * packets it is receiving.  You can try reducing the --framerate to
 * give it more time beween each frame to receive packets.
 *
 */

using namespace std;

#include <set>
#include <map>
#include <list>
#include <vector>
#include <util/types.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <string>
#include <arpa/inet.h>

#include <SDL/SDL.h>

#if defined(_WIN32)
#include <windows.h>
#include <gl/gl.h>
#include <gl/glu.h>
#include <SDL/SDL_image.h>
#elif defined (__APPLE__)
#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <SDL_image/SDL_image.h>
#else
#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL/SDL_image.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "viz.h"
#include "FontUtil.h"

#define NO_COLYSEUS_INCLUDE
#include <gameapi/GameVisualizer.h>
#include "TimeVal.h"
#include "Options.h"

// XXX HACK -- conflicts with gameapi/common.h
#define BBox VizBBox

///////////////////////////////////////////////////////////////////////////////
//
// Options
//

static uint32 gFrameRate;
static char   gMapfile[255];
static char   gMapDir[255];
static char   gDimFile[255];
static uint32 gVizPort;
static int    gScaleSize;
static bool   gShowPsubs;
static bool   gShowSubs;
static bool   gShowNonselPrims;
static bool   gShowNonselReps;
static bool   gShowWayPoints;
static bool   gShowBuildings;
static bool   gShowLinks;

OptionType gOptions[] = {
    { 'f', "framerate", OPT_INT,
      "number of screen refreshes per second", 
      &gFrameRate, "20", NULL},
    { 'i', "imagefile", OPT_STR,
      "image file for background map", 
      gMapfile, "empty.jpg", NULL},
    { 'm', "map", OPT_STR,
      "directory containing testgame map files (buildings,waypoints,etc.)", 
      gMapDir, "", NULL},
    { 'd', "dims", OPT_STR,
      "dimension file containing the dimensions of a quake* map",
      gDimFile, "", NULL},
    { '/', "scalesize", OPT_INT,
      "scale for dots", 
      &gScaleSize, "50", NULL},
    { 'p', "port", OPT_INT,
      "visualizer udp port", 
      &gVizPort, "60000", NULL},
    { 'P', "no-psubs", OPT_BOOL|OPT_NOARG,
      "don't show predicted subscriptions (toggle: p)", 
      &gShowPsubs, "1", (void *)"0"},
    { 'S', "no-subs", OPT_BOOL|OPT_NOARG,
      "don't show regular subscriptions (toggle: s)", 
      &gShowSubs, "1", (void *)"0"},
    { 'N', "no-nonsel-prims", OPT_BOOL|OPT_NOARG,
      "don't show non-selected primaries (toggle: n)", 
      &gShowNonselPrims, "1", (void *)"0"},
    { 'r', "nonsel-reps", OPT_BOOL|OPT_NOARG,
      "show non-selected replicas (toggle: r)", 
      &gShowNonselReps, "0", (void *)"1"},
    { 'W', "no-waypoints", OPT_BOOL|OPT_NOARG,
      "don't show waypoints (toggle: w)", 
      &gShowWayPoints, "1", (void *)"0"},
    { 'B', "no-buildings", OPT_BOOL|OPT_NOARG,
      "don't show buildings (toggle: b)", 
      &gShowBuildings, "1", (void *)"0"},
    { 'L', "no-links", OPT_BOOL|OPT_NOARG,
      "don't show interest links (toggle: l)", 
      &gShowLinks, "1", (void *)"0"},
    { 0, 0, 0, 0, 0, 0, 0 }
};

///////////////////////////////////////////////////////////////////////////////
//
// Data Structures
//

typedef struct {
    byte type;
    const char *name;
    byte color[3];
    real32 width, height; /* x, y */
} classinfo_t;

typedef struct {
    const char *name;
    real32 scale;
    real32 min[3];
    real32 max[3];
} mapinfo_t;

typedef struct {
    real32 min[3];
    real32 max[3];
} BBox;

typedef struct {
    byte mask;
    real32 orig[3];
    BBox sub;
    BBox psub;

    // cached values recomputed each frame
    real32 color[3];
    real32 width, height;
} Entity;

typedef struct {
    real32 pts[4][3];
} Building;

struct WayPoint {
    bool goal;
    real32 orig[3];
    real32 radius;
};

struct Edge {
    real32 start[3];
    real32 end[3];
    Edge(real32 s[3], real32 e[3]) {
	memcpy(start, s, sizeof(start));
	memcpy(end, e, sizeof(end));
    }
};

ostream& operator<<(ostream& out, const real32 vec[3])
{
    return ( out << "(" << vec[0] << "," << vec[1] << "," << vec[2] << ")" );
}

ostream& operator<<(ostream& out, const BBox& box)
{
    return ( out << "[min=" << box.min << " max=" << box.max << "]" );
}

ostream& operator<<(ostream& out, const Entity& ent)
{
//    return( out << "(mask=" << (int)ent.mask << " orig=" << ent.orig <<
//	    " sub=" << ent.sub << " psub=" << ent.psub << ")" );
    string type = "other";
    if (ent.mask & VisType_Bot) 
	type = "bot";
    else if (ent.mask & VisType_Player)
	type = "player";
    else if (ent.mask & VisType_Missile)
	type = "missile";

    out << type;
    if (ent.mask & STATUS_REPLICA) 
	out << "(replica)";
    else
	out << "(primary)";
    out << " origin=" << ent.orig;
    return out;
}

template<class T>
struct frame_collection {
    uint32 lastframe, currframe;
    list<T> *last;
    list<T> *curr;
};

typedef struct {
    // entities
    frame_collection<Entity> ents;
    // interest links
    frame_collection<VizInterestLink> links;
    // waypoint graph XXX deprecated
    frame_collection<real32 *> nodes;
    frame_collection< pair<real32 *, real32 *> > edges;

    /*
      uint32 curframe, wpcurframe, linkscurframe;
      uint32 lastframe;
      list<Entity> *lastframeents;
      list<Entity> *curframeents;

      list< real32 * > *lastframenodes;
      list< real32 * > *curframenodes;
      list< pair<real32 *, real32 *> > *lastframeedges;
      list< pair<real32 *, real32 *> > *curframeedges;

      list<VizInterestLink> *lastframelinks;
      list<VizInterestLink> *curframelinks;
    */
} serverinfo_t;

struct less_sockaddr_in {
    bool operator()(const sockaddr_in& a, const sockaddr_in& b) {
	return a.sin_addr.s_addr < b.sin_addr.s_addr ||
	    (a.sin_addr.s_addr == b.sin_addr.s_addr &&
	     a.sin_port < b.sin_port);
    }
};

bool operator==(const sockaddr_in& a, const sockaddr_in& b)
{
    return a.sin_addr.s_addr == b.sin_addr.s_addr &&
	a.sin_port == b.sin_port;
}

// entities
static classinfo_t gClassInfo[] = {
    { VisType_Missile, "missile", { 0x00, 0x00, 0xcc }, 0.5, 0.5 },
    { VisType_Bot, "bot", { 0xcc, 0x00, 0x00 }, 1, 1 },
    { VisType_Player, "player", { 0x00, 0xcc, 0x33 }, 1, 1 },
    { VisType_Item, "item", { 0x00, 0xcc, 0xcc }, 0.5, 0.5 },
    { VisType_Other, "other", { 0xcc, 0x55, 0xcc }, 0.5, 0.5 },
    { 0, NULL, { 0, 0, 0 }, 0, 0 }
};

// maps
static mapinfo_t gMapInfo[] = {
    { "map.jpg", 10, {-1408, -256, 0 }, { 896, 704, 1 } },
    { "bigmap.jpg", 100, {-4096, -4096, 0 }, { 4096, 4096, 1 } },
    { "empty.jpg", 50, {0, 0, -8 }, { 1300, 1300, 8 } },
    // { "q3dm1.jpg", 50, {-9, -221, -141}, {1353, 2501, 961} },
    { 0, 0, {0,0,0}, {0,0,0} }
};

// window
static int gWidth = 924;
static int gHeight = 924;
static int gVideoBpp = 0, gVideoFlags = 0;
// world
static BBox gWorld;
static real32 gRange[3];
static GLuint gMapTexture;
static vector<Building> gBuildings;
static vector<WayPoint> gWayPoints;
static vector<Edge>     gEdges;

typedef map<sockaddr_in, serverinfo_t, less_sockaddr_in> Servers;
typedef Servers::iterator ServersIter;

static int gSock;
static Servers gServers;
static sockaddr_in gSelected;

///////////////////////////////////////////////////////////////////////////////
//
// Packet Code
//

static byte sg_Buffer[UDP_MAXLEN];
static int sg_BufPos = 0;
static int sg_BufLen = 0;
static Entity sg_edict;

static uint32 read_int()
{
    uint32 l;
    memcpy(&l, sg_Buffer + sg_BufPos, 4);
    l = ntohl( *(uint32 *)&l );
    sg_BufPos += 4;
    return l;
}

// Fucking g++4.0.3 has a bug. When this function is compiled with -O2
// and without making it 'inline', g++ just generates the wrong code. 
//- Ashwin [10/27/2006]
inline float read_float()
{
    uint32 l;

    memcpy (&l, (sg_Buffer + sg_BufPos), 4);
    l = ntohl(l);
    sg_BufPos += 4;
    float f = *(float *)&l;
    return f;
}

static byte read_byte()
{
    return sg_Buffer[sg_BufPos++];
}

static void read_vec(real32 pt[3]) {
    float x[3];
    // convert to [0.0,1.0] coordinates
    for (int i = 0; i < 3; i++) {
	x[i] = read_float ();
    }
     // fprintf (stderr, "read_vec: %.8f %.8f %.8f\n", x[0], x[1], x[2]);
    for (int i = 0; i < 3; i++) {
	pt[i] = (x[i]-gWorld.min[i])/gRange[i];
    }
}

static void read_bbox(BBox& box) {
    // clip to world
    read_vec(box.min);
    read_vec(box.max);

    for (int i=0; i<3; i++) {
	box.min[i] = MAX(box.min[i], gWorld.min[i]);
	box.max[i] = MIN(box.max[i], gWorld.max[i]);
    }
}

static Entity *deserialize()
{
    sg_edict.mask = read_byte();
    read_vec(sg_edict.orig);
    read_bbox(sg_edict.sub);
    read_bbox(sg_edict.psub);

    for (int i=0; i<2; i++) {
	if (0.0 > sg_edict.orig[i] ||
	    1.0 < sg_edict.orig[i]) {
	    return NULL;
	}
    }

    return &sg_edict;
}

static VizInterestLink *deserialize_link()
{
    static VizInterestLink link;

    link.from[0] = (read_float()-gWorld.min[0])/gRange[0];
    link.from[1] = (read_float()-gWorld.min[1])/gRange[1];
    link.to[0] = (read_float()-gWorld.min[0])/gRange[0];
    link.to[1] = (read_float()-gWorld.min[1])/gRange[1];
    link.stable = read_byte();

    return &link;
}


///////////////////////////////////////////////////////////////////////////////
//
// Network Code
//

static void init_network();
static void on_data_receive();
static void update_ents(sockaddr_in addr, uint32 framenum, list<Entity>& ents);
static void update_links(sockaddr_in addr, uint32 framenum, list<VizInterestLink>& ents);
static void update_waypoints(sockaddr_in addr, uint32 framenum, 
			     list<real32 *>& nodes, 
			     list< pair<real32 *, real32 *> >& edges);
void setup_network()
{
    sockaddr_in addr;

    /* create the socket */
    gSock = socket( PF_INET, SOCK_DGRAM, 0 );
    ASSERT(gSock >= 0);

    /* initialize the address structure */
    bzero(&addr, sizeof(struct sockaddr_in));
    addr.sin_family      = AF_INET;
    addr.sin_port        = htons( gVizPort );
    addr.sin_addr.s_addr = htonl( INADDR_ANY );

    /* bind the socket so we receive data on it -- check for errors here! */
    int ret = bind(gSock, (sockaddr *)&addr, sizeof(sockaddr_in));
    ASSERT(ret >= 0);
}

void on_data_receive()
{   
    sockaddr_in addr;
    socklen_t size = sizeof(sockaddr_in);
    sg_BufPos = 0;
    sg_BufLen = recvfrom(gSock, sg_Buffer, UDP_MAXLEN, 0,
			 (struct sockaddr *)&addr, &size);
    ASSERT(sg_BufLen >= 0);

    byte t = read_byte();

    // XXX hack: for some special packets coming to us from 
    if (t == 0xff) { 
	byte server_identifier = read_byte ();
	addr.sin_port = 100;
	addr.sin_addr.s_addr = (int) server_identifier;

	sg_BufLen -= 2;
	t = read_byte ();
    }

    // could retrieve `addr' from different means I guess.
    if (t == MSGTYPE_ENTITIES) {
	uint32 framenum = read_int();
//	cerr << " read frame " << framenum << endl;
#if 0
    {
	FILE *fp = fopen ("/tmp/read-log", "a");
	fprintf (fp, "ent-framenum=%d len=%d:::\n", framenum, sg_BufLen);
	for (int i = 0; i < sg_BufLen; i++) {
	    fprintf (fp, "%02x ", sg_Buffer[i]);
	    if (i != 0 && i % 20 == 0)
		fprintf (fp, "\n");
	}
	fprintf (fp, "\n");
	fclose (fp);
    }
#endif

	ASSERT( (sg_BufLen - 4 - 1) % EDICT_DUMP_SIZE == 0 );

	uint32 nents = (sg_BufLen - 4 - 1) / EDICT_DUMP_SIZE;

	list<Entity> ents;
	for (int i=0; i<(int)nents; i++) {
	    Entity *e = deserialize();
	    if (e) {
		// cerr << "got ent: " << *e << endl;
		ents.push_back(*e);
	    }
	}
	update_ents(addr, framenum, ents);

	serverinfo_t &info = gServers.find (addr)->second;
#if 0
	cerr << " ents.size = " << ents.size () << " info.curframe = " << info.ents.currframe << endl;
	if (info.ents.last) {
	    cerr << " last.size = " << info.ents.last->size () << endl;
	}
	if (info.ents.curr) {
	    cerr << " curr.size = " << info.ents.curr->size () << endl;
	}
#endif
    } else if (t == MSGTYPE_WAYPOINTS) {
	uint32 framenum = read_int();

	list<real32 *> nodes;
	list< pair<real32 *, real32 *> > edges;

	uint32 n = read_int();
	ASSERT( (int)n*12 <= sg_BufLen - 4 - 4 - 1 );
	for (uint32 i=0; i<n; i++) {
	    real32 *node = new real32[3];
	    read_vec(node);
	    nodes.push_back(node);
	}
	uint32 m = read_int();
	ASSERT( (int)m*2*12 <= (int)(sg_BufLen - 4 - 4 - 1 - n*12) );
	for (uint32 i=0; i<m; i++) {
	    real32 *a = new real32[3];
	    real32 *b = new real32[3];
	    read_vec(a);
	    read_vec(b);
	    edges.push_back(pair<real32 *,real32 *>(a,b));
	}
	ASSERT(n != 0 || m != 0);

	update_waypoints(addr, framenum, nodes, edges);
    } else if (t == MSGTYPE_INTGRAPH) {
	uint32 framenum = read_int();

	ASSERT( (sg_BufLen - 4 - 1) % LINK_DUMP_SIZE == 0 );

	uint32 n = (sg_BufLen - 4 - 1) / LINK_DUMP_SIZE;

	list<VizInterestLink> links;
	for (int i=0; i<(int)n; i++) {
	    VizInterestLink *link = deserialize_link();
	    if (link) {
		//DBG << "got ent: " << *e << endl;
		links.push_back(*link);
	    }
	}
	update_links(addr, framenum, links);
    } else {
	fprintf(stderr, "bad msg type %d\n", (int)t);
    }
}

serverinfo_t& get_serverinfo(sockaddr_in addr)
{
    ServersIter s = gServers.find(addr);
    if (s == gServers.end()) {
	// create a new entry for this server
	serverinfo_t info;
	bzero(&info, sizeof(info));
	gServers[addr] = info;
    }

    return gServers.find(addr)->second;
}

template<class T>
void update_collection(frame_collection<T>& c, uint32 framenum, list<T>& update)
{
    if (framenum > c.currframe) {
	// swap entity buffers
	if (c.last)
	    c.last->clear();
	else
	    c.last = new list<T>;
	c.lastframe = c.currframe;
	list<T> *last = c.last;
	c.last = c.curr;
	c.curr = last;
    }

    c.currframe = framenum;
    c.curr->splice(c.curr->end(), update);
}

void update_ents(sockaddr_in addr, uint32 framenum, list<Entity>& ents)
{
    serverinfo_t& info = get_serverinfo(addr);
    update_collection<Entity>(info.ents, framenum, ents);
}

void update_links(sockaddr_in addr, uint32 framenum, list<VizInterestLink>& links)
{
    serverinfo_t& info = get_serverinfo(addr);
    update_collection<VizInterestLink>(info.links, framenum, links);
}

void update_waypoints(sockaddr_in addr, uint32 framenum, 
		      list<real32 *>& nodes, 
		      list< pair<real32 *, real32 *> >& edges)
{
    ASSERT(false); // XXX Broken
}

// process X packets at a time
#define MAX_PACKET_BATCH 10

static int process_network(const TimeVal& timeout)
{
    fd_set fds;

    FD_ZERO(&fds);
    FD_SET(gSock, &fds);

    TimeVal start, to;
    int ready, count = 0;
    bool finished = false;

    while ( 1 ) {
	gettimeofday(&start, NULL);
	if (timeout <= start) {
	    if (select(gSock+1, &fds, NULL, NULL, &TIME_NONE) > 0) {
		cerr << "warning: no time to read all the packets!" << endl;
	    }
	    break;
	}
	to = TIME_NONE + (timeout - start);
	for (int i=0; i<MAX_PACKET_BATCH; i++) {
	    ready = select(gSock+1, &fds, NULL, NULL, &to);

	    if (ready > 0) {
		on_data_receive();
		count++;
	    } else {
		// timed-out
		finished = true;
		break;
	    }
	}
	/*
	if (finished)
	    break;
	*/
    }

    return count;
}

///////////////////////////////////////////////////////////////////////////////
//
// Input Handling Code
//

static void load_texture(const char *mapfile);

static void quit( int code )
{
    /*
     * Quit SDL so we can release the fullscreen
     * mode and restore the previous video settings,
     * etc.
     */
    SDL_Quit( );

    /* Exit program. */
    exit( code );
}

static void handle_key_down( SDL_keysym* keysym )
{
    switch( keysym->sym ) {
    case SDLK_ESCAPE:
    case SDLK_q:
	quit( 0 );
	break;
    case SDLK_p:
	gShowPsubs = !gShowPsubs;
	break;
    case SDLK_s:
	gShowSubs = !gShowSubs;
	break;
    case SDLK_n:
	gShowNonselPrims = !gShowNonselPrims;
	break;
    case SDLK_r:
	gShowNonselReps = !gShowNonselReps;
	break;
    case SDLK_w:
	gShowWayPoints = !gShowWayPoints;
	break;
    case SDLK_b:
	gShowBuildings = !gShowBuildings;
	break;
    case SDLK_l:
	gShowLinks = !gShowLinks;
	break;
    default:
	break;
    }

}

static void handle_mouse_down( SDL_MouseButtonEvent *ev )
{
    if (ev->state == SDL_PRESSED) {
	// toggle servers
	if (gSelected.sin_addr.s_addr == 0) {
	    if (gServers.size() > 0) {
		gSelected = gServers.begin()->first;
	    }
	} else {
	    if (ev->button == SDL_BUTTON_LEFT) {
		ServersIter s = gServers.find(gSelected);
		ASSERT (s != gServers.end());
		s++;
		if (s == gServers.end())
		    s = gServers.begin();
		gSelected = s->first;
		cout << "looking at: " << inet_ntoa(gSelected.sin_addr)
		     << ":" << ntohs(gSelected.sin_port) << endl;
	    } else if (ev->button == SDL_BUTTON_RIGHT) {
		ServersIter s = gServers.find(gSelected);
		ASSERT (s != gServers.end());
		if (s == gServers.begin())
		    s = gServers.end();
		s--;
		gSelected = s->first;
		cout << "looking at: " << inet_ntoa(gSelected.sin_addr)
		     << ":" << ntohs(gSelected.sin_port) << endl;
	    }
	}

    }
}

static void process_events( void )
{
    /* Our SDL event placeholder. */
    SDL_Event event;

    /* Grab all the events off the queue. */
    while( SDL_PollEvent( &event ) ) {

	switch( event.type ) {
	case SDL_KEYDOWN:
	    /* Handle key presses. */
	    handle_key_down( &event.key.keysym );
	    break;
	case SDL_MOUSEBUTTONDOWN:
	    handle_mouse_down( &event.button );
	    break;
	case SDL_VIDEORESIZE:
	    gWidth  = event.resize.w;
	    gHeight = event.resize.h;
	    SDL_SetVideoMode(gWidth, gHeight, gVideoBpp, gVideoFlags);
#ifdef _WIN32
	    // stupid windows requires texture reload on window resize
	    load_texture(gMapfile);
#endif
	    break;
	case SDL_QUIT:
	    /* Handle quit requests (like Ctrl-c). */
	    quit( 0 );
	    break;
	}	
    }

}

///////////////////////////////////////////////////////////////////////////////
//
// OpenGL Rendering Code
//

static void precompute_entity(const sockaddr_in& addr, Entity& ent)
{
    bool selected = addr == gSelected;

    for (int i=0; gClassInfo[i].name != NULL; i++) {
	if (gClassInfo[i].type & ent.mask) {
	    for (int j=0; j<3; j++) {
		ent.color[j] = (real32)gClassInfo[i].color[j]/255.0f;
		ent.width  = gScaleSize*gClassInfo[i].width/gRange[0];
		ent.height = gScaleSize*gClassInfo[i].width/gRange[1];
	    }
	    break;
	}
    }

    if (selected) {
	if (ent.mask & STATUS_REPLICA) {
	    for (int j=0; j<3; j++) {
		ent.color[j] *= 0.4;
	    }
	}
    } else {
	if (ent.mask & VisType_Player) {
	    ent.color [0] = 0.2f;
	    ent.color [1] = 0.8f;
	    ent.color [2] = 0.1f;
	}
	else {
	    ent.color[0] = 1.0f; 
	    ent.color[1] = 1.0f;
	    ent.color[2] = 0.0f;
	}
    }
}

static void render_aoi(const sockaddr_in& addr, const Entity& ent)
{
    bool selected = addr == gSelected;

    if ( selected && !(ent.mask & STATUS_REPLICA) && 
	    true 
	 /* (ent.mask & VisType_Bot || ent.mask & VisType_Missile) */ ) {

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	if (gShowPsubs) {
	    // predicted sub
	    glBegin(GL_POLYGON);
	    glColor4f(0.5*ent.color[0], 0.5*ent.color[1], 
		      0.5*ent.color[2], 0.3f);
	    glVertex2f(ent.psub.min[0], ent.psub.min[1]);
	    glVertex2f(ent.psub.min[0], ent.psub.max[1]);
	    glVertex2f(ent.psub.max[0], ent.psub.max[1]);
	    glVertex2f(ent.psub.max[0], ent.psub.min[1]);
	    glEnd();
	}

	if (gShowSubs) {
	    // instantaneous sub
	    glBegin(GL_POLYGON);
	    glColor4f(0.5*ent.color[0], 0.5*ent.color[1], 
		      0.5*ent.color[2], 0.5f);
	    glVertex2f(ent.sub.min[0], ent.sub.min[1]);
	    glVertex2f(ent.sub.min[0], ent.sub.max[1]);
	    glVertex2f(ent.sub.max[0], ent.sub.max[1]);
	    glVertex2f(ent.sub.max[0], ent.sub.min[1]);
	    glEnd();

	    glBegin(GL_LINE_STRIP);
	    glColor3f(0, 0, 0);
	    glColor4f(0.5*ent.color[0], 0.5*ent.color[1], 
		      0.5*ent.color[2], 0.7f);
	    glVertex2f(ent.sub.min[0], ent.sub.min[1]);
	    glVertex2f(ent.sub.min[0], ent.sub.max[1]);
	    glVertex2f(ent.sub.max[0], ent.sub.max[1]);
	    glVertex2f(ent.sub.max[0], ent.sub.min[1]);
	    glVertex2f(ent.sub.min[0], ent.sub.min[1]);
	    glEnd();
	}

	glDisable(GL_BLEND);
    }
}

static void render_entity(const sockaddr_in& addr, const Entity& ent)
{   
    bool selected = addr == gSelected;

    // don't draw replicas on other machines -- too much yellow crap
    if (!selected) {
	if (!gShowNonselReps && ent.mask & STATUS_REPLICA)
	    return;
	if (!gShowNonselPrims && !(ent.mask & STATUS_REPLICA))
	    return;
    }

    glBegin(GL_POLYGON);
    glColor3f(ent.color[0], ent.color[1], ent.color[2]);
    glVertex2f(ent.orig[0] - ent.width/2, ent.orig[1] - ent.height/2);
    glVertex2f(ent.orig[0] - ent.width/2, ent.orig[1] + ent.height/2);
    glVertex2f(ent.orig[0] + ent.width/2, ent.orig[1] + ent.height/2);
    glVertex2f(ent.orig[0] + ent.width/2, ent.orig[1] - ent.height/2);
    glEnd();

    glBegin(GL_LINE_STRIP);
    glColor3f(0, 0, 0);
    glVertex2f(ent.orig[0] - ent.width/2, ent.orig[1] - ent.height/2);
    glVertex2f(ent.orig[0] - ent.width/2, ent.orig[1] + ent.height/2);
    glVertex2f(ent.orig[0] + ent.width/2, ent.orig[1] + ent.height/2);
    glVertex2f(ent.orig[0] + ent.width/2, ent.orig[1] - ent.height/2);
    glVertex2f(ent.orig[0] - ent.width/2, ent.orig[1] - ent.height/2);
    glEnd();
}

static void render_waypoint_edge(const Edge& e)
{
    const real32 *from = e.start;
    const real32 *to   = e.end;

    // not sure where these artifacts come from...
    if (from[0] < 0.0 || from[1] < 0.0 || to[0] < 0.0 || to[1] < 0.0 ||
	from[0] > 1.0 || from[1] > 1.0 || to[0] > 1.0 || to[1] > 1.0)
	return;

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(0.0f, 1.0f, 0.0f, 0.25f);
    glBegin(GL_LINE_STRIP);
    glVertex2f(from[0], from[1]);
    glVertex2f(to[0], to[1]);
    glEnd();

    // XXX TODO: add an arrow or something to indicate direction
    // doesn't matter for now since we assume all are bidirectional

    glDisable(GL_BLEND);
}

static void render_waypoint_node(const WayPoint& wp)
{
    if (!wp.goal) return;

    real32 w = gScaleSize/gRange[0];

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(0.0f, 1.0f, 1.0f, 0.5f);
    glBegin(GL_POLYGON);
    glVertex2f(wp.orig[0] - w/2, wp.orig[1] - w/2);
    glVertex2f(wp.orig[0] - w/2, wp.orig[1] + w/2);
    glVertex2f(wp.orig[0] + w/2, wp.orig[1] + w/2);
    glVertex2f(wp.orig[0] + w/2, wp.orig[1] - w/2);
    glEnd();
    glDisable(GL_BLEND);
}

static void render_building(const Building& b)
{
    glBegin(GL_POLYGON);
    glColor3f(0.0f, 0.0f, 0.5f);
    for (uint32 i=0; i<4; i++) {
	glVertex2f(b.pts[i][0], b.pts[i][1]);
    }
    glEnd();
}

static void render_link(const VizInterestLink& link)
{
    const real32 *from = link.from;
    const real32 *to   = link.to;
    bool stable = link.stable;

    // not sure where these artifacts come from...
    if (from[0] < 0.0 || from[1] < 0.0 || to[0] < 0.0 || to[1] < 0.0 ||
	from[0] > 1.0 || from[1] > 1.0 || to[0] > 1.0 || to[1] > 1.0)
	return;

    if (stable) {
	glColor3f(1.0f, 0.0f, 1.0f);
    } else {
	glColor3f(0.5f, 0.0f, 0.5f);
    }
    glBegin(GL_LINE_STRIP);
    glVertex2f(from[0], from[1]);
    glVertex2f(to[0], to[1]);
    glEnd();
}

static void draw_screen(int width, int height )
{
    if (gSelected.sin_addr.s_addr == 0 && gServers.size() > 0) {
	gSelected = gServers.begin()->first;
    }

    glClearColor(1, 1, 1, 0);
    glClear(GL_COLOR_BUFFER_BIT);

    glViewport(0, 0, width, height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, 1, 1, 0, 0, 1);
    glMatrixMode(GL_MODELVIEW);

    // draw the map first
    glBindTexture (GL_TEXTURE_2D, gMapTexture);
    glEnable( GL_TEXTURE_2D );
    glBegin (GL_QUADS);
    glTexCoord2f (0.0, 0.0);
    glVertex2f (0.0, 0.0);
    glTexCoord2f (0.0, 1.0);
    glVertex2f (0.0, 1.0);
    glTexCoord2f (1.0, 1.0);
    glVertex2f (1.0, 1.0);
    glTexCoord2f (1.0, 0.0);
    glVertex2f (1.0, 0.0);
    glEnd ();
    glDisable( GL_TEXTURE_2D );

    // now draw buildings
    if (gShowBuildings) {
	for (uint32 i=0; i<gBuildings.size(); i++) {
	    render_building(gBuildings[i]);
	}
    }

    // now draw way points
    if (gShowWayPoints) {
	/*
	  ServersIter s = gServers.find(gSelected);
	  if (s != gServers.end()) {
	  if (s->second.lastframenodes) {
	  for (list< pair<real32 *, real32 *> >::iterator it = 
	  s->second.lastframeedges->begin();
	  it != s->second.lastframeedges->end(); it++) {
	  render_waypoint_edge(it->first, it->second);
	  }

	  for (list<real32 *>::iterator it = 
	  s->second.lastframenodes->begin();
	  it != s->second.lastframenodes->end(); it++) {
	  render_waypoint_node(*it);
	  }
	  }
	  }
	*/
	for (uint32 i=0; i<gWayPoints.size(); i++) {
	    render_waypoint_node(gWayPoints[i]);
	}
	for (uint32 i=0; i<gEdges.size(); i++) {
	    render_waypoint_edge(gEdges[i]);
	}
    }

    // first precompute per-frame entity info
    for (ServersIter s = gServers.begin(); s != gServers.end(); s++) {
	if (s->second.ents.last) {
	    for (list<Entity>::iterator it = s->second.ents.last->begin();
		 it != s->second.ents.last->end(); it++) {
		precompute_entity(s->first, *it);
	    }
	}
    }

    // then render the area of interest bboxes
    for (ServersIter s = gServers.begin(); s != gServers.end(); s++) {
	if (s->second.ents.last) {
	    for (list<Entity>::iterator it = s->second.ents.last->begin();
		 it != s->second.ents.last->end(); it++) {
		render_aoi(s->first, *it);
	    }
	}
    }

    // then render the entities of non-selected servers
    for (ServersIter s = gServers.begin(); s != gServers.end(); s++) {
	if (s->first == gSelected) continue;

	if (s->second.ents.last) {
	    for (list<Entity>::iterator it = s->second.ents.last->begin();
		 it != s->second.ents.last->end(); it++) {
		render_entity(s->first, *it);
	    }
	}
    }

    ServersIter s = gServers.find(gSelected);

    // then render interest links of selected server
    if (gShowLinks && s != gServers.end()) {
	if (s->second.links.last) {
	    for (list<VizInterestLink>::iterator it = s->second.links.last->begin();
		 it != s->second.links.last->end(); it++) {
		render_link(*it);
	    }
	}
    }

    // cerr << "rendering stuff " << endl;
    // then render the entities of the selected server
    if (s != gServers.end()) {
	if (s->second.ents.last) {
	    for (list<Entity>::iterator it = s->second.ents.last->begin();
		 it != s->second.ents.last->end(); it++) {
		render_entity(s->first, *it);
	    }
	}
    }

#if 0
    // render some text
    Text_Print("some text", Vec3(0,0,0), Vec3(0.01, 0.01, 0), 0.05);
#endif 

    SDL_GL_SwapBuffers( );
}

static void setup_opengl( )
{
    /* Our shading model--Gouraud (smooth). */
    glShadeModel( GL_SMOOTH );

    /* Culling. */
    glCullFace( GL_BACK );
    glFrontFace( GL_CCW );
    glEnable( GL_CULL_FACE );

    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
}

static void load_texture(const char *mapfile)
{
    // load map image as texture
    SDL_Surface *mapimg = IMG_Load(mapfile);
    ASSERT(mapimg);
    SDL_LockSurface(mapimg);

    cerr << "mapimg->w=" << mapimg->w << " mapimg->h=" << mapimg->h << endl;

    // w and h must be powers of 2
    for (int i=0; i<31; i++) {
	ASSERT(mapimg->w < (1<<i) || mapimg->w % (1<<i) == 0);
	ASSERT(mapimg->h < (1<<i) || mapimg->h % (1<<i) == 0);
    }

    if (mapimg->w > mapimg->h) 
	gHeight = (int) ((float) mapimg->h * (float) gWidth / (float) mapimg->w);
    else 
	gWidth = (int) ((float) mapimg->w * (float) gHeight / (float) mapimg->h);

    SDL_SetVideoMode (gWidth, gHeight, gVideoBpp, gVideoFlags);

    GLenum internal_format;
    GLenum img_format, img_type;
    switch (mapimg->format->BitsPerPixel) {
    case 32: 
	img_format = GL_RGBA; 
	img_type = GL_UNSIGNED_BYTE;
	internal_format = GL_RGBA8; 
	break;
    case 24: 
	img_format = GL_RGB; 
	img_type = GL_UNSIGNED_BYTE;
	internal_format = GL_RGB8; 
	break; 
    case 16: 
	img_format = GL_RGBA; 
	img_type = GL_UNSIGNED_SHORT_5_5_5_1;
	internal_format = GL_RGB5_A1; 
	break;
    default: 
	img_format = GL_LUMINANCE; 
	img_type = GL_UNSIGNED_BYTE;
	internal_format=GL_LUMINANCE8; 
	break;
    }

    glGenTextures(1, &gMapTexture);
    glBindTexture(GL_TEXTURE_2D, gMapTexture);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
    glTexImage2D (GL_TEXTURE_2D, 0, internal_format, mapimg->w, mapimg->h, 
		  0, img_format, img_type, mapimg->pixels);
    glError();

    SDL_UnlockSurface(mapimg);
    SDL_FreeSurface(mapimg);
}

static void load_mapdims(const char *filename)
{
    char line[255];
    FILE *f = fopen(filename, "r");
    if (f == NULL) {
	die("can't open mapdims file: %s", filename);
    }

    real32 x, y, z;

    fgets(line, 255, f);
    int ret = sscanf(line, "%f %f %f", &x, &y, &z);
    ASSERT(ret == 3);

    fclose(f);

    gWorld.min[0] = 0;
    gWorld.max[0] = x;
    gRange[0] = x;

    gWorld.min[1] = 0;
    gWorld.max[1] = y;
    gRange[1] = y;

    float frac = y/x;
    gHeight = (int)(frac*gWidth);
}

static void load_buildings(const char *filename)
{
    char line[255];
    FILE *f = fopen(filename, "r");
    if (f == NULL) {
	die("can't open mapbuildings file: %s", filename);
    }

    int count = 0;
    Building b;

    while (!feof(f)) {
	real32 x, y;

	fgets(line, 255, f);
	int ret = sscanf(line, "%f %f", &x, &y);
	ASSERT(ret == 2);

	uint32 index = count % 4;
	b.pts[index][0] = (x-gWorld.min[0])/gRange[0];
	b.pts[index][1] = (y-gWorld.min[1])/gRange[1];
	b.pts[index][2] = 0;

	count++;
	if (count % 4 == 0) {
	    gBuildings.push_back(b);
	}
    }

    // the last building frames the map, ignore it
    gBuildings.pop_back();

    fclose(f);
}

static void load_waypoints(const char *filename)
{
    char line[255];
    char *lret;
    int cret;

    FILE *fp = fopen(filename, "r");
    if (!fp) {
	die("couldn't open waypoint file %s", filename);
    }

    uint32 nodes;
    uint32 edges;

    // how many nodes and how many edges?
    lret = fgets(line, 255, fp);
    ASSERT(lret);
    cret = sscanf(line, "%d %d\n", &nodes, &edges);
    if (cret != 2) {
	die("bad 'nodes edges' line in waypoint file %s", filename);
    }
    ASSERT(nodes > 0);
    ASSERT(edges > 0);

    for (uint32 i=0; i<nodes; i++) {
	real32 x,y,z;
	real32 rad;
	int goal;

	lret = fgets(line, 255, fp);
	ASSERT(lret);
	cret = sscanf(line, "%f %f %f %f %d\n", &x,&y,&z,&rad,&goal);
	if (cret != 5) {
	    die("bad nodes line in waypoint file %s", filename);
	}

	WayPoint p;
	p.orig[0] = (x-gWorld.min[0])/gRange[0];
	p.orig[1] = (y-gWorld.min[1])/gRange[1];
	p.orig[2] = (z-gWorld.min[2])/gRange[2];
	p.radius  = rad;
	p.goal    = goal != 0;

	gWayPoints.push_back(p);
    }

    for (uint32 i=0; i<edges; i++) {
	int start;
	int end;

	lret = fgets(line, 255, fp);
	ASSERT(lret);
	cret = sscanf(line, "%d %d\n", &start, &end);
	if (cret != 2) {
	    die("bad edges line in waypoint file %s", filename);
	}

	ASSERT(start >= 0 && start < (int)gWayPoints.size());
	ASSERT(end >= 0 && end < (int)gWayPoints.size());

	Edge e(gWayPoints[start].orig, gWayPoints[end].orig);

	gEdges.push_back(e);
    }

    fclose(fp);
}

///////////////////////////////////////////////////////////////////////////////
//
// Main
//

int main( int argc, char* argv[] )
{
    argc = ProcessOptions(gOptions, argc, argv, false);
    PrintOptionValues (gOptions);

    {
	bool found = false;
	
	// set appropriate dimensions for map
	for (int n=0; gMapInfo[n].name != NULL; n++) {
	    if (!strcmp(gMapInfo[n].name, gMapfile)) {
		for (int i=0; i<3; i++) {
		    gWorld.min[i] = gMapInfo[n].min[i];
		    gWorld.max[i] = gMapInfo[n].max[i];
		    gRange[i] = gWorld.max[i] - gWorld.min[i];
		}
		gScaleSize = (int) gMapInfo[n].scale;
		
		found = true;
		break;		
	    }
	}

	if (!found && strcmp (gDimFile, "")) { 
	    FILE *fp = fopen (gDimFile, "r");
	    fscanf (fp, "%*s %f %f %f\n", &gWorld.min [0], &gWorld.min [1], &gWorld.min [2]);
	    fscanf (fp, "%*s %f %f %f\n", &gWorld.max [0], &gWorld.max [1], &gWorld.max [2]);

	    cerr << " min=" << gWorld.min << " max=" << gWorld.max << endl;
	    for (int i = 0; i < 3; i++) {
		gRange[i] = gWorld.max[i] - gWorld.min[i];
	    }

	    // for q3dm1, '40' was good, but we should probably scale things
	    // for smaller or larger maps. (larger map => smaller points)
	    // XXX: hmm does not look good.

	    fclose (fp);

	    found = true;
	}

	if (found) {
	    float y = gRange [1], x = gRange [0];
	    if (y > x) 
		gWidth = (int) ((x/y) * gHeight);
	    else
		gHeight = (int) ((y/x) * gWidth);
	}
		
	if (strcmp(gMapDir, "")) {
	    load_mapdims( (string(gMapDir) + "/dims").c_str() );
	    load_buildings( (string(gMapDir) + "/buildings").c_str() );
	    load_waypoints( (string(gMapDir) + "/waypoints").c_str() );
	}

	// initially select no servers
	bzero(&gSelected, sizeof(gSelected));
    }

    {
	// SDL Setup

	/* Information about the current video settings. */
	const SDL_VideoInfo* info = NULL;
	/* Color depth in bits of our window. */

	if( SDL_Init( SDL_INIT_VIDEO ) < 0 ) {
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    quit( 1 );
	}

	info = SDL_GetVideoInfo( );
	if( !info ) {
	    fprintf( stderr, "Video query failed: %s\n",
		     SDL_GetError( ) );
	    quit( 1 );
	}

	gVideoBpp = info->vfmt->BitsPerPixel;

	SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 5 );
	SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 5 );
	SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 5 );
	SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
	SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

	gVideoFlags = SDL_OPENGL | SDL_RESIZABLE;

	/*
	 * Set the video mode
	 */
	if(SDL_SetVideoMode( gWidth, gHeight, gVideoBpp, gVideoFlags ) == 0) {
	    /* 
	     * This could happen for a variety of reasons,
	     * including DISPLAY not being set, the specified
	     * resolution not being available, etc.
	     */
	    fprintf( stderr, "Video mode set failed: %s\n",
		     SDL_GetError( ) );
	    quit( 1 );
	}

	SDL_WM_SetCaption("Colyseus Visualizer", "Colyseus Visualizer");

#if 0
	if ( Text_Init() < 0) {
	    die("failed to init SDL_ttf");
	}
	if ( Text_SetFont("font.ttf", 80) < 0 ) {
	    die("failed to load font");
	}
#endif
    }

    setup_opengl ();
    
    load_texture (gMapfile);

    setup_network();

    /* inverse frame rate (ms) */
    int frameInterval = 1000/gFrameRate;

    uint32 pktCount = 0;

    uint32 frameCount = 0;
    TimeVal start, now, last;
    gettimeofday(&last, NULL);
    start = last;

    /*
     * Now we want to begin our normal app process--
     * an event loop with a lot of redrawing.
     */
    while( 1 ) {
	pktCount += process_network(last + frameInterval);
	gettimeofday(&now, NULL);
	last = last + frameInterval;
	if (last + frameInterval <= now)
	    last = now; // don't let time get too far off

	if (last - start >= 5000) {
	    printf("fps: %.3f  pps: %.3f  servers: %d\n", 
		   1000*((real32)frameCount)/(last-start),
		   1000*((real32)pktCount)/(last-start),
		   (int)gServers.size());
	    pktCount = 0;
	    frameCount = 0;
	    start = last;
	}
	/* Process incoming events. */
	process_events( );
	/* Draw the screen. */
	draw_screen(gWidth, gHeight);
	frameCount++;
    }

    /* Never reached. */
    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
