
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include <util/debug.h>
#include "common.h"
#include "FontUtil.h"

TTF_Font* gDefaultFont = NULL;

static int _round(double x)
{
    return (int)(x + 0.5);
}

static int nextpoweroftwo(int x)
{
    double logbase2 = log((float)x) / log((float)2);
    return _round(pow(2,ceil(logbase2)));
}

static void CloseDefaultFont()
{
    if (gDefaultFont != NULL)
	TTF_CloseFont(gDefaultFont);
}

int Text_Init(void)
{
    if(TTF_Init()) {
	fprintf(stderr, "warning: error initing SDL_ttf: %s\n", TTF_GetError());
	return -1;
    }
    atexit(TTF_Quit);
    atexit(CloseDefaultFont);
    return 0;
}

int Text_SetFont(const char *ttf_path, int size)
{
    CloseDefaultFont();

    if(!(gDefaultFont = TTF_OpenFont(ttf_path, size))) {
	fprintf(stderr, "warning: error loading font %s: %s\n", ttf_path, TTF_GetError());
	return -1;
    }
    return 0;
}

float Text_Print(const char *text,
		 const Vec3& color, 
		 const Vec3& location,
		 float height,
		 bool center)
{
    ASSERT(gDefaultFont != NULL);

    return Text_Print(text, gDefaultFont, color, location, height, center);
}
    
float Text_Print(const char *text,
		 TTF_Font *font,
		 const Vec3& color, 
		 const Vec3& location,
		 float height,
		 bool center)
{
    static SDL_Color c;

    c.r = _round(color[0]*255);
    c.g = _round(color[1]*255);
    c.b = _round(color[2]*255);

    return Text_Print(text, font, c, location[0], location[1], height, center);
}

/**
 * Derived from code at:
 *
 * http://www.gamedev.net/community/forums/topic.asp?topic_id=284259
 */
float Text_Print(const char *text, 
		 TTF_Font *font,
		 SDL_Color color,
		 float x, float y,
		 float height, bool center)
{
    SDL_Surface *initial;
    SDL_Surface *intermediary;
    SDL_Rect rect;
    int w,h;
    float realwidth, width;
    GLuint texture;
	
    /* Use SDL_TTF to render our text */
    initial = TTF_RenderText_Blended(font, text, color);
    
    /* Convert the rendered text to a known format */
    w = nextpoweroftwo(initial->w);
    h = nextpoweroftwo(initial->h);

    // xxx GL scaled fonts in textures look UGLY!
    // at some time, let's blit the buffer directly to the screen
    // (though this will require loading the correct font size...)
    realwidth = initial->w*height/h;
    width = w*height/h;

    // xxx fix endianness of masks
    intermediary = SDL_CreateRGBSurface(0, w, h, 32, 
					0x00ff0000, 0x0000ff00, 
					0x000000ff, 0xff000000);
    // unset the alpha channel so that the alpha data is copied verbatim
    SDL_SetAlpha(initial, 0, 0);
    SDL_SetAlpha(intermediary, 0, 0);

    SDL_BlitSurface(initial, 0, intermediary, 0);

    /*
    INFO << "alpha? " 
	 << (initial->flags & SDL_SRCALPHA) 
	 << " " << (intermediary->flags & SDL_SRCALPHA) << endl;
    INFO << "initial=" << initial->h << "x" << initial->w
	 << " inter=" << intermediary->h << "x" << intermediary->w << endl;

    for (int i=0; i<initial->h; i++) {
	for (int j=0; j<20; j++) {
	    printf("%02X ", 0xff & ( ((int *)initial->pixels)[i*initial->w+j] >> 24) );
	}
	printf("\n");
    }
    printf("\n");

    for (int i=0; i<intermediary->h; i++) {
	for (int j=0; j<20; j++) {
	    printf("%02X ", 0xff & ( ((int *)intermediary->pixels)[i*intermediary->w+j] >> 24) );
	}
	printf("\n");
    }
    fflush(stdout);
    exit(0);
    */
    
    /* Tell GL about our new texture */
    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexImage2D(GL_TEXTURE_2D, 0, 4, w, h, 0, GL_BGRA, 
		 GL_UNSIGNED_BYTE, intermediary->pixels );
    
    /* GL_NEAREST looks horrible, if scaled... */
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

    /* prepare to render our texture */
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, texture);
    glColor3f(1.0f, 1.0f, 1.0f);

    if (center) {
	x = x - realwidth/2;
	y = y - height/2;
    }

    /* Draw a quad at location */
    glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f);
    glVertex2f(x, y);
    glTexCoord2f(0.0f, 1.0f);
    glVertex2f(x, y + height);
    glTexCoord2f(1.0f, 1.0f);
    glVertex2f(x + width, y + height);
    glTexCoord2f(1.0f, 0.0f);
    glVertex2f(x + width, y);
    glEnd();

    /* Bad things happen if we delete the texture before it finishes */
    glFinish();

    /* Clean up */
    SDL_FreeSurface(initial);
    SDL_FreeSurface(intermediary);
    glDeleteTextures(1, &texture);

    glDisable(GL_TEXTURE_2D);

    return width;
}
