////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <util/Packet.h>
#include <om/OMMessage.h>
#include <util/Options.h>
#include <util/GPL/HuffmanPacket.h>
#include <Colyseus.h>

static char g_DumpFile [256];
static bool g_FreqCounts;
static bool g_HuffmanEnc;
static bool g_FieldStats;
static FILE *dfp = NULL;
static unsigned long counts [256];

static OptionType g_Options[] =
{
    { '#', "General params for readdumps", OPT_SEP, "", NULL, "", NULL },
    { '/', "dumpfile", OPT_STR, "file containing the MsgAppUpdate dumps", 
	g_DumpFile, "", NULL },
    { '/', "freqcounts", OPT_BOOL | OPT_NOARG, "dump frequency counts for all the message bytes", 
	&g_FreqCounts, "0", (void *) "1" },
    { '/', "huffenc", OPT_BOOL | OPT_NOARG, "perform huffman encoding and output <len, enclen>",
	&g_HuffmanEnc, "0", (void *) "1" },
    { '/', "fieldstats", OPT_BOOL | OPT_NOARG, "dump delta masks for the updates",
	&g_FieldStats, "0", (void *) "1" },
    { 0, 0, 0, 0, 0, 0, 0 }
};


int main(int argc, char **argv)
{
    InitializeColyseus (&argc, argv, g_Options, false);

    // argc = ProcessOptions(g_Options, argc, argv, true);
    // DBG_INIT(&g_LocalSID);

    if (g_DumpFile [0] == '\0') {
	Debug::die ("--dumpfile option must be provided");
    }

    dfp = fopen (g_DumpFile, "r");
    if (!dfp) 
	Debug::die ("Could not open file [%s]", g_DumpFile);

    Packet pkt (0xffff);
    if (g_FreqCounts) { 
	for (int i = 0; i < 256; i++) 
	    counts [i] = 0;
    }

    while (true) {
	int len;
	void *buf;

	pkt.ResetBufPosition ();
	if (fread (&len, sizeof (len), 1, dfp) == 0)
	    break;

	buf = pkt.GetBuffer ();
	if (fread (buf, len, 1, dfp) == 0)
	    break;
	
	MsgAppUpdate *umsg = new MsgAppUpdate (&pkt);

	if (g_HuffmanEnc) {
	    HuffmanPacket *hpkt = new HuffmanPacket (umsg->GetLength () + 100);
	    umsg->Serialize (hpkt);
	    fprintf (stdout, "%10d %10d\n", umsg->GetLength (), hpkt->GetSize ());
	    delete hpkt;
	}
	else if (g_FreqCounts) {
	    unsigned char *sbuf = (unsigned char *) buf;
	    for (int i = 0; i < len; i++) { 
		// fprintf (stderr, "i=%d sbuf[i]=%d\n", i, (unsigned int) sbuf [i]);
		counts [ (unsigned int) (sbuf[i]) ] += 1;
		if (counts [ (unsigned int) (sbuf [i]) ] >= (0x1U << 31))
		    Debug::die ("Overflow for character %d", (unsigned int) sbuf [i]);
	    }
	}
	else if (g_FieldStats) {
	    for (GUpdateMapIter it = umsg->updates.begin(); it != umsg->updates.end(); ++it) {
		GUpdate *upd = it->second;
		DeltaMask &mask = upd->mask;

		char tbuf [256];
		upd->guid.ToString (tbuf);
		
		if (!upd->isNew) {
		    cout << tbuf << "\t" << mask.GetLength () << "\t" << upd->GetLength () << "\t" << mask << endl;
		}
	    }

	    cout << "==========" << endl;
	}
	else {

	    for (GUpdateMapIter it = umsg->updates.begin(); it != umsg->updates.end(); ++it) {
		GUpdate *upd = it->second;
		char tbuf [256];
		upd->guid.ToString (tbuf);
		fprintf (stdout, "%s\t%c\t%d\n", tbuf, (upd->isNew ? 'n' : 'u'), upd->GetLength ());
	    }
	    fprintf (stdout, "=========\n");
	}
	delete umsg;
    }

    if (g_FreqCounts) { 
	fprintf (stdout, "// Frequency counts generated using MsgAppUpdate dumps...\n");
	fprintf (stdout, "//\n");
	fprintf (stdout, "static int msg_hData [256] = {\n");

	unsigned long max = counts [0];
	for (int i = 0; i < 256; i++) { 
	    if (counts [i] > max)
		max = counts [i];
	}
	for (int i = 0; i < 256; i++) {
	    fprintf (stdout, "\t%15lu,     // %3d\n", (unsigned long) ((double) counts [i] * (double) 250315 / (double) max), i);
	}
	fprintf (stdout, "};\n");
    }

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
