////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gameapi/common.h>
#include <gameapi/GameManager.h>
#include <gameapi/GameDatabase.h>
#include <gameapi/GameStore.h>
#include "SimpleGame.h"
#include "SimplePlayer.h"
#include "SimpleMissile.h"
#include "EmptyWorld.h"
#include "SimpleInterestFactory.h"
#include "SimpleWorld.h"
#include "SimpleClientHandler.h"
#include "WayPointManager.h"
#include "DonnybrookManager.h"
#include "Options.h"

static uint32 g_RandSeed;
static bool g_DisableColyseus;
static char g_MapDir[255];
static uint32 g_NumBots;
static real32 g_MapWidth;
static real32 g_MapHeight;
static real32 g_MapZAxis;
static real32 g_AoiSize;
static real32 g_WayPointParetoAlpha;
static uint32 g_RegionsPerX;
static uint32 g_RegionsPerY;
static uint32 g_SubPredictionPlayer;
static uint32 g_SubPredictionMissile;
static uint32 g_PubInterval;
static uint32 g_MaxDisLat;
static uint32 g_IdleTime;
static char   s_GameType [16];

extern char g_DonnybrookTraceFile [1024];

static OptionType g_Options[] =
{
    { '#', "General params for testgame", OPT_SEP, "", NULL, "", NULL },
    { '/', "seed", OPT_INT, 
	"random seed", 
	&g_RandSeed, "42", NULL },
    { '/', "no-colyseus", OPT_BOOL|OPT_NOARG, 
	"disable colyseus (run game locally only)", 
	&g_DisableColyseus, "0", (void *)"1" },
    { '#', "Game parameters", OPT_SEP, "", NULL, "", NULL },
    { '/', "map", OPT_STR,
	"map directory", 
	g_MapDir, "", NULL},
    { '/', "bots", OPT_INT, 
	"number of bots to start on this server", 
	&g_NumBots, "4", NULL },
    { '#', "Area of interest management", OPT_SEP, "", NULL, "", NULL },
    { '/', "aoisize", OPT_FLT, 
	"width and height of area of interest (area = aoisize^2)", 
	&g_AoiSize, "650", NULL },
    { '/', "regions-per-x", OPT_INT, 
	"how to split up the xrange into regions", 
	&g_RegionsPerX, "1", NULL },
    { '/', "regions-per-y", OPT_INT, 
	"how to split up the yrange into regions", 
	&g_RegionsPerY, "1", NULL },

    { '#', "Gameplay or workload", OPT_SEP, "", NULL, "", NULL },
    { '/', "waypoint-alpha", OPT_FLT,
	"waypoint popularity pareto parameter (Pr[X=x]=x^-a)", 
	&g_WayPointParetoAlpha, "0.1", NULL},
    { '/', "bot-wander", OPT_BOOL|OPT_NOARG, 
	"enable bot random wandering at waypoints", 
	&g_SimplePlayerWander, "0", (void *)"1" },
    { '/', "gametype", OPT_STR, 
	"gameplay mode [dm|tt|ctf]",
	&s_GameType, "dm", NULL },

    { '#', "Prediction and timers", OPT_SEP, "", NULL, "", NULL },
    { '/', "maxdislat", OPT_INT, 
	"max discovery latency for prediction fudge (msec)", 
	&g_MaxDisLat, "300", NULL },
    { '/', "pub-interval", OPT_INT, 
	"how often an object should publish (msec)", 
	&g_PubInterval, "1000", NULL },
    { '/', "sub-prediction-player", OPT_INT, 
	"how many msec to predict player subscriptions", 
	&g_SubPredictionPlayer, "5000", NULL },
    { '/', "sub-prediction-missile", OPT_INT, 
	"how many msec to predict missile subscriptions", 
	&g_SubPredictionMissile, "500", NULL },

    { '#', "Donnybrook trace generation", OPT_SEP, "", NULL, "", NULL },
    { '/', "donnybrook", OPT_BOOL | OPT_NOARG,
	"enable donnybrook trace generation [colyseus should be disabled as well]", 
	&g_EnableDonnybrook, "0", (void *) "1" },
    { '/', "donnybrook-output", OPT_STR, 
	"where the donnybrook trace should go to",
	&g_DonnybrookTraceFile, "/dev/stdout", NULL },

    { '#', "Miscellaneous", OPT_SEP, "", NULL, "", NULL },
    { '/', "idletime", OPT_INT,
	"time to idle after hitting timelimit",
	&g_IdleTime, "120", NULL },
    { 0, 0, 0, 0, 0, 0, 0 }
};

static void load_mapdims(const char *filename)
{
    char line[255];
    FILE *f = fopen(filename, "r");
    if (f == NULL) {
	Debug::die("can't open mapdims file: %s", filename);
    }

    real32 x, y, z;

    fgets(line, 255, f);
    int ret = sscanf(line, "%f %f %f", &x, &y, &z);
    ASSERT(ret == 3);

    fclose(f);

    g_MapWidth  = x;
    g_MapHeight = y;
    g_MapZAxis  = z;
}

int main(int argc, char **argv)
{
    // This function shows the the general process for initializing
    // a game implementing the Colyeus gameapi. Comments precedded by
    // *** are general to any game, while other comments are specific
    // to testgame.

    // ***
    // Construct and obtain a reference to the GameManager.
    // The GameManager is the sole interface that sits between
    // application defined classes and Colyseus.
    GameManager *m = GameManager::GetInstance(&argc, argv, g_Options);
    if (!strcmp (s_GameType, "dm"))
	g_SimpleGameOptions.gametype = GAMETYPE_DM;
    else if (!strcmp (s_GameType, "tt"))
	g_SimpleGameOptions.gametype = GAMETYPE_TWOTEAM;
    else if (!strcmp (s_GameType, "ctf"))
	g_SimpleGameOptions.gametype = GAMETYPE_CTF;
    else 
	Debug::die ("Invalid game type specified [%s], valid values are [dm|tt|ctf]", s_GameType);

    // Initialize debugging
    DBG_INIT(&g_LocalSID);

    // Set game specific parameters. These are specific to testgame
    srand(g_RandSeed);
    srand48(g_RandSeed);
    SimpleGame::SetMaxDisLat(g_MaxDisLat);
    SimpleGame::SetPubInterval(g_PubInterval);
    SimplePlayer::SetSubPrediction(g_SubPredictionPlayer);
    SimpleMissile::SetSubPrediction(g_SubPredictionMissile);

    // Get parameters for the world map. This specific to testgame
    if (!strcmp(g_MapDir, "")) {
	Debug::die("testgame requires the --map option");
    }

    load_mapdims( (string(g_MapDir) + "/dims").c_str() );

    BBox extent( Vec3(0,0,-g_MapZAxis/2),
	    Vec3(g_MapWidth,g_MapHeight,g_MapZAxis/2) );
    BBox aoi( Vec3(-g_AoiSize/2,-g_AoiSize/2,-20), 
	    Vec3(g_AoiSize/2,g_AoiSize/2,20) );

    // ***
    // Initialize a GameModule: this defined what happens each
    // game frame and during initialization and shutdown. This can
    // be thought of as equivalent to the "Virtual machine" or "DLL"
    // interface that Quake and other FPSes export to MODs.
    SimpleGame          *g = new SimpleGame();

    // ***
    // Initialize a GameClientHandler: this defines how data is sent to
    // game clients. It contains callbacks that are called before, after,
    // and between game frames (which is defined in the GameModule)
    SimpleClientHandler *c = new SimpleClientHandler();

    // ***
    // Initialize a GameWorld: this defines the "world" in which the
    // game is played. The game world class is assumed to be static
    // and loading on game world content occurs outside the scope of
    // Colyseus (i.e., is handled by the application). When new game
    // objects are created or replicated they are "linked" and "unlinked"
    // from the game world when they enter and leave it, respectively
    SimpleWorld         *w = new SimpleWorld(extent, aoi, 
	    g_RegionsPerX,
	    g_RegionsPerY);

    // ***
    // Initialize a GameInterestFactory: this optional application class
    // defines how object interests (i.e., subscriptions) are filtered
    // and combined during each frame.
    SimpleInterestFactory *f = new SimpleInterestFactory();

    // Load waypoints for this testgame map. This is specific to testgame
    FileWayPointFactory wpf( (string(g_MapDir) + "/waypoints").c_str() );
    w->Init(&wpf, g_WayPointParetoAlpha);

    INFO << "world=" << w->GetExtent() << endl;

    // ***
    // Initialize the GameManager with the application defined classes:
    // GameModule, GameWorld, GameClientHandler, GameInterestFactory
    m->Init(g, w, c, f, NULL, g_DisableColyseus);

    // ***
    // Get a reference to the object store which contains all game objects
    // from the GameManager.
    GameStore *s = m->GetStore();

    if (g_EnableDonnybrook)
	g_DonnybrookManager = new DonnybrookManager (m, g_NumBots);

    // Add initial testgame objects (bots) to the game store (they will
    // automatically be linked to the GameWorld)
    for (uint32 i=0; i<g_NumBots; i++) {
	TDB (-1) << " initing bot " << i << endl;

	SimplePlayer *o1 = new SimplePlayer(m, i % 2 /* team */);

	Vec3 start = w->GetRandomPoint();
	start[2] = 0;
	o1->SetOrigin(start);

	if (g_EnableDonnybrook) 
	    g_DonnybrookManager->RecordObjectCreate (o1);

	s->Add(o1);
    }

    // Print some benchmark statistics
    Benchmark::print ();
    Benchmark::restart ();

    // ***
    // Start the game. The GameManager will run each frame, appropriately
    // calling back the GameModule and GameClientHandler, as well as other
    // application defined classes initialized above
    m->Run();

    if (g_EnableDonnybrook)
	delete g_DonnybrookManager; // generates events at the end of the run

    // DEBUGGING: after the game exits, wait for a while (for experiments
    // to complete)
    cerr << "* idling for " << g_IdleTime << " seconds " << endl;
    m->Idle(g_IdleTime * 1000);
    cerr << "exiting gracefully" << endl;
    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
