////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  SpaceWorld.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#ifndef __SPACE_WORLD_H__
#define __SPACE_WORLD_H__

#include <mercury/common.h>
#include <gameapi/GameWorld.h>
#include <gameapi/GameObject.h>

// content masks
#define TRACE_ALL         0xFFFFFFFF
#define TRACE_OBJECT      1 
#define TRACE_WORLD_SOLID 2
#define TRACE_WORLD_EDGE  4 

struct TraceResult
{
    bool hit;           // hit something?
    bool startSolid;    // started in a solid?
    real32 fraction;    // fraction of trace before hit
    uint32 contentMask; // content type of thing hit
    GameObject *hitObj; // object we hit (or NULL)
    Vec3 hitPos;        // point at where we hit
    //Vec3 hitNorm;       // normal to plane at point where we hit XXX TODO
};

struct TraceFilterFunc {
    virtual ~TraceFilterFunc () {}
    /**
     * Ignore object if this returns true.
     */
    virtual bool operator()(GameObject *obj) const { return true; };
};

struct TraceObjFilterFunc : public TraceFilterFunc {
    GameObject *obj;
    TraceObjFilterFunc(GameObject *obj) : obj(obj) {}
    virtual bool operator()(GameObject *other) const {
	return obj == other;
    }
};

///////////////////////////////////////////////////////////////////////////////

class SpaceWorld : public GameWorld {
public:
    virtual BBox Clip(const BBox& box) {
	BBox ret;
	for (int i=0; i<3; i++) {
	    ret.min[i] = MAX(GetExtent().min[i], box.min[i]);
	    ret.max[i] = MIN(GetExtent().max[i], box.max[i]);
	}
	return ret;
    }
    virtual Vec3 Clip(const Vec3& clip) {
	Vec3 ret;
	for (int i=0; i<3; i++) {
	    ret[i] = MAX(GetExtent().min[i], clip[i]);
	    ret[i] = MIN(GetExtent().max[i], clip[i]);
	}
	return ret;
    }

    virtual void Trace(TraceResult& res, 
	    const BBox& ext, const Vec3& start, const Vec3& end,
	    const TraceFilterFunc& ignoreFunc, 
	    uint32 contentMask = TRACE_ALL) = 0;
    virtual void Trace(TraceResult& res, 
	    const BBox& ext, const Vec3& start, const Vec3& end,
	    GameObject *ignore = NULL, 
	    uint32 contentMask = TRACE_ALL) {
	Trace(res, ext, start, end, TraceObjFilterFunc(ignore), contentMask);
    }

    virtual Vec3 GetRandomPoint() = 0;
    virtual void GetVisible(list<GameObject *> *vis, 
	    DynamicGameObject *obj,
	    const TraceFilterFunc& ignoreFunc) = 0;
    virtual void GetVisible(list<GameObject *> *vis, 
	    DynamicGameObject *obj,
	    GameObject *ignore = NULL) {
	GetVisible(vis, obj, TraceObjFilterFunc(ignore));
    }
    virtual bool IsVisible(DynamicGameObject *from, const Vec3& to,
	    list<BBox> *from_aoi = NULL) = 0;
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
