////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#ifndef __SIMPLE_PLAYER_H__
#define __SIMPLE_PLAYER_H__

#include "SimpleGame.h"
#include "SimpleMovable.h"
#include "WayPointManager.h"

typedef enum { SP_FIGHTING_AT_WAYPOINT, SP_FIGHTING_TO_WAYPOINT,
    SP_AT_WAYPOINT, SP_TO_WAYPOINT } SimplePlayerState;

typedef enum { CTF_NOCOMMAND, CTF_GET_FLAG, CTF_RETURN_TO_BASE } CtfCommandType;

typedef enum { FLAG_NONE, FLAG_RED, FLAG_BLUE } FlagType;

extern bool g_SimplePlayerWander;

class SimpleWorld;

class SimplePlayer : public SimpleMovable {
private:
    uint16 health;
    uint16 ammo;
    uint32 frags;

    SimplePlayerState state;
    FlagType flag;
    CtfCommandType ctf_state;
    uint16 nextAttack;

    const WayPoint *longTermGoal;
    const WayPoint *goal;
    Vec3  wanderGoal;

    // local variables (not distributed)
    gRef<SimplePlayer> target;    // recalc when looking for enemy
    list<const WayPoint *> path;  // recalc when missing
    list<GameObjectRef> visible;  // recalc each frame

    uint32 visiblePlayers, visibleMissiles;
    bool hitFromBehind;  
    uint16 team;

    // local estimate of client cost (Bps)
    CostMetric clientOutbound;

    void FindVisible(SimpleWorld *w);
    void FindEnemy(SimpleWorld *w);
    void PickWanderGoal(SimpleWorld *w);
    bool PickLongTermGoal(SimpleWorld *w);
    void HandleFightingTransition(SimpleWorld *w, real32 delta);
    void HandleAtWaypointTransition(SimpleWorld *w, real32 delta);
    void HandleToWaypointTransition(SimpleWorld *w, real32 delta);
    void HandleFightingThink(SimpleWorld *w, real32 delta);
    void HandleAtWaypointThink(SimpleWorld *w, real32 delta);
    bool CTFThink (SimpleWorld *w, real32 delta);
    void HandleToWaypointThink(SimpleWorld *w, real32 delta);
    void EvasiveAction();
    void PerformPhysics(SimpleWorld *w, real32 delta);

    bool SetGoalAndPath (SimpleWorld *w, const WayPoint *g);

protected:

    void SetDefaults();

public:

    // based on q2/q3 size (q3 is 30x30)
    static const real32 SIZE   = 32;
    static const real32 HEIGHT = 2;  // doesn't matter, we have no zaxis
    static const real32 FOV    = 60; // degrees around viewangles that is "in view"  (FOV/2 on each side)

    // based on q3 max walk velocity
    static const uint32 MAX_HEALTH = 1000;
    static const uint32 MAX_AMMO = 1000;
    // (100+200)/10 (suppose 1/2 time walking, 1/2 time running, no accel)
    static const real32 WALK_VELOCITY = 15.0;

    // these probs approximately estimated from q3dm7 and q3dm14 games
    static const real32 PROB_RUN_AWAY = 0.12/5;
    static const real32 PROB_FIGHT = 0.14/5;
    static const real32 PROB_LEAVE = 0.0125;

    // if this far away from a target, try to move closer
    static const real32 FIGHT_FOLLOW_RADIUS = 650/2; // about 1/2 sub-size
    static const real32 WANDER_GOAL_DIST = 650/2; // about 1/2 sub-size
    static const real32 WANDER_GOAL_RADIUS = 10;

    static uint32 SUB_PREDICTION;

    static void SetSubPrediction(uint32 msec) {
	ASSERT(!SimpleGame::IsInited());
	SUB_PREDICTION = msec;
    }

    static const char *StateToString(SimplePlayerState s);

    SimplePlayer(GameManager *manager, uint16 team);
    SimplePlayer(GObjectInfoIface *info);

    virtual ~SimplePlayer();

    uint16 GetHealth();
    void SetHealth(uint16 v);
    uint16 GetAmmo();
    void SetAmmo(uint16 v);

    uint32 GetFrags();
    void SetFrags(uint32 v);
    void IncFrags();

    SimplePlayerState GetState();
    void SetState(SimplePlayerState s);

    uint16 GetNextAttack();
    void SetNextAttack(uint16 frames);

    const WayPoint *GetLongTermGoal();
    void SetLongTermGoal(const WayPoint *goal);
    const WayPoint *GetGoal();
    void SetGoal(const WayPoint *goal);

    const Vec3& GetWanderGoal();
    void SetWanderGoal(const Vec3& goal);

    uint16 GetTeam ();
    void SetTeam (uint16 t);

    virtual const uint32 GetDeltaMaskBits();

    virtual uint32 GetAreaOfInterest(list<BBox> *toFill, GameWorld *w);
    virtual uint32 IsInterested(const Vec3& pt, gTypeCode t);
    virtual uint32 IsInterested(const BBox& vol, gTypeCode t);
    virtual void RunThink(GameManager *manager, real32 delta);

    virtual const DeltaMask& GetInitDeltaMask();
    virtual void PackUpdate(Packet *buffer, const DeltaMask& mask);
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
	    list<GameObjectRef *> *unresolved);

    virtual void PrintFields(ostream& out);

    CostMetric GetClientOutbound() const { return clientOutbound; }
    void SetClientOutbound(CostMetric c) { clientOutbound = c; }
    list<GameObjectRef> *GetVisibleObjects() { return &visible; }
    virtual CostMetric GetFixedCost() { return GetClientOutbound(); }

    virtual VisType GetVisualizerType () { return VisType_Bot; }

    ///////////////////////////////////////////////////////////////////////////

    void DoDamage(GameObject *source, 
	    DynamicGameObject *inflictor, 
	    uint32 dmg);
    void Suicide();
    void Respawn(GameWorld *w);
    void Attack(SimplePlayer *targ);

    bool IsCarryingFlag () {
	return flag != FLAG_NONE;
    }

private:
    bool inFieldOfView (GameObject *obj);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
