////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <gameapi/GameManager.h>
#include "SimpleTypeCodes.h"
#include "EmptyWorld.h"
#include "SimplePlayer.h"
#include "SimpleMissile.h"
#include "SimpleGame.h"

    const real32 SimpleMissile::FLY_VELOCITY ;
const uint16 SimpleMissile::LIFETIME ;
const uint32 SimpleMissile::DAMAGE ;
uint32 SimpleMissile::SUB_PREDICTION = 500;

static bool inited = false;
static DeltaMask initDeltaMask;
static const BBox vol(Vec3(0,0,0), Vec3(0,0,0));

SimpleMissile::SimpleMissile(GameManager *manager) :
    SimpleMovable(manager, SIMPLE_MISSILE) 
{
    SetLifetime(LIFETIME);
    RegisterRef(&owner);
}

SimpleMissile::SimpleMissile(GObjectInfoIface *info) :
    SimpleMovable(info, SIMPLE_MISSILE)
{
    SetVolume(vol);
    RegisterRef(&owner);
}

SimpleMissile::~SimpleMissile()
{
    SetVolume(vol);
}

uint32 SimpleMissile::GetAreaOfInterest(list<BBox> *toFill, GameWorld *w) {
    Vec3 orig = GetOrigin();
    // Suppose missiles just need to see 5 frames in front of them
    Vec3 end  = orig + GetVelocity()*5;

    BBox ret;
    for (int i=0; i<3; i++) {
	ret.min[i] = MIN(orig[i], end[i]);
	ret.max[i] = MAX(orig[i], end[i]);
    }
    toFill->push_back( w->Clip(ret) );

    uint32 retpred = SUB_PREDICTION;
#if 0
    if (g_Preferences.slowdown_factor > 1.0f) 
	retpred = (uint32) (retpred * g_Preferences.slowdown_factor);
#endif
    return retpred;
}

uint32 SimpleMissile::IsInterested(const Vec3& pt, gTypeCode t)
{
    list<BBox> aoi;

    GetPredictedAreaOfInterest(&aoi, GameManager::GetInstance()->GetWorld());

    for (list<BBox>::iterator it = aoi.begin(); it != aoi.end(); it++) {
	if ((*it).Contains(pt)) {
	    switch (t) {
	    case SIMPLE_PLAYER:
		return MIN(3 * SimpleGame::frameInterval, 
			   SimpleGame::frameInterval * lifetime);
	    default:
		// nothing
		break;
	    }
	}
    }

    return 0;
}

uint32 SimpleMissile::IsInterested(const BBox& vol, gTypeCode t)
{
    list<BBox> aoi;

    GetPredictedAreaOfInterest(&aoi, GameManager::GetInstance()->GetWorld());

    for (list<BBox>::iterator it = aoi.begin(); it != aoi.end(); it++) {
	if ((*it).Overlaps(vol)) {
	    switch (t) {
	    case SIMPLE_PLAYER:
		return MIN(5 * SimpleGame::frameInterval, 
			   SimpleGame::frameInterval * lifetime);
	    default:
		// nothing
		break;
	    }
	}
    }

    return 0;
}

void SimpleMissile::RunThink(GameManager *manager, real32 delta)
{
    ASSERT(GetVelocity().Length() > 0);

    Vec3 old_origin = GetOrigin();

    Vec3 new_origin = GetOrigin() + delta*GetVelocity();
    EmptyWorld *w = static_cast<EmptyWorld *>( manager->GetWorld() );

    Vec3 hit_pt;
    GameObject *hit_obj = NULL;

    bool h = w->Fly(GetVolume(), GetOrigin(), new_origin, 
		    hit_pt, hit_obj, this);

    if (h) {
	new_origin = (hit_pt - 0.01*GetVelocity().Normalize());
	if (hit_obj) {
	    //DB(1) << "Mis: " << this << endl;
	    if (hit_obj->GetType() == SIMPLE_PLAYER) {
		SimplePlayer *p = dynamic_cast<SimplePlayer *>(hit_obj);
		p->DoDamage( owner, this, DAMAGE );
		//DB(1) << "Hit: " << p << endl;
	    } else {
		//DB(1) << "Hit something else" << endl;
	    }
	}
	MarkForDeletion();
    } else {
	ASSERT(new_origin != old_origin);
    }

    if (w->GetExtent().Contains(new_origin))
	SetOrigin(new_origin);

    SetLifetime( GetLifetime() - 1 );
    if ( GetLifetime() <= 0) {
	//DB(1) << "Mis: " << this << endl;
	//DB(1) << "Died!" << endl;
	MarkForDeletion();
    }
}

const DeltaMask& SimpleMissile::GetInitDeltaMask()
{
    if (!inited) {
	initDeltaMask = SimpleMovable::GetInitDeltaMask();
	initDeltaMask.Set( SimpleMovable::GetDeltaMaskBits() );
	initDeltaMask.Set( SimpleMovable::GetDeltaMaskBits() + 1 );
	inited = true;
    }

    return initDeltaMask;
}

void SimpleMissile::PackUpdate(Packet *buffer, const DeltaMask& mask)
{
    SimpleMovable::PackUpdate(buffer, mask);
    if (mask.IsSet( SimpleMovable::GetDeltaMaskBits() )) {
	buffer->WriteShort( lifetime );
    }
    if (mask.IsSet( SimpleMovable::GetDeltaMaskBits() + 1)) {
	owner.Serialize(buffer);
    }
}

void SimpleMissile::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				 list<GameObjectRef *> *unresolved)
{
    SimpleMovable::UnpackUpdate(buffer, mask, unresolved);
    if (mask.IsSet( SimpleMovable::GetDeltaMaskBits() )) {
	lifetime = buffer->ReadShort();
    }
    if (mask.IsSet( SimpleMovable::GetDeltaMaskBits() + 1)) {
	owner = gRef<SimplePlayer>(buffer);
	if (!owner.IsResolved()) {
	    unresolved->push_back(&owner);
	}
    }
}

uint32 SimpleMissile::ExpectedLifetime()
{
    return SimpleGame::frameInterval * lifetime;
}

void SimpleMissile::PrintFields(ostream& out)
{
    SimpleMovable::PrintFields(out);
    out << " lifetime=" << lifetime;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
