////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <gameapi/GameManager.h>
#include <gameapi/GameStore.h>
#include <gameapi/GameLogs.h>
#include "SimpleClientHandler.h"
#include "SimplePlayer.h"
#include "SimpleTypeCodes.h"

#define MAX_BUF_SIZE 0xFFFF // XXX

const float SimpleClientHandler::EMA_ALPHA;

gRetCode SimpleClientHandler::RunPostFrame(GameManager *manager)
{
    static Packet buf(MAX_BUF_SIZE); 

    GameStore *s = manager->GetStore();

    DynamicGameObject *o;
    s->BeginDynamic();
    while ((o = s->NextDynamic()) != NULL) {
	if (o->IsReplica() || o->GetType() != SIMPLE_PLAYER)
	    continue;
	SimplePlayer *p = dynamic_cast<SimplePlayer *>(o);

	ClientMap::iterator ref = m_Clients.find(p->GetGUID());
	SimpleClientInfo *i;
	if (ref == m_Clients.end()) {
	    i = new SimpleClientInfo;
	    m_Clients.insert(pair<GUID,SimpleClientInfo *>(p->GetGUID(), i));
	} else {
	    i = ref->second;
	}
	// XXX handle deletions on client "disconnect" (don't have those yet :)

	// serialize this player's own object to the client (+ delta mask)
	buf.ResetBufPosition();
	DeltaMask mask = p->GetDeltaMask();
	mask.Serialize(&buf);
	p->PackUpdate(&buf, mask);

	// for each non-static object that this player can see,
	// serialize it to that client (+ the delta mask)
	list<GameObjectRef> *vis = p->GetVisibleObjects();
	GUIDSet seen;
	for (list<GameObjectRef>::iterator it = vis->begin(); 
	     it != vis->end(); it++) {
	    if ((GameObject *)(*it) != NULL && (*it)->IsDynamic()) {
		DynamicGameObject *o2 = 
		    dynamic_cast<DynamicGameObject *>((GameObject *)*it);
		DeltaMask mask2;
		if (i->known.find(o2->GetGUID()) == i->known.end()) {
		    // encode from baseline
		    mask2 = o2->GetInitDeltaMask();
		} else {
		    // encode from last update -- for now assume no packet
		    // loss and we always encode with reference to last frame
		    //
		    // updates from this server
		    mask2 = o2->GetDeltaMask();
		    // updates from remote servers
		    mask2.Merge( o2->GetRemoteDeltaMask() );
		    i->known.insert(o2->GetGUID());
		}
		mask2.Serialize(&buf);
		o2->PackUpdate(&buf, mask2);

		if (buf.GetBufPosition () > MAX_BUF_SIZE-512) {
		    WARN << "client outbound buffer got way too big: " 
			 << buf.GetBufPosition () << endl;
		    break;
		}

		seen.insert(o2->GetGUID());
	    }
	}

	// Eliminate objects no longer in our known set
	for (GUIDSetIter it = i->known.begin(); it != i->known.end(); ) {
	    GUIDSetIter oit = it;
	    oit++;
	    if (seen.find(*it) == seen.end())
		i->known.erase(it);
	    it = oit;
	}

	uint32 proto  = 28 /* udp/ip */;
	uint32 header = 4 /* framenum */ + 2 /* num updates */ + 2 /* seqno */;

	// estimate the update size as update + header + protocol overhead
	// in reality we probably also need to add some more header info
	// but that is probably minimal. we're also missing deletion info.
	CostMetric update = proto + header + buf.GetBufPosition ();

	// reestimate the client outbound bandwidth as EMA
	CostMetric curr = p->GetClientOutbound();
	CostMetric next = EMA_ALPHA*curr + (1-EMA_ALPHA)*update;
	p->SetClientOutbound(next);

	if (g_MeasurementParams.enabled) {
	    ClientLogEntry e;
	    e.frameno = (uint16)manager->GetFrameNumber();
	    e.guid = p->GetGUID();
	    e.update = update;
	    e.estimate = next;

	    LOG(ClientLog, e);
	}
    }

    // now we need to clear the remote delta masks because we are finished
    // processing them (so those updates don't show up in the next frame)
    s->BeginDynamic();
    while ((o = s->NextDynamic()) != NULL) {
	if (o->IsDynamic()) {
	    DynamicGameObject *d = dynamic_cast<DynamicGameObject *>(o);
	    d->ClearRemoteDeltaMask();
	}
    }

    return GAME_OK;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
