////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __PREDICTIVE_INTEREST_FILTER_H__
#define __PREDICTIVE_INTEREST_FILTER_H__

#include <gameapi/GameInterestFactory.h>
#include <gameapi/GameObject.h>

    // these are generally per-game parameters
    struct PredictiveParams
    {
	// max velocity of any object in the game (as distance/unit_time)
	real32 maxVelocity;
	// the min "radius" (width/2) of location publications
	// XXX this is coupled with maxVelocity... should be per-object...
	real32 locRadius;
	// the min location publication interval of any object in the game
	// XXX this is coupled with maxVelocity and locRadius
	uint32 pubInterval;
	// per unit time (msec)
	uint32 timeUnit;
	// "worst case" discovery time (higher = larger subs, but less
	// inconsistency) (msec)
	// XXX should estimate this directly based on history...
	uint32 maxDiscoveryLat;

	PredictiveParams() {}

	PredictiveParams(real32 maxVelocity,
			 real32 locRadius,
			 uint32 pubInterval,
			 uint32 timeUnit,
			 uint32 maxDiscoveryLat) :
	    maxVelocity(maxVelocity), locRadius(locRadius), 
	     pubInterval(pubInterval), timeUnit(timeUnit), 
	     maxDiscoveryLat(maxDiscoveryLat) {}
    };

/**
 * This is an interest filter that "predicts" the described interest for two
 * dual reasons:
 *
 * (1) We don't want to have to re-subscribe to the world every frame, so
 *     predict what we will likely subscribe to in the next few frames.
 *     (controlled by the "predict" parameter)
 *
 * (2) We want to account for discovery-time (i.e., merc matching) fudge
 *     time so we don't miss stuff that is published near the peripherary
 *     of our interest.
 *
 * This filter assumes that the input is the un-modified area of interest
 * output directly by the GameObject. That is, a non-predictive version of
 * the interest.
 *
 * XXX: Currently this filter only works for "free-space" type worlds, since
 * it just expands the bounding box. It will not work for irregular worlds...
 *
 * XXX: Currently this filter only works if there is only one bbox in the
 * area of interest.
 */
class PredictiveInterestFilter : public InterestFilter {
 private:
    const PredictiveParams *m_Params;
    uint32 m_Predict;
    real32 m_ObjVelocity;
    GUID   m_GUID;
    // not safe to keep this -- it might be nullified!
    //DynamicGameObject *m_Obj;

    BBox    m_LastPredicted;
    uint32  m_LastTTL;
    TimeVal m_LastTime;

    // exponentially weighted moving average of our speed
    // moving in each of 4 primary directions
    real32  m_HistUp, m_HistDown, m_HistLeft, m_HistRight;
 public:

    // Dunno if this is just a hack or not, but I'm using two different
    // alphas depending on whether the update decreases or increases the
    // value. If it increases the value, then pay lots of attention to the
    // new update, but not so much if it decreases. The intuition behind
    // this is that we want to overestimate where we might be if we are not
    // sure, because otherwise we will end up resubscribing a lot
    static const real32 EMA_INC_ALPHA = 0.8;  // 1.x frames until full dec
    static const real32 EMA_DEC_ALPHA = 0.02; // 5 seconds until full dec

    /**
     * @param obj the object this filter is predicting for
     * @param objVelocity the max (or expected max) velocity of this object
     * @param predict how far into the future to predict the interest
     * @param params the game specific prediction parameters.
     */
    PredictiveInterestFilter(DynamicGameObject *obj,
			     real32 objVelocity,
			     uint32 predict, 
			     const PredictiveParams *params);
    virtual ~PredictiveInterestFilter();

    AreaOfInterest *Filter(AreaOfInterest *in);
    void Delete(AreaOfInterest *aoi);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
