////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <gameapi/GameObject.h>
#include <gameapi/GameManager.h>
#include <gameapi/GameStore.h>
#include "PredictiveInterestFilter.h"
#include "SimpleMovable.h"

    const real32 PredictiveInterestFilter::EMA_INC_ALPHA;  // 1.x frames until full dec
const real32 PredictiveInterestFilter::EMA_DEC_ALPHA; // 5 seconds until full dec

PredictiveInterestFilter::PredictiveInterestFilter(DynamicGameObject *obj,
						   real32 objVelocity,
						   uint32 predict, 
						   const PredictiveParams *params) :
    m_GUID(obj->GetGUID()), m_ObjVelocity(objVelocity), m_Predict(predict), 
    m_LastTTL(0), m_LastTime(TIME_NONE), m_Params(params),
    // initially no information, so equally likely to be moving
    // in any direction at maximum velocity!
    m_HistUp(objVelocity), m_HistDown(objVelocity),
    m_HistLeft(objVelocity), m_HistRight(objVelocity)
{
    ASSERT(obj);
    ASSERT(params);
}

PredictiveInterestFilter::~PredictiveInterestFilter()
{
}

AreaOfInterest *PredictiveInterestFilter::Filter(AreaOfInterest *in)
{
    // XXX handle multiple boxes in the future
    ASSERT(in->size() <= 1);

    //DB(1) << "obj: " << m_Obj << endl;

    // XXX handle other object types?
    GameObject *obj = GameManager::GetInstance()->GetStore()->Find(m_GUID);
    ASSERT( obj != NULL );
    SimpleMovable *mobj = 
	dynamic_cast<SimpleMovable *>(obj);
    if (!mobj) {
	WARN << "can't handle other object types!" << endl;
	ASSERT(0);
    }

    //
    // Zero step. Update estimates of where we expect to be moving
    // in the next few seconds.
    //
    real32 xupdate = mobj->GetVelocity()[0];
    real32 yupdate = mobj->GetVelocity()[1];

    switch (mobj->GetType()) {
    case SIMPLE_MISSILE: {
	// for missiles, movement is deterministic, we know exactly where
	// we will be moving
	if (yupdate > 0) {
	    m_HistUp = yupdate;
	    m_HistDown = 0;
	} else {
	    m_HistUp = 0;
	    m_HistDown = yupdate;
	}
	if (xupdate > 0) {
	    m_HistRight = xupdate;
	    m_HistLeft = 0;
	} else {
	    m_HistRight = 0;
	    m_HistLeft = xupdate;
	}
	break;
    }
    case SIMPLE_PLAYER: {
	if (yupdate > m_HistUp) {
	    m_HistUp = EMA_INC_ALPHA*yupdate + (1-EMA_INC_ALPHA)*m_HistUp;
	} else {
	    m_HistUp = EMA_DEC_ALPHA*yupdate + (1-EMA_DEC_ALPHA)*m_HistUp;
	}
	if (yupdate < m_HistDown) {
	    m_HistDown = EMA_INC_ALPHA*yupdate + (1-EMA_INC_ALPHA)*m_HistDown;
	} else {
	    m_HistDown = EMA_DEC_ALPHA*yupdate + (1-EMA_DEC_ALPHA)*m_HistDown;
	}

	if (xupdate > m_HistRight) {
	    m_HistRight = EMA_INC_ALPHA*xupdate + (1-EMA_INC_ALPHA)*m_HistRight;
	} else {
	    m_HistRight = EMA_DEC_ALPHA*xupdate + (1-EMA_DEC_ALPHA)*m_HistRight;
	}
	if (xupdate < m_HistLeft) {
	    m_HistLeft = EMA_INC_ALPHA*xupdate + (1-EMA_INC_ALPHA)*m_HistLeft;
	} else {
	    m_HistLeft = EMA_DEC_ALPHA*xupdate + (1-EMA_DEC_ALPHA)*m_HistLeft;
	}
	break;
    }
    default:
	ASSERT(0);
    }

    AreaOfInterest *out = new AreaOfInterest();
    if (in->size() == 0)
	return out;

    AnnotatedBBox *ibox = in->front();
    BBox *ibboxp = ibox->GetBBox();

    if (m_LastTime + m_LastTTL > GameManager::GetInstance()->ThisFrameTime() &&
	m_LastPredicted.Contains(*ibboxp)) {
	// new sub inside old pred-sub
	return out;
    } else {
	m_LastTime = GameManager::GetInstance()->ThisFrameTime();
	m_LastTTL  = m_Predict;
    }

    //DB(1) << "in: " << *ibox->GetGUIDs()->begin() << endl;
    //DB(1) << "in: " << *ibboxp << endl;

    //
    // First predict how far this guy will move in m_Predict time
    //

    //Vec3 predict_vec = mobj->GetVelocity()*( (real32)m_Predict/m_Params->timeUnit );
    Vec3 predict_vec_min = Vec3(m_HistLeft, m_HistDown, 0) * ( (real32)m_Predict/m_Params->timeUnit );
    Vec3 predict_vec_max = Vec3(m_HistRight, m_HistUp, 0) * ( (real32)m_Predict/m_Params->timeUnit );

    for (int i=0; i<3; i++) {
	m_LastPredicted.min[i] = 
	    MIN(ibboxp->min[i], ibboxp->min[i] + predict_vec_min[i]);
	m_LastPredicted.max[i] = 
	    MAX(ibboxp->max[i], ibboxp->max[i] + predict_vec_max[i]);
    }
    // these are max bounds
    //real32 predict_ext = m_Predict*m_ObjVelocity/m_Params->timeUnit;

    GameWorld *w = GameManager::GetInstance()->GetWorld();

    //
    // Then add on fudge units on each side of the bbox to account for
    // object discovery delay.
    //

    real32 fudge_ext = MAX( 0,
			    // the amount of "fudge" we have to add is the
			    // velocity of this object + the velocity of any
			    // other object moving toward one another
			    // times the amount of time that might elapse
			    // between an object changing position and us
			    // being notified of that change (= inverse of
			    // the rate of publications + how long it takes
			    // the publication to get delivered to us)
			    // - how far the publications themselves are
			    // predicted (since they themselves are fudged)
			    (m_Params->maxVelocity/*+m_ObjVelocity*/)*
			    (m_Params->pubInterval+m_Params->maxDiscoveryLat)/
			    m_Params->timeUnit - m_Params->locRadius );

    AnnotatedBBox *obox = 
	new AnnotatedBBox( w->Clip( BBox(m_LastPredicted.min - fudge_ext,
					 m_LastPredicted.max + fudge_ext) ),
			   ibox->GetGUIDs(),
			   m_Predict );
    out->push_back(obox);

    //DB(1) << "pred_vec: " << predict_vec << endl;
    //DB(1) << "fudge_ext: " << fudge_ext << endl;

    //DB(1) << "ot: " << *obox->GetGUIDs()->begin() << endl;
    //DB(1) << "ot: " << *obox->GetBBox() << endl;

    return out;
}

void PredictiveInterestFilter::Delete(AreaOfInterest *aoi)
{
    for (AreaOfInterestIter it = aoi->begin(); it != aoi->end(); it++) {
	delete *it;
    }
    delete aoi;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
