////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  EmptyWorld.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

 ***************************************************************************/

#include "EmptyWorld.h"

///////////////////////////////////////////////////////////////////////////////

// From: http://www.gamasutra.com/features/19991018/Gomez_3.htm

// Alternate representation of BBox (mid + ext instead of min + max)
class AABB
{
public:

    Vec3 P; //position
    Vec3 E; //x,y,z extents

    AABB( const Vec3& p,
	    const Vec3& e ): P(p), E(e)
    {}

    AABB( const Vec3& pos, const BBox& bbox ) :
	P( pos ),
	E( (bbox.max[0] - bbox.min[0])/2,
		(bbox.max[1] - bbox.min[1])/2,
		(bbox.max[2] - bbox.min[2])/2 )
    {}

    //returns true if this is overlapping b
    const bool overlaps( const AABB& b ) const
    {
	const Vec3 T = b.P - P; // vector from A to B
	return fabs(T[0]) <= (E[0] + b.E[0])
	    &&
	    fabs(T[1]) <= (E[1] + b.E[1])
	    &&
	    fabs(T[2]) <= (E[2] + b.E[2]);
    }

    //NOTE: since the vector indexing operator is not const,
    //we must cast away the const of the this pointer in the
    //following min() and max() functions
    //min x, y, or z

    const VEC_TYPE min( uint32 i ) const
    {
	return ((AABB*)this)->P[i] - ((AABB*)this)->E[i];
    }

    //max x, y, or z
    const VEC_TYPE max( uint32 i ) const
    {
	return ((AABB*)this)->P[i] + ((AABB*)this)->E[i];
    }

};

///////////////////////////////////////////////////////////////////////////////

//Sweep two AABB's to see if and when they first
//and last were overlapping
const bool AABBSweep
(
 const Vec3& Ea, //extents of AABB A
 const Vec3& A0, //its previous position
 const Vec3& A1, //its current position
 const Vec3& Eb, //extents of AABB B
 const Vec3& B0, //its previous position
 const Vec3& B1, //its current position
 real32& u0, //normalized time of first collision
 real32& u1 //normalized time of second collision
 )
{
    const AABB A( A0, Ea );//previous state of AABB A
    const AABB B( B0, Eb );//previous state of AABB B
    const Vec3 va = A1 - A0;//displacement of A
    const Vec3 vb = B1 - B0;//displacement of B
    //the problem is solved in A's frame of reference
    Vec3 v = vb - va;
    //relative velocity (in normalized time)
    Vec3 u_0(0,0,0);
    //first times of overlap along each axis
    Vec3 u_1(1,1,1);
    //last times of overlap along each axis

    //check if they were overlapping
    // on the previous frame
    if( A.overlaps(B) )
    {
	DB(0) << "overlaps!" << endl;
	u0 = u1 = 0;
	return true;
    }

    //find the possible first and last times
    //of overlap along each axis
    for( long i=0 ; i<3 ; i++ )
    {
	if( A.max(i)<B.min(i) && v[i]<0 )
	{
	    u_0[i] = (A.max(i) - B.min(i)) / v[i];
	}
	else if( B.max(i)<A.min(i) && v[i]>0 )
	{
	    u_0[i] = (A.min(i) - B.max(i)) / v[i];
	}
	if( B.max(i)>A.min(i) && v[i]<0 )
	{
	    u_1[i] = (A.min(i) - B.max(i)) / v[i]; 
	}
	else if( A.max(i)>B.min(i) && v[i]>0 )
	{
	    u_1[i] = (A.max(i) - B.min(i)) / v[i]; 
	}
    }

    //possible first time of overlap
    u0 = MAX( u_0[0], MAX(u_0[1], u_0[2]) );

    //possible last time of overlap
    u1 = MIN( u_1[0], MIN(u_1[1], u_1[2]) );

    //they could have only collided if
    //the first time of overlap occurred
    //before the last time of overlap
    return u0 > 0 && u0 <= 1 && u0 <= u1;

}

const bool BBoxSweep(const BBox& b1, const Vec3& start1, const Vec3& end1,
	const BBox& b2, const Vec3& start2, const Vec3& end2,
	float& u0, float& u1)
{
    AABB ab1(start1, b1);
    AABB ab2(start2, b2);

    DB(10) << "start1=" << start1 << " b1=" << b1 << endl;
    DB(10) << "start2=" << start2 << " b2=" << b2 << endl;
    DB(10) << "ab1.P=(" << ab1.P[0] << "," << ab1.P[1] << "," << ab1.P[2] << ")" << endl;
    DB(10) << "ab1.E=(" << ab1.E[0] << "," << ab1.E[1] << "," << ab1.E[2] << ")" << endl;
    DB(10) << "ab2.P=(" << ab2.P[0] << "," << ab2.P[1] << "," << ab2.P[2] << ")" << endl;
    DB(10) << "ab2.E=(" << ab2.E[0] << "," << ab2.E[1] << "," << ab2.E[2] << ")" << endl;


    return AABBSweep(ab1.E, ab1.P, end1,
	    ab2.E, ab2.P, end2,
	    u0, u1);
}

///////////////////////////////////////////////////////////////////////////////

uint32 EmptyWorld::GetAreaOfInterest(list<BBox> *toFill, DynamicGameObject *obj) {
    BBox ret;
    Vec3 orig = obj->GetOrigin();
    for (int i=0; i<3; i++) {
	ret.min[i] = MAX(dims.min[i], orig[i]+aoi.min[i]);
	ret.max[i] = MIN(dims.max[i], orig[i]+aoi.max[i]);
	ASSERTDO(ret.min[i] <= ret.max[i], { 
		INFO << "orig=" << orig << endl;
		INFO << "dims.min[" << i << "]=" << dims.min[i] << " dims.max[" << i << "]=" << dims.max[i];
		INFO << "aoi.min[" << i << "]=" << aoi.min[i] << " aoi.max[" << i << "]=" << aoi.max[i];
		INFO << "ret.min[" << i << "]=" << ret.min[i] << " ret.max[" << i << "]=" << ret.max[i];
		});
    }
    toFill->push_back(ret);
    return 2000; // XXX todo
}

///////////////////////////////////////////////////////////////////////////////

bool EmptyWorld::Fly(const BBox& vol,
	const Vec3& start,
	const Vec3& end,
	Vec3 &hit_pt,
	GameObject *& hit_obj,
	GameObject *ignore)
{
    TraceResult r;
    Trace(r, vol, start, end, ignore);

    // xxx hack! Jeff: we don't know why this occurs, but sometimes it does
    if ( isnan(r.hitPos[0]) ||
	    isnan(r.hitPos[1]) ||
	    isnan(r.hitPos[2]) )
	return false;

    hit_obj = r.hitObj;
    hit_pt  = r.hitPos;
    return r.hit;
}

bool EmptyWorld::FlyThrough(const Vec3& start,
	const Vec3& end,
	Vec3 &hit_pt)
{
    static BBox zero(Vec3(0,0,0),Vec3(0,0,0));
    TraceResult r;
    Trace(r, zero, start, end, NULL, TRACE_WORLD_SOLID|TRACE_WORLD_EDGE);

    // xxx hack! Jeff: we don't know why this occurs, but sometimes it does
    if ( isnan(r.hitPos[0]) ||
	    isnan(r.hitPos[1]) ||
	    isnan(r.hitPos[2]) )
	return false;

    hit_pt = r.hitPos;
    return r.hit;
}

void EmptyWorld::Trace(TraceResult& res, 
	const BBox& vol, const Vec3& start, const Vec3& end,
	const TraceFilterFunc& ignoreFunc, 
	uint32 contentMask)
{
    bzero(&res, sizeof(res));
    res.fraction = -1;

    if (contentMask & TRACE_OBJECT) {
	for (set<GameObject *>::iterator it = objs.begin();
		it != objs.end(); it++) {
	    GameObject *next = *it;
	    if (ignoreFunc(next)) continue;

	    real32 u0, u1;

	    bool h = BBoxSweep(vol, start, end, 
		    next->GetVolume(), 
		    next->GetOrigin(),
		    next->GetOrigin(), u0, u1);
	    if (h && (res.fraction == -1 || u0 < res.fraction)) {
		res.hit = true;
		res.fraction = u0;
		res.hitPos = start + u0*(end - start);
		res.hitObj = next;
		res.contentMask = TRACE_OBJECT;
	    }
	}
    }

    if (contentMask & TRACE_WORLD_EDGE) {
	// check against borders of world
	for (int j=0; j<6; j++) {
	    Vec3 wmin = dims.min;
	    Vec3 wmax = dims.max;
	    Vec3 mid;
	    if (j < 3) {
		wmax[j] = wmin[j];
	    } else {
		wmin[j%3] = wmax[j%3];
	    }
	    mid = wmin + (wmax - wmin)/2;
	    BBox wall(wmin, wmax);
	    real32 u0, u1;
	    bool h = BBoxSweep(vol, start, end, wall, mid, mid, u0, u1);
	    if (h && (res.fraction == -1 || u0 < res.fraction)) {
		res.hit = true;
		res.fraction = u0;
		res.hitPos = start + u0*(end - start);
		res.hitObj = NULL;
		res.contentMask = TRACE_WORLD_EDGE;
	    }
	}
    }
}

Vec3 EmptyWorld::GetRandomPoint() 
{
    return dims.min +
	Vec3( (dims.max[0]-dims.min[0])*drand48(),
		(dims.max[1]-dims.min[1])*drand48(),
		(dims.max[2]-dims.min[2])*drand48() );
}

void EmptyWorld::GetVisible(list<GameObject *> *vis, 
	DynamicGameObject *obj,
	const TraceFilterFunc& ignoreFunc) 
{
    list<BBox> aoi;
    obj->GetAreaOfInterest(&aoi, this);

    // XXX TODO: more efficient!
    for (set<GameObject *>::iterator it = objs.begin();
	    it != objs.end(); it++) {
	GameObject *next = *it;
	if (ignoreFunc(next)) continue;
	if ( IsVisible(obj, next->GetOrigin(), &aoi) ) {
	    vis->push_back( next );
	}
    }
}

bool EmptyWorld::IsVisible(DynamicGameObject *from, const Vec3& to,
	list<BBox> *from_aoi)
{
    list<BBox> aoi;
    if (!from_aoi) {
	from->GetAreaOfInterest(&aoi, this);
	from_aoi = &aoi;
    }
    for (list<BBox>::iterator it = from_aoi->begin(); 
	    it != from_aoi->end(); it++) {
	if ( (*it).Contains( to ) ) {
	    return true;
	}
    }
    return false;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
