#include <gameapi/GameManager.h>
#include "DonnybrookManager.h"
#include "SimpleMissile.h"
#include "SimplePlayer.h"

bool g_EnableDonnybrook;
char g_DonnybrookTraceFile [1024];
DonnybrookManager *g_DonnybrookManager = NULL;

DonnybrookManager::DonnybrookManager (GameManager *manager, int machines) 
    : m_GameManager (manager), m_NumMachines (machines)
{
    m_Players = 0;
    m_NonPlayers = 0;

    m_Logfp = fopen (g_DonnybrookTraceFile, "w");
    if (!m_Logfp) 
	Debug::die ("Could not open (%s) for writing", g_DonnybrookTraceFile);    

    generateStartEvent ();
    generateChurnEvents (EV_UP);
}

DonnybrookManager::~DonnybrookManager () 
{
    ASSERT (m_Logfp != NULL);

    generateChurnEvents (EV_DOWN);
    generateQuitEvent ();

    fclose (m_Logfp);
}

void DonnybrookManager::RecordObjectUpdate (SimpleMovable *obj)
{
    DbrookObjMapIter it = m_ObjectMap.find (obj->GetGUID ());
    ASSERT (it != m_ObjectMap.end ());
        
    generateUpdateEvent (obj, it->second);
}

void DonnybrookManager::RecordObjectDelete (SimpleMovable *obj)
{
    DbrookObjMapIter it = m_ObjectMap.find (obj->GetGUID ());
    ASSERT (it != m_ObjectMap.end ());
        
    generateDeleteEvent (obj, it->second);

    // should we garbage collect info or just let it pending? no real 
    // harm i guess... hopefully other testgame code checks for 
    // sanity in terms of not accessing deleted objects.
}

void DonnybrookManager::RecordObjectCreate (SimpleMovable *obj) 
{
    int dbrook_guid;
    bool is_player = false;

    if (obj->GetVisualizerType () == VisType_Bot) {
	is_player = true;
	dbrook_guid = m_Players;
	m_Players++;
    }
    else {
	dbrook_guid = m_NumMachines + m_NonPlayers;
	m_NonPlayers++;
    }

    DbrookObjInfo info;
    info.colyseus_guid = obj->GetGUID ();
    info.dbrook_guid   = dbrook_guid;

    // assign a machine to this object.
    if (is_player) {
	SimplePlayer *player = dynamic_cast<SimplePlayer *>(obj);
 	
	info.machine = dbrook_guid;
	info.parent  = -1;
	info.classtype = OBJ_PLAYER;
	info.team = player->GetTeam();
    }
    else {
	// XXX: sort of a hack, update this when Items are added 
	// to test-game as well. 
	//
	SimpleMissile *missile = dynamic_cast<SimpleMissile *> (obj);
	GameObject *owner = missile->owner;

	ASSERT (owner != NULL);

	DbrookObjMapIter it = m_ObjectMap.find (owner->GetGUID ());
	ASSERT (it != m_ObjectMap.end ());

	info.parent  = it->second.dbrook_guid;
	info.machine = it->second.machine;     // Or else, this could be (int) (drand48 () * m_NumMachines)
	info.classtype = OBJ_MISSILE;
	info.team = missile->owner != NULL ? missile->owner->GetTeam() : -1;
    }
    m_ObjectMap [obj->GetGUID ()] = info;

    generateCreateEvent (obj, info);
}

void DonnybrookManager::RecordDamage (SimpleMovable *source, SimpleMovable *target)
{
    DbrookObjMapIter it_source = m_ObjectMap.find (source->GetGUID ());
    ASSERT (it_source != m_ObjectMap.end ());

    DbrookObjMapIter it_target = m_ObjectMap.find (target->GetGUID ());
    ASSERT (it_target != m_ObjectMap.end ());

    generateDamageEvent (it_source->second, it_target->second);
}

void DonnybrookManager::generateDamageEvent (DbrookObjInfo& source, DbrookObjInfo& target)
{
    int curframe = m_GameManager->GetFrameNumber ();

    fprintf (m_Logfp, "%-10d %-10d %-10d %-10d %-10d %-10d\n",
	    curframe,
	    EV_DAMAGE,
	    source.machine,
	    target.machine,
	    source.dbrook_guid,
	    target.dbrook_guid
	    );
}

void DonnybrookManager::generateChurnEvents (int state) 
{
    ASSERT (m_Logfp != NULL);

    fprintf (m_Logfp, "%-10s %-10s %-10s\n", "# frame", "UPDN", "node");
    for (int i = 0; i < m_NumMachines; i++) { 
	fprintf (m_Logfp, "%-10d %-10d %-10d\n", m_GameManager->GetFrameNumber (), state, i);
    }    
}

void DonnybrookManager::generateStartEvent ()
{
    fprintf (m_Logfp, "%-10s %-10s\n", "# nMach", "nObjs");
    fprintf (m_Logfp, "%-10d %-10s\n", m_NumMachines, "[placeholder]");
}

void DonnybrookManager::generateQuitEvent () 
{
    fprintf (m_Logfp, "%-10s %-10s\n", "# frame", "QUIT");
    fprintf (m_Logfp, "%-10d %-10d\n", m_GameManager->GetFrameNumber (), EV_QUIT);
}

int DonnybrookManager::getFullObjectSize (SimpleMovable *obj) 
{
    return 100;
}

int DonnybrookManager::getDeltaObjectSize (SimpleMovable *obj)
{
    return 20;
}

void DonnybrookManager::generateUpdateEvent (SimpleMovable *obj, DbrookObjInfo& info)
{
    int curframe = m_GameManager->GetFrameNumber ();
    Vec3 pos = obj->GetOrigin ();
    Vec3 vel = obj->GetVelocity ();
    Vec3 ang = obj->GetViewAngles ();

    bool flagCarrier = false;
    if (obj->GetVisualizerType () == VisType_Bot) {
	SimplePlayer *player = dynamic_cast<SimplePlayer *>(obj);
	flagCarrier = player->IsCarryingFlag();
    }

    fprintf (m_Logfp, "%-10d %-10d %-10d %-10d %-10d %-10d %-10.3f %-10.3f %-10.3f %-10.3f %-10.3f %-10.3f %d\n", 
	    curframe, 
	    EV_UPDATE,
	    info.machine, 
	    info.dbrook_guid,
	    getFullObjectSize (obj), getDeltaObjectSize (obj),
	    pos [0], pos [1], vel [0], vel [1], ang [0], ang [1],
	    flagCarrier ? 1 : 0 
	    );

}

void DonnybrookManager::generateCreateEvent (SimpleMovable *obj, DbrookObjInfo& info)
{
    int curframe = m_GameManager->GetFrameNumber ();
    Vec3 pos = obj->GetOrigin ();
    Vec3 vel = obj->GetVelocity ();
    Vec3 ang = obj->GetViewAngles ();

    fprintf (m_Logfp, "%-10d %-10d %-10d %-10d %-10d %-10d %-10d %-10.3f %-10.3f %-10.3f %-10.3f %-10.3f %-10.3f %d\n", 
	    curframe, 
	    EV_CREATE,
	    info.machine, 
	    info.dbrook_guid, info.classtype, info.parent,
	    getFullObjectSize (obj),
	    pos [0], pos [1], vel [0], vel [1], ang [0], ang [1],
	    info.team 
	    );
}

void DonnybrookManager::generateDeleteEvent (SimpleMovable *obj, DbrookObjInfo& info)
{
    int curframe = m_GameManager->GetFrameNumber ();
    fprintf (m_Logfp, "%-10d %-10d %-10d %-10d\n", 
	    curframe,
	    EV_DELETE,
	    info.machine,
	    info.dbrook_guid);
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
