////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/***************************************************************************
  main.cpp

  SampleApp - simple test application where bots make a random walk along
  a 1D line.

usage: sampleapp [port]

... where [port] is the port this program should run on. (We assume
the bootstrap server is running on 15000)

To see the game in action, you probably want to do the following to
get rid of debugging messages:

sampleapp [port] 2> /dev/null


begin		   : October 10, 2003
copyright	   : (C) 2003 Jeffrey Pang           ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <cstdlib>
#include <cstdio>
#include <typeinfo>
#include <map>

#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <Colyseus.h>
#include <om/GObject.h>
#include <om/Manager.h>
#include <om/Event.h>

#ifndef MAX
#define MAX(x,y) ((x)>(y)?(x):(y))
#endif
#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif

#define TYPE_PERSON 1
#define TYPE_DOG    2

extern char g_ProgramName[];

///////////////////////////////////////////////////////////////////////////////
////////// Application Classes

class Char;

class SimpleStore : public ObjectStore {
public:
    map<GUID, Char *, less_GUID> store;
    map<GUID, Char *, less_GUID>::iterator curr;

    SimpleStore() {};
    virtual ~SimpleStore() {};

    void Begin();
    GObject *Next();
    GObject *Find(guid_t guid);
    void Add(GObject *obj);
    void Remove(guid_t guid);
};

class SimpleGame : public GameAdaptor {
public:
    SimpleStore m_store;
    SimpleStore m_pending;

    SimpleGame() {}
    virtual ~SimpleGame() {}

    ObjectStore *GetObjectStore();
    ObjectStore *GetPendingStore();
    GObject *Construct(Event *ev, SIDMap *unresolved);
};


/**
 * An entity bot in our game. Represented by a random capital letter.
 */
class Char : public GObject {
private:
    int m_type;
    /* symbol we use to represent our character */
    char m_sprite;
    /* location of our character [0,79] */
    uint32 m_loc;
    bool m_hit;

    bool m_replica;
    bool m_dirty;
    GUID m_guid;
    SID m_sid;

    Char *partner;

    GUIDSet refs;
    GUIDSet missing;

public:
    /* how far our character sees in either direction */
    static const int sight = 40;

    /**
     * Local constructor
     */
    Char(int type, GUID id, SID sid);

    /**
     * Network constructor
     */ 
    Char(Event * pub, SIDMap *unresolved);

    virtual ~Char() {}

    int GetLoc();
    void SetLoc(int loc);
    void SetPartner(Char *p);
    void Hit();
    bool IsHit() { return m_hit; }
    void MakeMove();
    void MoveLeft();
    void MoveRight();
    int GetSprite();

    ////////// Implements GObject //////////

    const guid_t GetGUID();
    const sid_t GetSID();
    bool IsReplica();
    bool IsDirty();
    void SetDirty(bool v);
    void FillEvent(Event *ev);
    void FillFetchResp(Event *resp);
    void FillInterest(Interest *in);
    void HandleEvent(Event *ev, SIDMap *unresolved);
    const GUIDSet *GetRefs();
    const GUIDSet *GetUnresolvedRefs();
    void ResolveRef(guid_t guid, GObject *obj);

};

///////////////////////////////////////////////////////////////////////////////
////////// Application Globals

/**
 * The gameworld -- a 1D array of 79 spaces (fits in a console :)
 */
char *gameworld;

static Manager *controller;
static SimpleGame adaptor;

/**
 * The bot character we own.
 */
Char *me;
Char *dog;

///////////////////////////////////////////////////////////////////////////////

void SimpleStore::Begin() {
    curr = store.begin();
}

GObject *SimpleStore::Next() {
    if (curr == store.end()) {
	return NULL;
    } else {
	GObject *next = curr->second;
	curr++;
	return next;
    }
}

GObject *SimpleStore::Find(guid_t guid) {
    map<GUID, Char *, less_GUID>::iterator p = store.find(guid);
    if (p == store.end()) {
	return NULL;
    } else {
	return p->second;
    }
}

void SimpleStore::Add(GObject *obj) {
    store[obj->GetGUID()] = (Char *)obj;
}

void SimpleStore::Remove(guid_t guid) {
    store.erase(guid);
}

ObjectStore *SimpleGame::GetObjectStore() {
    return &m_store;
}

ObjectStore *SimpleGame::GetPendingStore() {
    return &m_pending;
}

GObject *SimpleGame::Construct(Event *ev, SIDMap *unresolved) {
    return new Char(ev, unresolved);
}

Char::Char(int type, GUID id, SID sid) : 
    GObject(), m_type(type), m_guid(id), m_sid(sid), m_hit(false) {
    if (type == TYPE_DOG) 
	m_sprite = (char) ('a' + (rand() % 26));
    else
	m_sprite = (char) ('A' + (rand() % 26));
    m_loc = rand() % 79;
    m_dirty = false;
    m_replica = false;
    partner = NULL;
}

Char::Char(Event * pub, SIDMap *unresolved) : GObject() {
    m_guid = pub->GetGUID();
    m_sid  = pub->GetSID();

    m_type = 0;
    m_sprite = '?';
    m_loc = 0;
    m_dirty = false;
    m_replica = true;
    partner = NULL;

    HandleEvent(pub, unresolved);
}

int Char::GetLoc() {
    return m_loc;
}

void Char::SetLoc(int loc) {
    m_loc = loc;
}

void Char::SetPartner(Char *p) {
    if (partner != NULL) {
	refs.erase(partner->GetGUID());
    }
    partner = p;
    if (p != NULL) {
	refs.insert(p->GetGUID());
    }
}

void Char::Hit() {
    SetDirty(true);
    m_hit = !m_hit;
}

void Char::MakeMove() {
    int move = rand() % 3;
    switch (move) {
    case 0:
	break;
    case 1:
	MoveLeft();
	break;
    case 2:
	MoveRight();
	break;
    }
}

void Char::MoveLeft() {
    if (m_loc > 0) {
	SetDirty(true);
	m_loc--;
    }
}

void Char::MoveRight() {
    if (m_loc < 78) {
	SetDirty(true);
	m_loc++;
    }
}

int Char::GetSprite() {
    return m_sprite;
}

const guid_t Char::GetGUID() {
    return m_guid;
}

const sid_t Char::GetSID() {
    return m_sid;
}

bool Char::IsReplica() {
    return m_replica;
}

bool Char::IsDirty() {
    return m_dirty;
}

void Char::SetDirty(bool v) {
    m_dirty = v;
}

void Char::FillEvent(Event *ev) {
    FillFetchResp(ev);
    SetDirty(false);
}

void Char::FillFetchResp(Event *resp) {
    resp->AddTuple("type", Value((uint32) m_type));
    resp->AddTuple("sprite", Value((uint32) m_sprite));
    resp->AddTuple("hit", Value((char) m_hit));
    resp->AddTuple("x_coordinate", Value(m_loc));
    if (partner != NULL)
	resp->AddTuple("partner", Value(partner->GetGUID(), 
					partner->GetSID()));
}

void Char::FillInterest(Interest *in) {
    static string attr("x_coordinate");
    static string attr2("type");
    uint32 min = MAX(0, GetLoc() - Char::sight);
    uint32 max = MIN(78, GetLoc() + Char::sight);

    in->AddConstraint(attr, OP_GREATER_EQUAL, Value(min));
    in->AddConstraint(attr, OP_LESS_EQUAL, Value(max));
    in->AddConstraint(attr2, OP_EQUAL, Value((uint32)TYPE_PERSON));
    // don't want dogs unless with people! :)
}

void Char::HandleEvent(Event *ev, SIDMap *unresolved) {
    m_type   = ev->GetAttribute("type")->m_Ival;
    m_sprite = (char) ev->GetAttribute("sprite")->m_Ival;
    m_loc    = ev->GetAttribute("x_coordinate")->m_Ival;
    m_hit    = ev->GetAttribute("hit")->m_Cval;

    Value *pval = ev->GetAttribute("partner");
    missing.clear();
    refs.clear();
    if (pval != NULL) {
	guid_t partner_guid = *(pval->m_IDval.guid);
	sid_t  primary_loc  = *(pval->m_IDval.sid);
	partner = (Char *)adaptor.GetObjectStore()->Find(partner_guid);
	if (partner == NULL) {
	    partner = (Char *)adaptor.GetPendingStore()->Find(partner_guid);
	}
	if (partner == NULL) {
	    // oops! don't have partner yet!
	    (*unresolved)[partner_guid] = primary_loc;
	    missing.insert(partner_guid);
	}
	refs.insert(partner_guid);

    }
}

const GUIDSet *Char::GetRefs() {
    return &refs;
}

const GUIDSet *Char::GetUnresolvedRefs() {
    return &missing;
}

void Char::ResolveRef(guid_t guid, GObject *obj) {
    if (missing.find(guid) != missing.end()) {
	partner = (Char *)obj;
	missing.erase(guid);
    }
}

///////////////////////////////////////////////////////////////////////////////
////////// Application Code

/**
 * Initialize the local game world and mercury framework.
 *
 * @param port the port to start the local mercury rounter on
 */
void InitGame()
{

    // init rand seed
    srand((int) getpid());

    // setup game world
    gameworld = new char[80];
    gameworld[79] = '\0';

    // start merc controller
    controller = new Manager(&adaptor);

    // set up our character
    me  = new Char(TYPE_PERSON, 
		   controller->CreateGUID(), controller->GetSID());
    dog = new Char(TYPE_DOG,
		   controller->CreateGUID(), controller->GetSID());

    cout << "me:  " << me->GetGUID() << endl;
    cout << "dog: " << dog->GetGUID() << endl;

    // register my character
    adaptor.GetObjectStore()->Add(me);
    adaptor.GetObjectStore()->Add(dog);

    // my partner is my dog!
    me->SetPartner(dog);
    dog->SetPartner(me);
}

/**
 * Start the game loop (infinite loop)
 */
void RunGame()
{
    Interest *in = NULL;

    while (1) {

	Char *obj;
	ObjectStore *store = adaptor.GetObjectStore();
	store->Begin();
	while ( (obj = (Char *)store->Next()) != NULL) {
	    if (obj->IsReplica()) {
		// enemy!
		obj->Hit();
	    }
	}

	////////// update state
	// ask our entity to move
	me->MakeMove();
	dog->MakeMove();

	// process network
	controller->Exec();

	// update local gameworld
	for (int i = 0; i < 79; i++) {
	    gameworld[i] = '.';
	}

	if (me->GetLoc() - Char::sight >= 0)
	    gameworld[me->GetLoc() - Char::sight] = '[';
	if (me->GetLoc() + Char::sight < 79)
	    gameworld[me->GetLoc() + Char::sight] = ']';

	store->Begin();
	while ( (obj = (Char *)store->Next()) != NULL) {
	    if (obj->IsHit()) {
		gameworld[obj->GetLoc()] = '!';
	    } else {
		gameworld[obj->GetLoc()] = obj->GetSprite();
	    }
	}

	////////// render frame
	printf("%s\n", gameworld);

	////////// sleep
	sleep(1);
    }

}

/**
 * Entry point
 */
int main(int argc, char **argv)
{

    ////////// read arguments
    if (argc != 2) {
	fprintf(stderr, "usage: sampleapp [port]\n");
	return 1;
    }
    int port = atoi(argv[1]);

    strcpy(g_ProgramName, argv[0]);

    ////////// set mercury preferences
    strcpy(g_Preferences.bootstrap, "127.0.0.1:15000");
    strcpy(g_Preferences.hostname, "127.0.0.1");
    g_Preferences.port = port;
    g_Preferences.pub_delay = 100;
    g_Preferences.send_pubs = true;
    g_Preferences.send_subs = true;
    g_Preferences.use_softsubs = true;
    g_Preferences.sub_lifetime = 2000;
    g_Preferences.send_backpub = false;
    g_VerbosityLevel = -1;

    ////////// initialize the game and mercury
    InitGame();

    ////////// run the game
    RunGame();

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
