#!/usr/bin/perl

use strict;
use Getopt::Std;
use FileHandle;
use vars qw($opt_s $opt_f $opt_o $opt_b);

getopts("s:f:o:b:");

# number of servers
my $servers = defined $opt_s ? $opt_s : 10;
# bandwidth field
my $field   = defined $opt_f ? $opt_f : "TOTIN";
# output file prefix
my $outprefix = defined $opt_o ? $opt_o : $field;
# number of buckets per point
my $bucket_size = defined $opt_b ? $opt_b : 500;

my @times;
my @tot;
my @vals;
my @min;
my @max;

if (@ARGV != 1) {
    print STDERR "usage: plot_tot_bwidth.pl [options] [bwidth_file]\n";
    print STDERR "options:\n";
    print STDERR "\t-s int\tthe number of servers ($servers)\n";
    print STDERR "\t-f str\tthe field in bwidth_file to examine ($field)\n";
    print STDERR "\t-o file\tthe prefix to the output ($outprefix)\n";
    print STDERR "\t-b int\ttime to aggregate per data point ($bucket_size)\n";
    print STDERR "\nTakes the output of bwidth_time.pl and generates data files for avg/std,\n";
    print STDERR "min, and max plots for a particular field.\n";
    exit(1);
}

my @data;
my $start;

open(IN, "<$ARGV[0]") || die $!;
while (<IN>) {
    if (/(\d+)\t(\d+).*$field=([\d\.]+).*/) {
	my $time = $1;
	if (!defined $start) { $start = $time; }
	my $sid  = $2;
	my $val  = $3;

	my $index = int(($time-$start)/$bucket_size);
	#print "$index\n";

	# assume we see a value for every quantam
	if (!exists $data[$index]->{$sid}) {
	    $data[$index]->{$sid} = [ $val, 1 ];
	} else {
	    $data[$index]->{$sid}->[0] += $val;
	    $data[$index]->{$sid}->[1]++;
	}

    } else {
	warn "bad line: $_\n";
    }
}
    
for (my $i=0; $i<@data; $i++) {
    foreach my $sid (keys %{$data[$i]}) {
	my $time = $i*$bucket_size + $start;
	my $val  = $data[$i]->{$sid}->[0]/$data[$i]->{$sid}->[1];
	#print "val=" . $data[$i]->{$sid}->[0] . " count=" .
	#    $data[$i]->{$sid}->[1] . "\n";

	if (!defined($times[0]) || $time != $times[$#times]) {
	    push @times, $time;
	    push @min, $val;
	    push @max, $val;
	    push @vals, [ $val ];
	    push @tot, $val;
	} else {
	    if ($val < $min[$#min]) {
		$min[$#min] = $val;
	    }
	    if ($val > $max[$#max]) {
		$max[$#max] = $val;
	    }
	    push @{$vals[$#vals]}, $val;
	    $tot[$#tot] += $val;
	}
    }
}

my @avg;
my @std;

for (my $i=0; $i<=$#tot; $i++) {
    $avg[$i] = $tot[$i]/$servers;
}
for (my $i=0; $i<=$#vals; $i++) {
    my @set = @{$vals[$i]};
    for (my $j=0; $j<=$#set; $j++) {
	$std[$i] += ($set[$j]-$avg[$i])**2;
    }
}
for (my $i=0; $i<=$#std; $i++) {
    $std[$i] = sqrt($std[$i]/$servers);
}


output("max", \@max);
output("avg", \@avg, \@std);
output("min", \@min);

sub output {
    my $suffix = shift;
    my @vals   = @{shift()};
    my $stdp   = shift;
    open(OUT, ">$outprefix.$suffix") || die $!;
    for (my $i=0; $i<=$#times; $i++) {
	print OUT "$times[$i] $vals[$i]";
	if (defined $stdp) { print OUT " " . $stdp->[$i]; }
	print OUT "\n";
    }
    close(OUT);
}
