#!/usr/bin/perl

use strict;
use Getopt::Std;
use FileHandle;
use vars qw($opt_q $opt_o $opt_f $opt_t);

getopts("q:o:f:t:");

# only consider object ids that are in this file
my $filter  = defined $opt_f ? $opt_f : undef;
my %consider;
if ($filter) {
    open (FILTER, "<$filter") || die $!;
    while (<FILTER>) {
	chomp $_;
	$consider{$_} = 1;
    }
    close(FILTER);
}
# start time
my $start   = defined $opt_t ? $opt_t : 0;
# time unit to aggrate migrations over (in time units of the trace)
my $quantam = defined $opt_q ? $opt_q : 10000;
# output file
my $out = defined $opt_o ? new FileHandle(">$opt_o") : *STDOUT;
if (! defined $out) { die $!; }

if (@ARGV != 2) {
    print STDERR "usage: migration_freq.pl [options] [logfile] [tracefile]\n";
    print STDERR "options:\n";
    print STDERR "\t-q int\tquantam -- time unit to aggregate migrations over ($quantam)\n";
    print STDERR "\t-s int\tstart time -- when the trace starts ($start)\n";
    print STDERR "\t-o file\toutput file -- where to print output ($out)\n";
    print STDERR "\t-f file\tfilter file -- only consider objects in this file ($filter)\n";
    print STDERR "\nComputes the migration frequency of individual objects (# per quantam).\n";
    print STDERR "output format: <quantam_start> <avg> <std> <min> <max>\n";
    exit 1;
}

my $logfile = $ARGV[0];
my $tracefile = $ARGV[1];

my $slice = {};

open(LOG, "<$logfile") || die $!;
open(TRACE, "<$tracefile") || die $!;

my $num_objects = 0;
my $trace_time = $start;
my $last = $start;
while (<LOG>) {
    # use the trace to figure out how many objects are in the system
    my($time) = (/^(\d+)/);

    if ($time % 10000 == 0) {
	print STDERR ".$time";
    }

    while ($trace_time < $time) {
	my $line = <TRACE>;
	last if !defined $line;
	($trace_time) = ($line =~ /^(\d+)/);
	if ($line =~ /^\d+ create [-\d]+ (\d+)/) {
	    if (!defined $filter || $consider{$1}) {
		$num_objects++;
	    }
	} elsif ($line =~ /^\d+ destroy (\d+)/) {
	    if (!defined $filter || $consider{$1}) {
		$num_objects--;
	    }
	}
    }

    #print STDERR ($lineno++) . ": $time $start\n";

    while ($time - $quantam > $last) {
	output_slice($last, $slice);
	undef $slice;
	$last = $last + $quantam;
	$slice = {};
    }

    if (/(\d+) MIGRATE (\d+) (\d+) (\d+) (\d+)/) {
	my $src  = $2;
	my $dst  = $3;
	my $obj  = $4;
	my $size = $5;
	
	if (!defined $filter || $consider{$obj}) {
	    if (defined $slice->{$obj}) {
		$slice->{$obj}++;
	    } else {
		$slice->{$obj} = 1;
	    }
	}
    }
}
output_slice($last, $slice);

print STDERR "\n";

close(LOG);
close(TRACE);
close($out);

sub output_slice {
    my $time  = shift;
    my %slice = %{shift(@_)};
 
    my $min = 999999999;
    my $max = 0;
    my $total = 0;

    if (scalar(%slice) < $num_objects) {
	$min = 0;
    }
    foreach my $obj (keys %slice) {
	if ($slice{$obj} < $min) {
	    $min = $slice{$obj};
	}
	if ($slice{$obj} > $max) {
	    $max = $slice{$obj};
	}
	$total += $slice{$obj};
    }

    my $avg = $num_objects==0?0:$total/$num_objects;
    my $std = 0;
    foreach my $obj (keys %slice) {
	$std += ($avg - $slice{$obj})**2;
    }
    $std = $num_objects==0?0:sqrt($std/$num_objects);

    print $out "$time $avg $std $min $max\n";
}
