#!/usr/bin/perl

use strict;
use Getopt::Std;
use FileHandle;
use vars qw($opt_q $opt_t $opt_s $opt_o $opt_c);

getopts("q:s:t:o:c");

my $PER_CLIENT_IN  = 3;
my $PER_CLIENT_OUT = 4;

# just count the number of messages (msgs/ms instead of KBps)
my $count   = defined $opt_c;
# number of servers
my $servers = defined $opt_s ? $opt_s : 10;
# start time
my $start   = defined $opt_t ? $opt_t : 0;
# time slice to aggregate bandwidth over (in time units of the trace)
my $quantam = defined $opt_q ? $opt_q : 500;
# output file
my $out = defined $opt_o ? new FileHandle(">$opt_o") : *STDOUT;
if (! defined $out) { die $!; }

my @INBOUND = ( 'PUB_RECV',
		'UPDATE_RECV',
		'MIGRATE_RECV',
		'CLIENT_RECV' );
my @OUTBOUND = ( 'SUB_SEND',
		 'PUB_SEND',
		 'UPDATE_SEND',
		 'MIGRATE_SEND',
		 'CLIENT_SEND' );
my @STATS = (@INBOUND, @OUTBOUND);

if (@ARGV != 1) {
    print STDERR "usage: bwidth_time.pl [options] [logfile]\n";
    print STDERR "options:\n";
    print STDERR "\t-s int\tthe number of servers used ($servers)\n";
    print STDERR "\t-t int\tthe start time of the trace ($start)\n";
    print STDERR "\t-q int\tthe quantam size to aggrate bwidth usage ($quantam)\n";
    print STDERR "\t-o file\toutput file ($out)\n";
    print STDERR "\t-c \tjust count the number of messages/clients ($count)\n";
    print STDERR "\nFor each quantam, for each server, print out bandwidth usage statistics about\n";
    print STDERR "each of {PUB,UPDATE,MIGRATE,CLIENT} and TOT (total) inbound (RECV) and outbound\n";
    print STDERR "(SEND) message usage.\n";
    exit 1;
}

my $logfile = $ARGV[0];
my $ROUTING_INDEX = $servers;

open(LOG, "<$logfile") || die $!;

my $last  = $start;
my $stats = create_slice();
my $client = create_slice();
my $lineno = 1;
while (<LOG>) {
    my($time) = (/^(\d+)/);
    #print STDERR ($lineno++) . ": $time $start\n";

    while ($time - $quantam > $last) {
	foreach my $sid (0..($servers-1)) {
	    client_update($stats, ($last+$quantam), $sid, undef, undef);
	}
	output_slice($last, $stats);
	undef $stats;
	$last = $last + $quantam;
	$stats = create_slice();
    }

    if (/(\d+) SUB_SEND (\d+) (\d+)/) {
	my $src  = $2;
	my $size = $3;
	$stats->[$src]->{'SUB_SEND'} += $count?1:$size;

	$stats->[$ROUTING_INDEX]->{'ROUTING_RECV'} += $count?1:$size;
    } elsif (/(\d+) PUB_SEND (\d+) (\d+) (\d+)/) {
	my $src  = $2;
	my $size = $4;
	$stats->[$src]->{'PUB_SEND'} += $count?1:$size;

	$stats->[$ROUTING_INDEX]->{'ROUTING_RECV'} += $count?1:$size;
    } elsif (/(\d+) PUB_RECV (\d+) (\d+) (\d+)/) {
	my $dst  = $2;
	my $size = $4;
	$stats->[$dst]->{'PUB_RECV'} += $count?1:$size;

	$stats->[$ROUTING_INDEX]->{'ROUTING_SEND'} += $count?1:$size;
    } elsif (/(\d+) UPDATE (\d+) (\d+) (\d+) (\d+)/) {
	my $src  = $2;
	my $dst  = $3;
	my $obj  = $4;
	my $size = $5;
	$stats->[$src]->{'UPDATE_SEND'} += $count?1:$size;
	$stats->[$dst]->{'UPDATE_RECV'} += $count?1:$size;
    } elsif (/(\d+) MIGRATE (\d+) (\d+) (\d+) (\d+)/) {
	my $src  = $2;
	my $dst  = $3;
	my $obj  = $4;
	my $size = $5;
	$stats->[$src]->{'MIGRATE_SEND'} += $count?1:$size;
	$stats->[$dst]->{'MIGRATE_RECV'} += $count?1:$size;
    } elsif (/(\d+) CLIENT (\d+) (\d+) (\d+)/) {
	my $sid  = $2;
	my $recv = $3;
	my $send = $4;

	if ($count) {
	    $recv = $recv/$PER_CLIENT_IN;
	    $send = $send/$PER_CLIENT_OUT;
	}
	client_update($stats, $time, $sid, $recv, $send);
    } elsif (/(\d+) DEBUG .*/) {
	# ignore
    } else {
	die "malformed log line:  $_\n";
    }
}

foreach my $sid (0..($servers-1)) {
    client_update($stats, ($last+$quantam), $sid, undef, undef);
}
output_slice($last, $stats);

exit(0);

sub client_update {
    my ($stats, $time, $sid, $recv, $send) = @_;

    # convert constant bwidth into a chunk of data sent
    $stats->[$sid]->{'CLIENT_RECV'} += 
	($time-$client->[$sid]->{'clin_time'})*$client->[$sid]->{'clin_val'};
    $stats->[$sid]->{'CLIENT_SEND'} += 
	($time-$client->[$sid]->{'clout_time'})*$client->[$sid]->{'clout_val'};

    # update the constant rate
    $client->[$sid]->{'clin_time'}  = $time;
    $client->[$sid]->{'clout_time'} = $time;
    if (defined $recv) {
	$client->[$sid]->{'clin_val'}   = $recv;
    }
    if (defined $send) {
	$client->[$sid]->{'clout_val'}   = $send;
    }
}

sub create_slice {
    my @slice;

    for (my $i=0; $i<$servers+1; $i++) {
	$slice[$i] = {};
	foreach my $stat (@STATS) {
	    $slice[$i]->{$stat} = 0;
	}
    }

    return \@slice;
}

sub output_slice {
    my $time  = shift;
    my @slice = @{shift(@_)};
    
    for (my $i=0; $i<@slice-1; $i++) {
	print $out "$time\t$i";
	foreach my $stat (@STATS) {
	    print $out sprintf "\t$stat=%g", ($slice[$i]->{$stat}/$quantam);
	}
	my $total_in;
	my $total_out;
	foreach my $stat (@INBOUND) {
	    $total_in += $slice[$i]->{$stat}/$quantam;
	}
	foreach my $stat (@OUTBOUND) {
	    $total_out += $slice[$i]->{$stat}/$quantam;
	}
	my $route_in = $stats->[$ROUTING_INDEX]->{'ROUTING_RECV'}/($servers*$quantam);
	$total_in += $route_in;
	print $out sprintf "\tROUTEIN=%.5g", $route_in;
	my $route_out = $stats->[$ROUTING_INDEX]->{'ROUTING_SEND'}/($servers*$quantam);
	$total_out += $route_out;
	print $out sprintf "\tROUTEOUT=%.5g", $route_out;

	print $out sprintf "\tTOTIN=%.5g", $total_in;
	print $out sprintf "\tTOTOUT=%.5g", $total_out;
	print $out "\n";
    }
}
